package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogGeoMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.GeoMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.Region;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.GEO_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.REGIONAL_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.REGIONAL_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.GeoMultiplierValues.GEO_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.regions.Region.CRIMEA_REGION_ID;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.regions.Region.SNG_REGION_ID;
import static ru.yandex.direct.regions.Region.UKRAINE_REGION_ID;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class BidModifierGeoTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierGeo, BidModifierRegionalAdjustment, BidModifierGeoTypeSupport.GeoMultiplierKey> {
    private static final Set<Long> TRANSLOCAL_REGIONS = ImmutableSet.of(
            Region.SNG_REGION_ID,
            Region.UKRAINE_REGION_ID,
            Region.RUSSIA_REGION_ID,
            Region.CRIMEA_REGION_ID
    );

    private final ClientService clientService;

    @Autowired
    public BidModifierGeoTypeSupport(ShardHelper shardHelper,
                                     LogBidModifiersService logBidModifiersService, ClientService clientService) {
        super(shardHelper, logBidModifiersService);
        this.clientService = clientService;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.GEO_MULTIPLIER;
    }

    @Override
    public Class<BidModifierGeo> getBidModifierClass() {
        return BidModifierGeo.class;
    }

    @Override
    public Class<BidModifierRegionalAdjustment> getAdjustmentClass() {
        return BidModifierRegionalAdjustment.class;
    }

    @Override
    protected GeoMultiplierKey getKey(BidModifierRegionalAdjustment adjustment) {
        return new GeoMultiplierKey(adjustment.getRegionId(), adjustment.getHidden());
    }

    @Override
    public void setAdjustments(BidModifierGeo modifier, List<BidModifierRegionalAdjustment> adjustments) {
        modifier.setRegionalAdjustments(adjustments);
    }

    @Override
    public Map<Long, List<BidModifierRegionalAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                              Collection<Long> ids) {
        Result<Record> records = dslContext.select(REGIONAL_ADJUSTMENT_FIELDS)
                .from(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(REGIONAL_ADJUSTMENT_MAPPER::fromDb).collect(toList()))
                .toMap();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierRegionalAdjustment> adjustments, DSLContext txContext) {
        InsertHelper<GeoMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, GEO_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(REGIONAL_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierRegionalAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<GeoMultiplierValuesRecord, BidModifierRegionalAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, GEO_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, GEO_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(GEO_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    public boolean areEqual(BidModifierGeo a, BidModifierGeo b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getRegionalAdjustments()),
                        new HashSet<>(b.getRegionalAdjustments()));
    }

    @Override
    public BidModifierGeo createEmptyBidModifierFromRecord(Record record) {
        return GEO_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierGeo> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierGeo> bidModifiers, boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(REGIONAL_ADJUSTMENT_FIELDS)
                .from(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierGeo> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierRegionalAdjustment> adjustments =
                            recordsList.stream().map(REGIONAL_ADJUSTMENT_MAPPER::fromDb).collect(toList());
                    bidModifiersById.get(bidModifierId).withRegionalAdjustments(adjustments);
                });
    }

    @Override
    public List<BidModifierRegionalAdjustment> getAdjustments(BidModifierGeo modifier) {
        return modifier.getRegionalAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierRegionalAdjustment> added,
                                     List<BidModifierRegionalAdjustment> inserted) {
        Map<GeoMultiplierKey, Long> idsByKey =
                inserted.stream().collect(toMap(this::getKey, BidModifierRegionalAdjustment::getId));
        return added.stream()
                // Скрытые корректировки не нужно возвращать пользователю в качестве ответа,
                // даже если они были добавлены
                .map(it -> idsByKey.get(new GeoMultiplierKey(it.getRegionId(), false)))
                .collect(toList());
    }

    // regionId -> percent
    private Map<Long, Integer> findTranslocalRegions(List<BidModifierRegionalAdjustment> adjustments) {
        Map<Long, Integer> map = adjustments.stream().filter(o -> TRANSLOCAL_REGIONS.contains(o.getRegionId()))
                .collect(toMap(BidModifierRegionalAdjustment::getRegionId, BidModifierRegionalAdjustment::getPercent));
        return map.containsKey(Region.CRIMEA_REGION_ID) ? emptyMap() : map;
    }

    /**
     * Возвращает Крым, если в контексте текущего клиента выбраны такие регионы, в которые нужно его добавить
     * (например, выбрана Россия или СНГ в контексте русского клиента; или выбрана Украина в контексте украинского)
     * Если таких ситуаций нет или Крым указан явно в adjustments, то возвращается empty.
     */
    private Optional<BidModifierRegionalAdjustment> getHiddenRegion(List<BidModifierRegionalAdjustment> adjustments,
                                                                    ClientId clientId) {
        Map<Long, Integer> translocalRegions = findTranslocalRegions(adjustments);
        if (!translocalRegions.isEmpty()) {
            long clientCountryId = clientService.getCountryRegionIdByClientId(clientId).orElse(GLOBAL_REGION_ID);
            long superRegionId =
                    clientCountryId == Region.UKRAINE_REGION_ID ? clientCountryId : Region.RUSSIA_REGION_ID;
            // Для украинского клиента, если не задана корректировка для Украины, Крым может наследовать корректировку для СНГ
            if (superRegionId == UKRAINE_REGION_ID && !translocalRegions.containsKey(UKRAINE_REGION_ID)) {
                superRegionId = SNG_REGION_ID;
            }
            if (translocalRegions.containsKey(superRegionId)) {
                return Optional.of(new BidModifierRegionalAdjustment().withRegionId(CRIMEA_REGION_ID)
                        .withPercent(translocalRegions.get(superRegionId)));
            }
        }
        return Optional.empty();
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierGeo modifier,
                                              BidModifierRegionalAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogGeoMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.geo_multiplier,
                oldPercent, adjustment.getPercent(), adjustment.getRegionId(), adjustment.getHidden());
    }

    /**
     * Реализация для Гео учитывает скрытые регионы.
     */
    @Override
    protected List<BidModifierRegionalAdjustment> computeAllAdjustments(BidModifierGeo modifier,
                                                                        @Nullable BidModifierGeo existingModifier, ClientId clientId) {
        List<BidModifierRegionalAdjustment> allAdjustments = new ArrayList<>(modifier.getRegionalAdjustments());
        if (existingModifier != null) {
            allAdjustments.addAll(filterList(existingModifier.getRegionalAdjustments(), it -> !it.getHidden()));
        }
        return addHiddenRegionsIfNeed(allAdjustments, clientId);
    }

    @Override
    protected List<BidModifierRegionalAdjustment> computeOnlyNewAdjustments(@Nullable BidModifierGeo modifier,
                                                                            @Nullable BidModifierGeo existingModifier, ClientId clientId) {
        return modifier != null ? addHiddenRegionsIfNeed(getAdjustments(modifier), clientId) : emptyList();
    }

    /**
     * Добавляет при необходимости скрытый регион в список adjustments.
     * После этого коду, который занимается сохранением в БД, про это уже думать не обязательно.
     */
    protected List<BidModifierRegionalAdjustment> addHiddenRegionsIfNeed(
            List<BidModifierRegionalAdjustment> adjustments, ClientId clientId) {
        LocalDateTime now = LocalDateTime.now();
        return StreamEx.of(adjustments).append(
                StreamEx.of(getHiddenRegion(adjustments, clientId))
                        .map(adjustment -> new BidModifierRegionalAdjustment()
                                .withRegionId(adjustment.getRegionId())
                                .withPercent(adjustment.getPercent())
                                .withHidden(true)
                                .withLastChange(now))
        ).collect(toList());
    }

    public static class GeoMultiplierKey {
        private final long regionId;
        private final boolean hidden;

        public GeoMultiplierKey(long regionId, boolean hidden) {
            this.regionId = regionId;
            this.hidden = hidden;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            GeoMultiplierKey that = (GeoMultiplierKey) o;
            return regionId == that.regionId &&
                    hidden == that.hidden;
        }

        @Override
        public int hashCode() {
            return Objects.hash(regionId, hidden);
        }
    }

    @Override
    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierGeo> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = GEO_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    private static boolean containsAtLeastOneRegion(BidModifierGeo bidModifier, Set<Long> superRegions) {
        return bidModifier.getRegionalAdjustments().stream()
                .anyMatch(it -> superRegions.contains(it.getRegionId()));
    }

    private Map<Long, List<BidModifierRegionalAdjustment>> getHiddenAdjustmentsByParentIds(
            Collection<Long> hierarchicalMultipliersIds, DSLContext dslContext) {
        Result<Record> records = dslContext.select(REGIONAL_ADJUSTMENT_FIELDS)
                .from(GEO_MULTIPLIER_VALUES)
                .where(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(hierarchicalMultipliersIds))
                .and(GEO_MULTIPLIER_VALUES.IS_HIDDEN.eq(1L))
                .fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(GEO_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(REGIONAL_ADJUSTMENT_MAPPER::fromDb).collect(toList()))
                .toMap();
    }

    @Override
    public void delete(DSLContext txContext, ClientId clientId, long operatorUid,
                       Collection<BidModifierGeo> bidModifiers) {
        // Обрабатываем Крым
        Set<Long> superRegions = getSuperRegions(clientId);

        Set<Long> affectedHierarchicalIds = bidModifiers.stream()
                .filter(bidModifier -> containsAtLeastOneRegion(bidModifier, superRegions))
                .map(BidModifier::getId)
                .collect(toSet());

        Set<Long> idsToDelete = new HashSet<>(
                bidModifiers.stream().map(this::getAdjustments)
                        .flatMap(Collection::stream)
                        .map(BidModifierRegionalAdjustment::getId)
                        .collect(toSet()));

        if (!affectedHierarchicalIds.isEmpty()) {
            Map<Long, List<BidModifierRegionalAdjustment>> hiddenAdjustmentsByParentIds =
                    getHiddenAdjustmentsByParentIds(affectedHierarchicalIds, txContext);

            List<Long> hiddenMultiplierIds = hiddenAdjustmentsByParentIds.values().stream()
                    .flatMap(Collection::stream)
                    .map(BidModifierRegionalAdjustment::getId)
                    .collect(toList());

            idsToDelete.addAll(hiddenMultiplierIds);
        }

        if (!idsToDelete.isEmpty()) {
            deleteAndLogChanges(bidModifiers, idsToDelete, txContext, operatorUid);
        }
    }

    private Set<Long> getSuperRegions(ClientId clientId) {
        boolean isUkrainianClient = UKRAINE_REGION_ID ==
                clientService.getCountryRegionIdByClientId(clientId).orElse(GLOBAL_REGION_ID);

        return isUkrainianClient ?
                ImmutableSet.of(UKRAINE_REGION_ID, SNG_REGION_ID)
                : ImmutableSet.of(RUSSIA_REGION_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierRegionalAdjustment>> changes,
                               List<BidModifierGeo> bidModifiers, DSLContext dslContext) {
        // Обрабатываем Крым
        Set<Long> superRegions = getSuperRegions(clientId);

        // Те корректировки из супер-регионов, которые будут изменены
        Map<Long, AppliedChanges<BidModifierRegionalAdjustment>> superRegionAffectingChanges = changes.stream()
                .filter(it -> superRegions.contains(it.getModel().getRegionId()))
                .collect(toMap(it -> it.getModel().getId(), identity()));

        // Если ни одного супер-региона не будет изменено, можно не добавлять скрытые регионы
        if (superRegionAffectingChanges.isEmpty()) {
            updatePercentsCore(changes, dslContext);
            logUpdateChanges(operatorUid, changes, bidModifiers);
            return;
        }

        // Ищем корректировки со скрытыми регионами и добавляем их в changes

        // ID наборов корректировок, содержащих в себе изменённые корректировки на супер-регионы
        Set<Long> superRegionAffectingIds = superRegionAffectingChanges.keySet();

        Map<Long, Integer> percentByBidModifierId = bidModifiers.stream()
                .filter(modifier -> hasAnyAdjustmentOf(modifier, superRegionAffectingIds))
                .collect(toMap(BidModifier::getId,
                        m -> getPercentToSetOnHiddenRegions(m, superRegionAffectingChanges)));

        Map<Long, List<BidModifierRegionalAdjustment>> hiddenAdjustmentsByParentId =
                getHiddenAdjustmentsByParentIds(percentByBidModifierId.keySet(), dslContext);

        // Добавляем скрытые adjustment'ы, которые были загружены, к имеющимся наборам
        // Это понадобится для последующего логирования
        for (BidModifierGeo bidModifier : bidModifiers) {
            Long id = bidModifier.getId();
            if (hiddenAdjustmentsByParentId.containsKey(id)) {
                bidModifier.getRegionalAdjustments().addAll(hiddenAdjustmentsByParentId.get(id));
            }
        }

        List<AppliedChanges<BidModifierRegionalAdjustment>> allChanges = new ArrayList<>(changes);

        List<AppliedChanges<BidModifierRegionalAdjustment>> hiddenAppliedChanges =
                EntryStream.of(hiddenAdjustmentsByParentId)
                        .mapKeyValue((bidModifierId, hiddenAdjustments) -> {
                            Integer percentToSet = percentByBidModifierId.get(bidModifierId);
                            return hiddenAdjustments.stream()
                                    .map(adjustment -> applyChanges(adjustment, percentToSet))
                                    .collect(toList());
                        }).flatMap(Collection::stream)
                        .toList();

        allChanges.addAll(hiddenAppliedChanges);

        // Применяем изменения ко всем корректировкам и логируем изменения
        updatePercentsCore(allChanges, dslContext);
        logUpdateChanges(operatorUid, allChanges, bidModifiers);
    }

    private AppliedChanges<BidModifierRegionalAdjustment> applyChanges(
            BidModifierRegionalAdjustment adjustment,
            Integer newPercent) {
        ModelChanges<BidModifierRegionalAdjustment> modelChanges =
                new ModelChanges<>(adjustment.getId(),
                        BidModifierRegionalAdjustment.class);
        modelChanges.process(newPercent, BidModifierAdjustment.PERCENT);
        return modelChanges.applyTo(adjustment);
    }

    // Получает для указанного набора корректировок процент, который должен быть установлен на его скрытом регионе
    private int getPercentToSetOnHiddenRegions(BidModifierGeo modifier,
                                               Map<Long, AppliedChanges<BidModifierRegionalAdjustment>> superRegionAffectingChanges) {
        Set<Long> superRegionAffectingIds = superRegionAffectingChanges.keySet();

        return modifier.getRegionalAdjustments().stream()
                .map(BidModifierRegionalAdjustment::getId)
                .filter(superRegionAffectingIds::contains)
                .map(superRegionAffectingChanges::get)
                .map(changes -> changes.getNewValue(BidModifierAdjustment.PERCENT))
                .findFirst().orElseThrow(IllegalStateException::new);
    }

    private boolean hasAnyAdjustmentOf(BidModifierGeo modifier, Set<Long> adjustmentIds) {
        return modifier.getRegionalAdjustments().stream()
                .anyMatch(it -> adjustmentIds.contains(it.getId()));
    }

    private void updatePercentsCore(List<AppliedChanges<BidModifierRegionalAdjustment>> changes,
                                    DSLContext dslContext) {
        JooqUpdateBuilder<GeoMultiplierValuesRecord, BidModifierRegionalAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(BidModifierRegionalAdjustment.PERCENT,
                GEO_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(GEO_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(GEO_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();
    }

    @Override
    public void prepareSystemFields(List<BidModifierGeo> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getRegionalAdjustments()
                .forEach(regionalAdjustment -> regionalAdjustment
                        .withHidden(false)
                        .withLastChange(now)));
    }
}
