package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogInventoryMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.InventoryType;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.InventoryMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.INVENTORY_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.INVENTORY_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.INVENTORY_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.InventoryMultiplierValues.INVENTORY_MULTIPLIER_VALUES;

@Component
@ParametersAreNonnullByDefault
public class BidModifierInventoryTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierInventory, BidModifierInventoryAdjustment, InventoryType> {
    @Autowired
    public BidModifierInventoryTypeSupport(ShardHelper shardHelper, LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.INVENTORY_MULTIPLIER;
    }

    @Override
    public Class<BidModifierInventory> getBidModifierClass() {
        return BidModifierInventory.class;
    }

    @Override
    public boolean areEqual(BidModifierInventory a, BidModifierInventory b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getInventoryAdjustments()), new HashSet<>(b.getInventoryAdjustments()));
    }

    @Override
    public BidModifierInventory createEmptyBidModifierFromRecord(Record record) {
        return INVENTORY_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierInventory> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public Map<Long, List<BidModifierInventoryAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                               Collection<Long> ids) {
        Result<Record> records = dslContext.select(INVENTORY_ADJUSTMENT_FIELDS)
                .from(INVENTORY_MULTIPLIER_VALUES)
                .where(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(INVENTORY_ADJUSTMENT_MAPPER::fromDb).collect(toList()))
                .toMap();
    }

    @Override
    public void setAdjustments(BidModifierInventory modifier, List<BidModifierInventoryAdjustment> adjustments) {
        modifier.setInventoryAdjustments(adjustments);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierInventory> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(INVENTORY_ADJUSTMENT_FIELDS)
                .from(INVENTORY_MULTIPLIER_VALUES)
                .where(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierInventory> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierInventoryAdjustment> adjustments =
                            recordsList.stream().map(INVENTORY_ADJUSTMENT_MAPPER::fromDb).collect(toList());
                    bidModifiersById.get(bidModifierId).withInventoryAdjustments(adjustments);
                });
    }

    @Override
    protected InventoryType getKey(BidModifierInventoryAdjustment adjustment) {
        return adjustment.getInventoryType();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierInventoryAdjustment> adjustments, DSLContext txContext) {
        InsertHelper<InventoryMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, INVENTORY_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(INVENTORY_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierInventoryAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<InventoryMultiplierValuesRecord, BidModifierInventoryAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, INVENTORY_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(INVENTORY_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    public Class<BidModifierInventoryAdjustment> getAdjustmentClass() {
        return BidModifierInventoryAdjustment.class;
    }

    @Override
    public List<BidModifierInventoryAdjustment> getAdjustments(BidModifierInventory modifier) {
        return modifier.getInventoryAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(INVENTORY_MULTIPLIER_VALUES)
                .where(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierInventoryAdjustment> addedAdjustments,
                                     List<BidModifierInventoryAdjustment> insertedMultipliers) {
        Map<InventoryType, Long> idsByKey =
                insertedMultipliers.stream().collect(toMap(this::getKey, BidModifierInventoryAdjustment::getId));
        return addedAdjustments.stream().map(adjustment ->
                idsByKey.get(adjustment.getInventoryType()))
                .collect(toList());
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierInventory modifier,
                                              BidModifierInventoryAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogInventoryMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.inventory_multiplier,
                oldPercent, adjustment.getPercent(), InventoryType.toSource(adjustment.getInventoryType()));
    }

    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierInventory> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = INVENTORY_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierInventoryAdjustment>> changes,
                               List<BidModifierInventory> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<InventoryMultiplierValuesRecord, BidModifierInventoryAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(BidModifierInventoryAdjustment.PERCENT,
                INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(INVENTORY_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(INVENTORY_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierInventory> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getInventoryAdjustments()
                .forEach(inventoryAdjustment -> inventoryAdjustment.withLastChange(now)));
    }
}
