package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogBidModifierData;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMobileMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.OsType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierMultipleValuesTypeSupport;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.MobileMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.MOBILE_MAPPER_WITHOUT_OS_TYPE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.MOBILE_MAPPER_WITH_OS_TYPE;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.MobileMultiplierValues.MOBILE_MULTIPLIER_VALUES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BidModifierMobileTypeSupport implements BidModifierMultipleValuesTypeSupport<BidModifierMobile,
        BidModifierMobileAdjustment> {

    private static final JooqMapperWithSupplier<BidModifierMobileAdjustment> MOBILE_MAPPER_ADJUSTMENT =
            JooqMapperWithSupplierBuilder.builder(BidModifierMobileAdjustment::new)
                    .map(property(BidModifierMobileAdjustment.ID, MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID))
                    .map(integerProperty(BidModifierMobileAdjustment.PERCENT, MOBILE_MULTIPLIER_VALUES.MULTIPLIER_PCT))
                    .map(convertibleProperty(BidModifierMobileAdjustment.OS_TYPE, MOBILE_MULTIPLIER_VALUES.OS_TYPE,
                            OsType::fromSource, OsType::toSource))
                    .map(property(BidModifierMobileAdjustment.LAST_CHANGE, MOBILE_MULTIPLIER_VALUES.LAST_CHANGE))
                    .build();

    private static final Set<Field<?>> MOBILE_MAPPER_ADJUSTMENT_FIELDS = Sets.union(
            ImmutableSet.copyOf(MOBILE_MAPPER_ADJUSTMENT.getFieldsToRead()),
            singleton(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID));
    private static final ImmutableSet<ActionType>
            ACTION_TYPES_RETURN_ADDED_INFO =
            ImmutableSet.of(ActionType.INSERT, ActionType.REPLACE, ActionType.UPDATE_PERCENT);

    private final ShardHelper shardHelper;
    private final LogBidModifiersService logBidModifiersService;

    public BidModifierMobileTypeSupport(ShardHelper shardHelper,
                                        LogBidModifiersService logBidModifiersService) {
        this.shardHelper = shardHelper;
        this.logBidModifiersService = logBidModifiersService;
    }

    @Override
    public void setAdjustments(BidModifierMobile modifier,
                               List<BidModifierMobileAdjustment> bidModifierMobileAdjustments) {
        checkArgument(bidModifierMobileAdjustments.size() == 1);
        modifier.setMobileAdjustment(bidModifierMobileAdjustments.get(0));
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.MOBILE_MULTIPLIER;
    }

    @Override
    public Class<BidModifierMobile> getBidModifierClass() {
        return BidModifierMobile.class;
    }

    @Override
    public BidModifierMobile createEmptyBidModifierFromRecord(Record record) {
        return MOBILE_MAPPER_WITHOUT_OS_TYPE.fromDb(record);
    }

    @Override
    public Class<BidModifierMobileAdjustment> getAdjustmentClass() {
        return BidModifierMobileAdjustment.class;
    }

    @Override
    public boolean areEqual(BidModifierMobile a, BidModifierMobile b) {
        return Objects.equals(a, b);
    }

    @Override
    public List<BidModifierMobileAdjustment> getAdjustments(BidModifierMobile modifier) {
        return singletonList(modifier.getMobileAdjustment());
    }

    @Override
    public Map<BidModifierKey, AddedBidModifierInfo> add(DSLContext txContext,
                                                         List<BidModifierMobile> newModifiers, Map<BidModifierKey,
            BidModifierMobile> existingModifierByKey,
                                                         ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        return save(txContext, operatorUid, existingModifierByKey, newModifiers, changedCampaignsAndAdGroups);
    }

    @Override
    public Set<CampaignIdAndAdGroupIdPair> addOrReplace(DSLContext txContext,
                                                        List<BidModifierMobile> newModifiers, Map<BidModifierKey,
            BidModifierMobile> lockedExistingModifiers,
                                                        ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        save(txContext, operatorUid, lockedExistingModifiers, newModifiers, changedCampaignsAndAdGroups);
        return changedCampaignsAndAdGroups;
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierMobileAdjustment>> appliedChanges,
                               List<BidModifierMobile> bidModifiers,
                               DSLContext dslContext) {
        updatePercentAction(dslContext, operatorUid, appliedChanges, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierMobile> bidModifiers) {
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> {
            bidModifier
                    .withEnabled(nvl(bidModifier.getEnabled(), true))
                    .withLastChange(now);
        });
        bidModifiers.forEach(bidModifier -> bidModifier.getMobileAdjustment().withLastChange(now));
    }

    @Override
    public void delete(DSLContext txContext, ClientId clientId, long operatorUid,
                       Collection<BidModifierMobile> bidModifierMobiles) {
        getLockOnHierarchicalMultipliers(txContext, mapList(bidModifierMobiles, BidModifier::getId));
        deleteAction(txContext, operatorUid, bidModifierMobiles);
    }

    /**
     * Сохранить новые корректировки {@code newModifiers} в БД, учитывая уже существующие {@code existingModifierByKey}.
     * <p>
     * Для каждой корректировки определяются операции необходимые для их сохранения, затем эти операции векторизуются
     * и выполняются массово.
     * <p>
     * Для примера, если корректировка существует в {@code existingModifierByKey}, но её нет в {@code newModifiers},
     * значит она подлежит удалению, если же корректировка есть и в {@code existingModifierByKey} и в {@code
     * newModifiers},
     * то нужно обновление. Необходимые действия определяются в {@link BidModifierMobileTypeSupport#chooseAction}.
     */
    private Map<BidModifierKey, AddedBidModifierInfo> save(
            DSLContext txContext,
            long operatorUid,
            Map<BidModifierKey, BidModifierMobile> existingModifierByKey,
            List<BidModifierMobile> newModifiers,
            Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups) {
        Map<BidModifierKey, BidModifierMobile> newModifierByKey = listToMap(newModifiers, BidModifierKey::new);

        Map<ActionType, List<ActionItem>> actionItemsByType =
                StreamEx.of(Sets.union(existingModifierByKey.keySet(), newModifierByKey.keySet()))
                        // учитываем только корректировки поддерживаемого типа
                        .filter(key -> key.getType() == this.getType())
                        // join по ключу в пару ("было", "стало")
                        .mapToEntry(key -> Pair.of(existingModifierByKey.get(key), newModifierByKey.get(key)))
                        .mapKeyValue(this::chooseAction)
                        .groupingBy(ActionItem::getType);

        // Заполняем множество объектов, к которым будут применены какие-либо изменения
        for (Map.Entry<ActionType, List<ActionItem>> entry : actionItemsByType.entrySet()) {
            if (entry.getKey() != ActionType.NOPE) {
                List<ActionItem> actionItems = entry.getValue();
                for (ActionItem actionItem : actionItems) {
                    BidModifierKey key = actionItem.getKey();
                    changedCampaignsAndAdGroups.add(new CampaignIdAndAdGroupIdPair()
                            .withCampaignId(key.getCampaignId())
                            .withAdGroupId(key.getAdGroupId()));
                }
            }
        }

        Map<ActionType, Function<List<ActionItem>, List<Long>>> actionToMethod = ImmutableMap.of(
                ActionType.DELETE,
                actionItems -> deleteAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier)),
                ActionType.INSERT,
                actionItems -> insertAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier)),
                ActionType.UPDATE_PERCENT,
                actionItems -> updatePercentAction(txContext, operatorUid,
                        mapList(actionItems, ActionItem::getChanges),
                        mapList(actionItems, ActionItem::getModifier)),
                ActionType.REPLACE,
                actionItems -> {
                    deleteAction(txContext, operatorUid, mapList(actionItems, ActionItem::getReplacedModifier));
                    return insertAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier));
                },
                ActionType.NOPE,
                actionItems -> mapList(actionItems, i -> i.getModifier().getMobileAdjustment().getId())
        );

        return EntryStream.of(actionToMethod)
                .flatMapKeyValue((actionType, method) -> {
                    List<ActionItem> actionItems = actionItemsByType.get(actionType);
                    if (actionItems == null) {
                        return Stream.empty();
                    }
                    List<Long> result = method.apply(actionItems);
                    return EntryStream.zip(actionItems, result)
                            .mapKeys(ActionItem::getKey)
                            .mapValues(id -> createAddedInfo(actionType, id));
                })
                .mapToEntry(Map.Entry::getKey, Map.Entry::getValue)
                .toMap();
    }

    /**
     * Определить необходимые действия для переданной пары корректировок с ключем {@code key}.
     * Переданная пара {@code existingAndFreshPair} -- это существующая и новая корректировка.
     * На основе их совместного рассмотрения становятся понятны изменения, которые нужно произвести в БД
     */
    private ActionItem chooseAction(BidModifierKey key,
                                    Pair<BidModifierMobile, BidModifierMobile> existingAndFreshPair) {
        BidModifierMobile existing = existingAndFreshPair.getLeft();
        BidModifierMobile fresh = existingAndFreshPair.getRight();

        checkArgument(existing != null || fresh != null);
        checkArgument(existing == null
                || (existing.getId() != null && existing.getMobileAdjustment().getId() != null));

        if (existing == null) {
            return ActionItem.insert(key, fresh);
        }

        if (fresh == null) {
            return ActionItem.delete(key, existing);
        }

        BidModifierMobileAdjustment existingAdjustment = existing.getMobileAdjustment();
        BidModifierMobileAdjustment freshAdjustment = fresh.getMobileAdjustment();
        if (existingAdjustment.getOsType() == freshAdjustment.getOsType()) {
            // обновить процент
            AppliedChanges<BidModifierMobileAdjustment> changes =
                    new ModelChanges<>(existingAdjustment.getId(), BidModifierMobileAdjustment.class)
                            .process(freshAdjustment.getPercent(), BidModifierMobileAdjustment.PERCENT)
                            .applyTo(existingAdjustment);
            if (changes.hasActuallyChangedProps()) {
                return ActionItem.updatePercent(key, existing, changes);
            } else {
                return ActionItem.nope(key, existing);
            }
        } else {
            // поменялся тип ОС, значит надо удалить старую, создать новую
            return ActionItem.replace(key, existing, fresh);
        }
    }

    private AddedBidModifierInfo createAddedInfo(ActionType actionType, Long id) {
        if (ACTION_TYPES_RETURN_ADDED_INFO.contains(actionType)) {
            return AddedBidModifierInfo.added(singletonList(id));
        } else {
            return AddedBidModifierInfo.notAdded();
        }
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierMobile> bidModifiers, boolean updLock) {
        Set<Long> idsOfModifiersWithOs = StreamEx.of(bidModifiers)
                // корректировка с указанием ОС записывается без процента в родительской таблице
                .filter(m -> m.getMobileAdjustment().getPercent() == null)
                .map(BidModifier::getId)
                .toSet();

        SelectConditionStep<Record> selectStep = dslContext.select(MOBILE_MAPPER_ADJUSTMENT_FIELDS)
                .from(MOBILE_MULTIPLIER_VALUES)
                .where(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(idsOfModifiersWithOs));
        Result<Record> records = updLock ? selectStep.forUpdate().fetch() : selectStep.fetch();

        Map<Long, BidModifierMobileAdjustment> adjustmentMap = StreamEx.of(records)
                .mapToEntry(
                        record -> record.get(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class),
                        MOBILE_MAPPER_ADJUSTMENT::fromDb)
                // расчитываем, что на каждую запись в родительской есть только одна дочерняя, иначе toMap здесь упадёт
                .toMap();
        StreamEx.of(bidModifiers)
                .mapToEntry(BidModifier::getId)
                .mapValues(adjustmentMap::get)
                .nonNullValues()
                .forKeyValue(BidModifierMobile::setMobileAdjustment);
    }

    @Override
    public Map<Long, List<BidModifierMobileAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                            Collection<Long> ids) {
        // расчитываем, что на каждую запись в родительской есть только одна дочерняя
        return dslContext.select(MOBILE_MAPPER_ADJUSTMENT_FIELDS)
                .from(MOBILE_MULTIPLIER_VALUES)
                .where(MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID.in(ids))
                .fetchMap(
                        record -> record.get(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class),
                        record -> singletonList(MOBILE_MAPPER_ADJUSTMENT.fromDb(record))
                );
    }

    private List<Long> insertAction(DSLContext txContext, long operatorUid,
                                    List<BidModifierMobile> modifiers) {
        Map<Boolean, List<BidModifierMobile>> modifiersByOsTypePresence = StreamEx.of(modifiers)
                .mapToEntry(modifier -> modifier.getMobileAdjustment().getOsType() != null, Function.identity())
                .grouping();
        EntryStream.of(modifiersByOsTypePresence)
                .forKeyValue((isWithOs, modifiersToInsert) -> {
                    if (isWithOs) {
                        insertWithOs(txContext, modifiersToInsert);
                    } else {
                        insertWithoutOs(txContext, modifiersToInsert);
                    }
                });

        // проверим, что все id были заполнены
        checkState(StreamEx.of(modifiers)
                .map(BidModifierMobile::getMobileAdjustment)
                .map(BidModifierAdjustment::getId)
                .anyMatch(Objects::nonNull));

        logInsert(operatorUid, modifiers);
        return mapList(modifiers, m -> m.getMobileAdjustment().getId());
    }

    private void insertWithoutOs(DSLContext txContext, List<BidModifierMobile> modifiers) {
        List<Long> ids = shardHelper.generateHierarchicalMultiplierIds(modifiers.size());
        EntryStream.zip(ids, modifiers)
                .forKeyValue((id, modifier) -> {
                    modifier.setId(id);
                    modifier.getMobileAdjustment().setId(id);
                });
        new InsertHelper<>(txContext, HIERARCHICAL_MULTIPLIERS)
                .addAll(MOBILE_MAPPER_WITHOUT_OS_TYPE, modifiers)
                .execute();
    }

    private void insertWithOs(DSLContext txContext, List<BidModifierMobile> modifiers) {
        List<Long> ids = shardHelper.generateHierarchicalMultiplierIds(modifiers.size() * 2);
        List<Long> parentIds = ids.subList(0, modifiers.size());
        List<Long> childIds = ids.subList(modifiers.size(), ids.size());
        EntryStream.zip(parentIds, modifiers).forKeyValue((id, modifier) -> modifier.setId(id));
        EntryStream.zip(childIds, modifiers).forKeyValue((id, modifier) -> modifier.getMobileAdjustment().setId(id));

        new InsertHelper<>(txContext, HIERARCHICAL_MULTIPLIERS)
                .addAll(MOBILE_MAPPER_WITH_OS_TYPE, modifiers)
                .execute();

        InsertHelper<MobileMultiplierValuesRecord> insertHelper =
                new InsertHelper<>(txContext, MOBILE_MULTIPLIER_VALUES);
        StreamEx.of(modifiers)
                .mapToEntry(BidModifier::getId, BidModifierMobile::getMobileAdjustment)
                .forKeyValue((parentId, adjustment) -> {
                    insertHelper.add(MOBILE_MAPPER_ADJUSTMENT, adjustment);
                    insertHelper.set(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, parentId);
                    insertHelper.newRecord();
                });
        insertHelper.executeIfRecordsAdded();
    }

    private List<Long> deleteAction(DSLContext txContext, long operatorUid,
                                    Collection<BidModifierMobile> bidModifierMobiles) {
        // txContext сюда должен приходить с начатой транзакцией и блокированными строками в HIERARCHICAL_MULTIPLIERS.

        // удаляем сначала из дочерней, потом из родительской таблицы
        Set<Long> mobileMultiplierValueIds = StreamEx.of(bidModifierMobiles)
                .remove(modifier -> modifier.getMobileAdjustment().getId().equals(modifier.getId()))
                .map(modifier -> modifier.getMobileAdjustment().getId())
                .toSet();
        txContext.deleteFrom(MOBILE_MULTIPLIER_VALUES)
                .where(MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID.in(mobileMultiplierValueIds))
                .execute();
        txContext.deleteFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        mapList(bidModifierMobiles, BidModifier::getId)))
                .execute();

        logDelete(operatorUid, bidModifierMobiles);

        return mapList(bidModifierMobiles, m -> m.getMobileAdjustment().getId());
    }

    private List<Long> updatePercentAction(DSLContext dslContext, long operatorUid,
                                           List<AppliedChanges<BidModifierMobileAdjustment>> appliedChanges,
                                           List<BidModifierMobile> bidModifiers) {
        Map<Boolean, List<AppliedChanges<BidModifierMobileAdjustment>>> changesByOsTypePresence =
                StreamEx.of(appliedChanges)
                        .mapToEntry(AppliedChanges::getModel, Function.identity())
                        .mapKeys(BidModifierMobileAdjustment::getOsType)
                        .mapKeys(Objects::nonNull)
                        .grouping();
        EntryStream.of(changesByOsTypePresence)
                .forKeyValue((isWithOs, changes) -> {
                    if (isWithOs) {
                        updatePercentWithOs(dslContext, changes);
                    } else {
                        updatePercentWithoutOs(dslContext, changes);
                    }
                });
        logUpdatePercentChanges(operatorUid, appliedChanges, bidModifiers);
        return mapList(bidModifiers, m -> m.getMobileAdjustment().getId());
    }

    private void updatePercentWithoutOs(DSLContext dslContext,
                                        List<AppliedChanges<BidModifierMobileAdjustment>> appliedChanges) {
        Set<Long> affectedIds = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(BidModifierAdjustment::getId)
                .toSet();

        JooqUpdateBuilder<HierarchicalMultipliersRecord, BidModifierMobileAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, appliedChanges);

        updateBuilder.processProperty(
                BidModifierMobileAdjustment.PERCENT, HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(HIERARCHICAL_MULTIPLIERS)
                .set(updateBuilder.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(affectedIds))
                .execute();
    }

    private void updatePercentWithOs(DSLContext dslContext,
                                     List<AppliedChanges<BidModifierMobileAdjustment>> appliedChanges) {
        Set<Long> affectedIds = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(BidModifierAdjustment::getId)
                .toSet();

        JooqUpdateBuilder<MobileMultiplierValuesRecord, BidModifierMobileAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID, appliedChanges);

        updateBuilder.processProperty(
                BidModifierMobileAdjustment.PERCENT, MOBILE_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(HIERARCHICAL_MULTIPLIERS.innerJoin(MOBILE_MULTIPLIER_VALUES)
                        .on(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID
                                .eq(MOBILE_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)))
                .set(updateBuilder.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .set(MOBILE_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(MOBILE_MULTIPLIER_VALUES.MOBILE_MULTIPLIER_VALUE_ID.in(affectedIds))
                .execute();
    }

    private void logInsert(long operatorUid, List<BidModifierMobile> bidModifiers) {
        List<LogBidModifierData> logItems = StreamEx.of(bidModifiers)
                .map(modifier -> new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                        .withInserted(singletonList(createLogMultiplierInfo(modifier))))
                .toList();
        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }

    private void logDelete(long operatorUid, Collection<BidModifierMobile> bidModifierMobiles) {
        List<LogBidModifierData> logItems = StreamEx.of(bidModifierMobiles)
                .map(modifier -> {
                    LogBidModifierData logBidModifierData =
                            new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                                    .withDeletedSet(modifier.getId());
                    if (!modifier.getId().equals(modifier.getMobileAdjustment().getId())) {
                        logBidModifierData.withDeleted(singletonList(createLogMultiplierInfo(modifier)));
                    }
                    return logBidModifierData;
                })
                .toList();
        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }

    private void logUpdatePercentChanges(long operatorUid, List<AppliedChanges<BidModifierMobileAdjustment>> changes,
                                         List<BidModifierMobile> bidModifiers) {
        Map<Long, BidModifierMobile> modifierByAdjustmentId = StreamEx.of(bidModifiers)
                .mapToEntry(b -> b.getMobileAdjustment().getId(), Function.identity())
                .toMap();
        List<LogBidModifierData> logDataItems = StreamEx.of(changes)
                .mapToEntry(AppliedChanges::getModel, ac -> ac.getOldValue(BidModifierMobileAdjustment.PERCENT))
                .mapKeyValue((newAdjustment, oldPercent) -> {
                    BidModifierMobile modifier = checkNotNull(modifierByAdjustmentId.get(newAdjustment.getId()));
                    return new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                            .withUpdated(singletonList(createLogMultiplierInfo(modifier, newAdjustment, oldPercent)));
                })
                .toList();
        logBidModifiersService.logBidModifiers(logDataItems, operatorUid);
    }

    private LogMobileMultiplierInfo createLogMultiplierInfo(BidModifierMobile modifier,
                                                            BidModifierMobileAdjustment adjustment,
                                                            @Nullable Integer oldPercent) {
        return new LogMobileMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.mobile_multiplier,
                oldPercent, adjustment.getPercent(), OsType.toSource(adjustment.getOsType()));
    }

    private LogMobileMultiplierInfo createLogMultiplierInfo(BidModifierMobile modifier) {
        BidModifierMobileAdjustment adjustment = modifier.getMobileAdjustment();
        return new LogMobileMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.mobile_multiplier,
                null, adjustment.getPercent(), OsType.toSource(adjustment.getOsType()));
    }

    private void getLockOnHierarchicalMultipliers(DSLContext txContext, Collection<Long> hierarchicalMultiplierIds) {
        txContext.select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(hierarchicalMultiplierIds))
                .forUpdate()
                .execute();
    }

    enum ActionType {
        INSERT,
        DELETE,
        UPDATE_PERCENT,
        REPLACE,
        NOPE
    }

    static class ActionItem {
        private final ActionType type;
        private final BidModifierKey key;
        private final BidModifierMobile modifier;
        private final AppliedChanges<BidModifierMobileAdjustment> changes;
        private final BidModifierMobile replacedModifier;

        private ActionItem(ActionType type, BidModifierKey key, @Nullable BidModifierMobile modifier,
                           @Nullable BidModifierMobile replacedModifier,
                           @Nullable AppliedChanges<BidModifierMobileAdjustment> changes) {
            this.type = type;
            this.key = key;
            this.modifier = modifier;
            this.changes = changes;
            this.replacedModifier = replacedModifier;
        }

        static ActionItem insert(BidModifierKey key, BidModifierMobile modifier) {
            return new ActionItem(ActionType.INSERT, key, modifier, null, null);
        }

        static ActionItem delete(BidModifierKey key, BidModifierMobile modifier) {
            return new ActionItem(ActionType.DELETE, key, modifier, null, null);
        }

        static ActionItem updatePercent(BidModifierKey key, BidModifierMobile modifier,
                                        AppliedChanges<BidModifierMobileAdjustment> changes) {
            return new ActionItem(ActionType.UPDATE_PERCENT, key, modifier, null, changes);
        }

        static ActionItem replace(BidModifierKey key, BidModifierMobile existing, BidModifierMobile fresh) {
            return new ActionItem(ActionType.REPLACE, key, fresh, existing, null);
        }

        static ActionItem nope(BidModifierKey key, BidModifierMobile existing) {
            return new ActionItem(ActionType.NOPE, key, existing, null, null);
        }

        public ActionType getType() {
            return type;
        }

        public BidModifierKey getKey() {
            return key;
        }

        public BidModifierMobile getModifier() {
            return modifier;
        }

        public BidModifierMobile getReplacedModifier() {
            return replacedModifier;
        }

        public AppliedChanges<BidModifierMobileAdjustment> getChanges() {
            return changes;
        }


    }
}
