package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogBidModifierData;
import ru.yandex.direct.common.log.container.bidmodifiers.LogTabletMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.TabletOsType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierMultipleValuesTypeSupport;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.TabletMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.TABLET_MAPPER_WITHOUT_OS_TYPE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.TABLET_MAPPER_WITH_OS_TYPE;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.TabletMultiplierValues.TABLET_MULTIPLIER_VALUES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BidModifierTabletTypeSupport implements BidModifierMultipleValuesTypeSupport<BidModifierTablet,
        BidModifierTabletAdjustment> {

    private static final JooqMapperWithSupplier<BidModifierTabletAdjustment> TABLET_MAPPER_ADJUSTMENT =
            JooqMapperWithSupplierBuilder.builder(BidModifierTabletAdjustment::new)
                    .map(property(BidModifierTabletAdjustment.ID, TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID))
                    .map(integerProperty(BidModifierTabletAdjustment.PERCENT, TABLET_MULTIPLIER_VALUES.MULTIPLIER_PCT))
                    .map(convertibleProperty(BidModifierTabletAdjustment.OS_TYPE, TABLET_MULTIPLIER_VALUES.OS_TYPE,
                            TabletOsType::fromSource, TabletOsType::toSource))
                    .map(property(BidModifierTabletAdjustment.LAST_CHANGE, TABLET_MULTIPLIER_VALUES.LAST_CHANGE))
                    .build();

    private static final Set<Field<?>> TABLET_MAPPER_ADJUSTMENT_FIELDS = Sets.union(
            ImmutableSet.copyOf(TABLET_MAPPER_ADJUSTMENT.getFieldsToRead()),
            singleton(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID));
    private static final ImmutableSet<ActionType>
            ACTION_TYPES_RETURN_ADDED_INFO =
            ImmutableSet.of(ActionType.INSERT, ActionType.REPLACE, ActionType.UPDATE_PERCENT);

    private final ShardHelper shardHelper;
    private final LogBidModifiersService logBidModifiersService;

    public BidModifierTabletTypeSupport(ShardHelper shardHelper,
                                        LogBidModifiersService logBidModifiersService) {
        this.shardHelper = shardHelper;
        this.logBidModifiersService = logBidModifiersService;
    }

    @Override
    public void setAdjustments(BidModifierTablet modifier,
                               List<BidModifierTabletAdjustment> bidModifierTabletAdjustments) {
        checkArgument(bidModifierTabletAdjustments.size() == 1);
        modifier.setTabletAdjustment(bidModifierTabletAdjustments.get(0));
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.TABLET_MULTIPLIER;
    }

    @Override
    public Class<BidModifierTablet> getBidModifierClass() {
        return BidModifierTablet.class;
    }

    @Override
    public BidModifierTablet createEmptyBidModifierFromRecord(Record record) {
        return TABLET_MAPPER_WITHOUT_OS_TYPE.fromDb(record);
    }

    @Override
    public Class<BidModifierTabletAdjustment> getAdjustmentClass() {
        return BidModifierTabletAdjustment.class;
    }

    @Override
    public boolean areEqual(BidModifierTablet a, BidModifierTablet b) {
        return Objects.equals(a, b);
    }

    @Override
    public List<BidModifierTabletAdjustment> getAdjustments(BidModifierTablet modifier) {
        return singletonList(modifier.getTabletAdjustment());
    }

    @Override
    public Map<BidModifierKey, AddedBidModifierInfo> add(DSLContext txContext,
                                                         List<BidModifierTablet> newModifiers, Map<BidModifierKey,
            BidModifierTablet> existingModifierByKey,
                                                         ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        return save(txContext, operatorUid, existingModifierByKey, newModifiers, changedCampaignsAndAdGroups);
    }

    @Override
    public Set<CampaignIdAndAdGroupIdPair> addOrReplace(DSLContext txContext,
                                                        List<BidModifierTablet> newModifiers, Map<BidModifierKey,
            BidModifierTablet> lockedExistingModifiers,
                                                        ClientId clientId, long operatorUid) {
        Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = new HashSet<>();
        save(txContext, operatorUid, lockedExistingModifiers, newModifiers, changedCampaignsAndAdGroups);
        return changedCampaignsAndAdGroups;
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierTabletAdjustment>> appliedChanges,
                               List<BidModifierTablet> bidModifiers,
                               DSLContext dslContext) {
        updatePercentAction(dslContext, operatorUid, appliedChanges, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierTablet> bidModifiers) {
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier
                .withEnabled(nvl(bidModifier.getEnabled(), true))
                .withLastChange(now));
        bidModifiers.forEach(bidModifier -> bidModifier.getTabletAdjustment().withLastChange(now));
    }

    @Override
    public void delete(DSLContext txContext, ClientId clientId, long operatorUid,
                       Collection<BidModifierTablet> bidModifierTablets) {
        getLockOnHierarchicalMultipliers(txContext, mapList(bidModifierTablets, BidModifier::getId));
        deleteAction(txContext, operatorUid, bidModifierTablets);
    }

    /**
     * Сохранить новые корректировки {@code newModifiers} в БД, учитывая уже существующие {@code existingModifierByKey}.
     * <p>
     * Для каждой корректировки определяются операции необходимые для их сохранения, затем эти операции векторизуются
     * и выполняются массово.
     * <p>
     * Для примера, если корректировка существует в {@code existingModifierByKey}, но её нет в {@code newModifiers},
     * значит она подлежит удалению, если же корректировка есть и в {@code existingModifierByKey} и в {@code
     * newModifiers},
     * то нужно обновление. Необходимые действия определяются в {@link BidModifierTabletTypeSupport#chooseAction}.
     */
    private Map<BidModifierKey, AddedBidModifierInfo> save(
            DSLContext txContext,
            long operatorUid,
            Map<BidModifierKey, BidModifierTablet> existingModifierByKey,
            List<BidModifierTablet> newModifiers,
            Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups) {
        Map<BidModifierKey, BidModifierTablet> newModifierByKey = listToMap(newModifiers, BidModifierKey::new);

        Map<ActionType, List<ActionItem>> actionItemsByType =
                StreamEx.of(Sets.union(existingModifierByKey.keySet(), newModifierByKey.keySet()))
                        // учитываем только корректировки поддерживаемого типа
                        .filter(key -> key.getType() == this.getType())
                        // join по ключу в пару ("было", "стало")
                        .mapToEntry(key -> Pair.of(existingModifierByKey.get(key), newModifierByKey.get(key)))
                        .mapKeyValue(this::chooseAction)
                        .groupingBy(ActionItem::getType);

        // Заполняем множество объектов, к которым будут применены какие-либо изменения
        for (Map.Entry<ActionType, List<ActionItem>> entry : actionItemsByType.entrySet()) {
            if (entry.getKey() != ActionType.NOPE) {
                List<ActionItem> actionItems = entry.getValue();
                for (ActionItem actionItem : actionItems) {
                    BidModifierKey key = actionItem.getKey();
                    changedCampaignsAndAdGroups.add(
                            new CampaignIdAndAdGroupIdPair()
                                    .withCampaignId(key.getCampaignId())
                                    .withAdGroupId(key.getAdGroupId()));
                }
            }
        }

        Map<ActionType, Function<List<ActionItem>, List<Long>>> actionToMethod = ImmutableMap.of(
                ActionType.DELETE,
                actionItems -> deleteAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier)),
                ActionType.INSERT,
                actionItems -> insertAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier)),
                ActionType.UPDATE_PERCENT,
                actionItems -> updatePercentAction(txContext, operatorUid,
                        mapList(actionItems, ActionItem::getChanges),
                        mapList(actionItems, ActionItem::getModifier)),
                ActionType.REPLACE,
                actionItems -> {
                    deleteAction(txContext, operatorUid, mapList(actionItems, ActionItem::getReplacedModifier));
                    return insertAction(txContext, operatorUid, mapList(actionItems, ActionItem::getModifier));
                },
                ActionType.NOPE,
                actionItems -> mapList(actionItems, i -> i.getModifier().getTabletAdjustment().getId())
        );

        return EntryStream.of(actionToMethod)
                .flatMapKeyValue((actionType, method) -> {
                    List<ActionItem> actionItems = actionItemsByType.get(actionType);
                    if (actionItems == null) {
                        return Stream.empty();
                    }
                    List<Long> result = method.apply(actionItems);
                    return EntryStream.zip(actionItems, result)
                            .mapKeys(ActionItem::getKey)
                            .mapValues(id -> createAddedInfo(actionType, id));
                })
                .mapToEntry(Map.Entry::getKey, Map.Entry::getValue)
                .toMap();
    }

    /**
     * Определить необходимые действия для переданной пары корректировок с ключем {@code key}.
     * Переданная пара {@code existingAndFreshPair} -- это существующая и новая корректировка.
     * На основе их совместного рассмотрения становятся понятны изменения, которые нужно произвести в БД
     */
    private ActionItem chooseAction(BidModifierKey key,
                                    Pair<BidModifierTablet, BidModifierTablet> existingAndFreshPair) {
        BidModifierTablet existing = existingAndFreshPair.getLeft();
        BidModifierTablet fresh = existingAndFreshPair.getRight();

        checkArgument(existing != null || fresh != null);
        checkArgument(existing == null
                || (existing.getId() != null && existing.getTabletAdjustment().getId() != null));

        if (existing == null) {
            return ActionItem.insert(key, fresh);
        }

        if (fresh == null) {
            return ActionItem.delete(key, existing);
        }

        BidModifierTabletAdjustment existingAdjustment = existing.getTabletAdjustment();
        BidModifierTabletAdjustment freshAdjustment = fresh.getTabletAdjustment();
        if (existingAdjustment.getOsType() == freshAdjustment.getOsType()) {
            // обновить процент
            AppliedChanges<BidModifierTabletAdjustment> changes =
                    new ModelChanges<>(existingAdjustment.getId(), BidModifierTabletAdjustment.class)
                            .process(freshAdjustment.getPercent(), BidModifierTabletAdjustment.PERCENT)
                            .applyTo(existingAdjustment);
            if (changes.hasActuallyChangedProps()) {
                return ActionItem.updatePercent(key, existing, changes);
            } else {
                return ActionItem.nope(key, existing);
            }
        } else {
            // поменялся тип ОС, значит надо удалить старую, создать новую
            return ActionItem.replace(key, existing, fresh);
        }
    }

    private AddedBidModifierInfo createAddedInfo(ActionType actionType, Long id) {
        if (ACTION_TYPES_RETURN_ADDED_INFO.contains(actionType)) {
            return AddedBidModifierInfo.added(singletonList(id));
        } else {
            return AddedBidModifierInfo.notAdded();
        }
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierTablet> bidModifiers, boolean updLock) {
        Set<Long> idsOfModifiersWithOs = StreamEx.of(bidModifiers)
                // корректировка с указанием ОС записывается без процента в родительской таблице
                .filter(m -> m.getTabletAdjustment().getPercent() == null)
                .map(BidModifier::getId)
                .toSet();

        SelectConditionStep<Record> selectStep = dslContext.select(TABLET_MAPPER_ADJUSTMENT_FIELDS)
                .from(TABLET_MULTIPLIER_VALUES)
                .where(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(idsOfModifiersWithOs));
        Result<Record> records = updLock ? selectStep.forUpdate().fetch() : selectStep.fetch();

        Map<Long, BidModifierTabletAdjustment> adjustmentMap = StreamEx.of(records)
                .mapToEntry(
                        record -> record.get(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class),
                        TABLET_MAPPER_ADJUSTMENT::fromDb)
                // расчитываем, что на каждую запись в родительской есть только одна дочерняя, иначе toMap здесь упадёт
                .toMap();
        StreamEx.of(bidModifiers)
                .mapToEntry(BidModifier::getId)
                .mapValues(adjustmentMap::get)
                .nonNullValues()
                .forKeyValue(BidModifierTablet::setTabletAdjustment);
    }

    @Override
    public Map<Long, List<BidModifierTabletAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                            Collection<Long> ids) {
        // расчитываем, что на каждую запись в родительской есть только одна дочерняя
        return dslContext.select(TABLET_MAPPER_ADJUSTMENT_FIELDS)
                .from(TABLET_MULTIPLIER_VALUES)
                .where(TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID.in(ids))
                .fetchMap(
                        record -> record.get(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class),
                        record -> singletonList(TABLET_MAPPER_ADJUSTMENT.fromDb(record))
                );
    }

    private List<Long> insertAction(DSLContext txContext, long operatorUid,
                                    List<BidModifierTablet> modifiers) {
        Map<Boolean, List<BidModifierTablet>> modifiersByOsTypePresence = StreamEx.of(modifiers)
                .mapToEntry(modifier -> modifier.getTabletAdjustment().getOsType() != null, Function.identity())
                .grouping();
        EntryStream.of(modifiersByOsTypePresence)
                .forKeyValue((isWithOs, modifiersToInsert) -> {
                    if (isWithOs) {
                        insertWithOs(txContext, modifiersToInsert);
                    } else {
                        insertWithoutOs(txContext, modifiersToInsert);
                    }
                });

        // проверим, что все id были заполнены
        checkState(StreamEx.of(modifiers)
                .map(BidModifierTablet::getTabletAdjustment)
                .map(BidModifierAdjustment::getId)
                .anyMatch(Objects::nonNull));

        logInsert(operatorUid, modifiers);
        return mapList(modifiers, m -> m.getTabletAdjustment().getId());
    }

    private void insertWithoutOs(DSLContext txContext, List<BidModifierTablet> modifiers) {
        List<Long> ids = shardHelper.generateHierarchicalMultiplierIds(modifiers.size());
        EntryStream.zip(ids, modifiers)
                .forKeyValue((id, modifier) -> {
                    modifier.setId(id);
                    modifier.getTabletAdjustment().setId(id);
                });
        new InsertHelper<>(txContext, HIERARCHICAL_MULTIPLIERS)
                .addAll(TABLET_MAPPER_WITHOUT_OS_TYPE, modifiers)
                .execute();
    }

    private void insertWithOs(DSLContext txContext, List<BidModifierTablet> modifiers) {
        List<Long> ids = shardHelper.generateHierarchicalMultiplierIds(modifiers.size() * 2);
        List<Long> parentIds = ids.subList(0, modifiers.size());
        List<Long> childIds = ids.subList(modifiers.size(), ids.size());
        EntryStream.zip(parentIds, modifiers).forKeyValue((id, modifier) -> modifier.setId(id));
        EntryStream.zip(childIds, modifiers).forKeyValue((id, modifier) -> modifier.getTabletAdjustment().setId(id));

        new InsertHelper<>(txContext, HIERARCHICAL_MULTIPLIERS)
                .addAll(TABLET_MAPPER_WITH_OS_TYPE, modifiers)
                .execute();

        InsertHelper<TabletMultiplierValuesRecord> insertHelper =
                new InsertHelper<>(txContext, TABLET_MULTIPLIER_VALUES);
        StreamEx.of(modifiers)
                .mapToEntry(BidModifier::getId, BidModifierTablet::getTabletAdjustment)
                .forKeyValue((parentId, adjustment) -> {
                    insertHelper.add(TABLET_MAPPER_ADJUSTMENT, adjustment);
                    insertHelper.set(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, parentId);
                    insertHelper.newRecord();
                });
        insertHelper.executeIfRecordsAdded();
    }

    private List<Long> deleteAction(DSLContext txContext, long operatorUid,
                                    Collection<BidModifierTablet> bidModifierTablets) {
        // txContext сюда должен приходить с начатой транзакцией и блокированными строками в HIERARCHICAL_MULTIPLIERS.

        // удаляем сначала из дочерней, потом из родительской таблицы
        Set<Long> tabletMultiplierValueIds = StreamEx.of(bidModifierTablets)
                .remove(modifier -> modifier.getTabletAdjustment().getId().equals(modifier.getId()))
                .map(modifier -> modifier.getTabletAdjustment().getId())
                .toSet();
        txContext.deleteFrom(TABLET_MULTIPLIER_VALUES)
                .where(TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID.in(tabletMultiplierValueIds))
                .execute();
        txContext.deleteFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        mapList(bidModifierTablets, BidModifier::getId)))
                .execute();

        logDelete(operatorUid, bidModifierTablets);

        return mapList(bidModifierTablets, m -> m.getTabletAdjustment().getId());
    }

    private List<Long> updatePercentAction(DSLContext dslContext, long operatorUid,
                                           List<AppliedChanges<BidModifierTabletAdjustment>> appliedChanges,
                                           List<BidModifierTablet> bidModifiers) {
        Map<Boolean, List<AppliedChanges<BidModifierTabletAdjustment>>> changesByOsTypePresence =
                StreamEx.of(appliedChanges)
                        .mapToEntry(AppliedChanges::getModel, Function.identity())
                        .mapKeys(BidModifierTabletAdjustment::getOsType)
                        .mapKeys(Objects::nonNull)
                        .grouping();
        EntryStream.of(changesByOsTypePresence)
                .forKeyValue((isWithOs, changes) -> {
                    if (isWithOs) {
                        updatePercentWithOs(dslContext, changes);
                    } else {
                        updatePercentWithoutOs(dslContext, changes);
                    }
                });
        logUpdatePercentChanges(operatorUid, appliedChanges, bidModifiers);
        return mapList(bidModifiers, m -> m.getTabletAdjustment().getId());
    }

    private void updatePercentWithoutOs(DSLContext dslContext,
                                        List<AppliedChanges<BidModifierTabletAdjustment>> appliedChanges) {
        Set<Long> affectedIds = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(BidModifierAdjustment::getId)
                .toSet();

        JooqUpdateBuilder<HierarchicalMultipliersRecord, BidModifierTabletAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, appliedChanges);

        updateBuilder.processProperty(
                BidModifierTabletAdjustment.PERCENT, HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(HIERARCHICAL_MULTIPLIERS)
                .set(updateBuilder.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(affectedIds))
                .execute();
    }

    private void updatePercentWithOs(DSLContext dslContext,
                                     List<AppliedChanges<BidModifierTabletAdjustment>> appliedChanges) {
        Set<Long> affectedIds = StreamEx.of(appliedChanges)
                .map(AppliedChanges::getModel)
                .map(BidModifierAdjustment::getId)
                .toSet();

        JooqUpdateBuilder<TabletMultiplierValuesRecord, BidModifierTabletAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID, appliedChanges);

        updateBuilder.processProperty(
                BidModifierTabletAdjustment.PERCENT, TABLET_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(HIERARCHICAL_MULTIPLIERS.innerJoin(TABLET_MULTIPLIER_VALUES)
                        .on(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID
                                .eq(TABLET_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)))
                .set(updateBuilder.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .set(TABLET_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(TABLET_MULTIPLIER_VALUES.TABLET_MULTIPLIER_VALUE_ID.in(affectedIds))
                .execute();
    }

    private void logInsert(long operatorUid, List<BidModifierTablet> bidModifiers) {
        List<LogBidModifierData> logItems = StreamEx.of(bidModifiers)
                .map(modifier -> new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                        .withInserted(singletonList(createLogMultiplierInfo(modifier))))
                .toList();
        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }

    private void logDelete(long operatorUid, Collection<BidModifierTablet> bidModifierTablets) {
        List<LogBidModifierData> logItems = StreamEx.of(bidModifierTablets)
                .map(modifier -> {
                    LogBidModifierData logBidModifierData =
                            new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                                    .withDeletedSet(modifier.getId());
                    if (!modifier.getId().equals(modifier.getTabletAdjustment().getId())) {
                        logBidModifierData.withDeleted(singletonList(createLogMultiplierInfo(modifier)));
                    }
                    return logBidModifierData;
                })
                .toList();
        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }

    private void logUpdatePercentChanges(long operatorUid, List<AppliedChanges<BidModifierTabletAdjustment>> changes,
                                         List<BidModifierTablet> bidModifiers) {
        Map<Long, BidModifierTablet> modifierByAdjustmentId = StreamEx.of(bidModifiers)
                .mapToEntry(b -> b.getTabletAdjustment().getId(), Function.identity())
                .toMap();
        List<LogBidModifierData> logDataItems = StreamEx.of(changes)
                .mapToEntry(AppliedChanges::getModel, ac -> ac.getOldValue(BidModifierTabletAdjustment.PERCENT))
                .mapKeyValue((newAdjustment, oldPercent) -> {
                    BidModifierTablet modifier = checkNotNull(modifierByAdjustmentId.get(newAdjustment.getId()));
                    return new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                            .withUpdated(singletonList(createLogMultiplierInfo(modifier, newAdjustment, oldPercent)));
                })
                .toList();
        logBidModifiersService.logBidModifiers(logDataItems, operatorUid);
    }

    private LogTabletMultiplierInfo createLogMultiplierInfo(BidModifierTablet modifier,
                                                            BidModifierTabletAdjustment adjustment,
                                                            @Nullable Integer oldPercent) {
        return new LogTabletMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.tablet_multiplier,
                oldPercent, adjustment.getPercent(), TabletOsType.toSource(adjustment.getOsType()));
    }

    private LogTabletMultiplierInfo createLogMultiplierInfo(BidModifierTablet modifier) {
        BidModifierTabletAdjustment adjustment = modifier.getTabletAdjustment();
        return new LogTabletMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.tablet_multiplier,
                null, adjustment.getPercent(), TabletOsType.toSource(adjustment.getOsType()));
    }

    private void getLockOnHierarchicalMultipliers(DSLContext txContext, Collection<Long> hierarchicalMultiplierIds) {
        txContext.select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(hierarchicalMultiplierIds))
                .forUpdate()
                .execute();
    }

    enum ActionType {
        INSERT,
        DELETE,
        UPDATE_PERCENT,
        REPLACE,
        NOPE
    }

    static class ActionItem {
        private final ActionType type;
        private final BidModifierKey key;
        private final BidModifierTablet modifier;
        private final AppliedChanges<BidModifierTabletAdjustment> changes;
        private final BidModifierTablet replacedModifier;

        private ActionItem(ActionType type, BidModifierKey key, @Nullable BidModifierTablet modifier,
                           @Nullable BidModifierTablet replacedModifier,
                           @Nullable AppliedChanges<BidModifierTabletAdjustment> changes) {
            this.type = type;
            this.key = key;
            this.modifier = modifier;
            this.changes = changes;
            this.replacedModifier = replacedModifier;
        }

        static ActionItem insert(BidModifierKey key, BidModifierTablet modifier) {
            return new ActionItem(ActionType.INSERT, key, modifier, null, null);
        }

        static ActionItem delete(BidModifierKey key, BidModifierTablet modifier) {
            return new ActionItem(ActionType.DELETE, key, modifier, null, null);
        }

        static ActionItem updatePercent(BidModifierKey key, BidModifierTablet modifier,
                                        AppliedChanges<BidModifierTabletAdjustment> changes) {
            return new ActionItem(ActionType.UPDATE_PERCENT, key, modifier, null, changes);
        }

        static ActionItem replace(BidModifierKey key, BidModifierTablet existing, BidModifierTablet fresh) {
            return new ActionItem(ActionType.REPLACE, key, fresh, existing, null);
        }

        static ActionItem nope(BidModifierKey key, BidModifierTablet existing) {
            return new ActionItem(ActionType.NOPE, key, existing, null, null);
        }

        public ActionType getType() {
            return type;
        }

        public BidModifierKey getKey() {
            return key;
        }

        public BidModifierTablet getModifier() {
            return modifier;
        }

        public BidModifierTablet getReplacedModifier() {
            return replacedModifier;
        }

        public AppliedChanges<BidModifierTabletAdjustment> getChanges() {
            return changes;
        }


    }
}
