package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogTrafaretPositionMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPositionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.TrafaretPosition;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.TrafaretPositionMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustmentMultiple.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.TRAFARET_POSITION_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.TRAFARET_POSITION_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.TRAFARET_POSITION_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.TrafaretPositionMultiplierValues.TRAFARET_POSITION_MULTIPLIER_VALUES;

@Component
@ParametersAreNonnullByDefault
public class BidModifierTrafaretPositionTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierTrafaretPosition, BidModifierTrafaretPositionAdjustment, TrafaretPosition> {
    @Autowired
    public BidModifierTrafaretPositionTypeSupport(ShardHelper shardHelper,
                                                  LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.TRAFARET_POSITION_MULTIPLIER;
    }

    @Override
    public Class<BidModifierTrafaretPosition> getBidModifierClass() {
        return BidModifierTrafaretPosition.class;
    }

    @Override
    public boolean areEqual(BidModifierTrafaretPosition a, BidModifierTrafaretPosition b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getTrafaretPositionAdjustments()),
                        new HashSet<>(b.getTrafaretPositionAdjustments()));
    }

    @Override
    public BidModifierTrafaretPosition createEmptyBidModifierFromRecord(Record record) {
        return TRAFARET_POSITION_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierTrafaretPosition> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public Map<Long, List<BidModifierTrafaretPositionAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                                      Collection<Long> ids) {
        Result<Record> records = dslContext.select(TRAFARET_POSITION_ADJUSTMENT_FIELDS)
                .from(TRAFARET_POSITION_MULTIPLIER_VALUES)
                .where(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> list.stream().map(TRAFARET_POSITION_ADJUSTMENT_MAPPER::fromDb).collect(toList()))
                .toMap();
    }

    @Override
    public void setAdjustments(BidModifierTrafaretPosition modifier,
                               List<BidModifierTrafaretPositionAdjustment> adjustments) {
        modifier.setTrafaretPositionAdjustments(adjustments);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierTrafaretPosition> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(TRAFARET_POSITION_ADJUSTMENT_FIELDS)
                .from(TRAFARET_POSITION_MULTIPLIER_VALUES)
                .where(TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierTrafaretPosition> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierTrafaretPositionAdjustment> adjustments =
                            recordsList.stream().map(TRAFARET_POSITION_ADJUSTMENT_MAPPER::fromDb).collect(toList());
                    bidModifiersById.get(bidModifierId).withTrafaretPositionAdjustments(adjustments);
                });
    }

    @Override
    protected TrafaretPosition getKey(BidModifierTrafaretPositionAdjustment adjustment) {
        return adjustment.getTrafaretPosition();
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierTrafaretPositionAdjustment> adjustments,
                                     DSLContext txContext) {
        InsertHelper<TrafaretPositionMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, TRAFARET_POSITION_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(TRAFARET_POSITION_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierTrafaretPositionAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<TrafaretPositionMultiplierValuesRecord, BidModifierTrafaretPositionAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID,
                        changes);

        updateBuilder.processProperty(PERCENT, TRAFARET_POSITION_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, TRAFARET_POSITION_MULTIPLIER_VALUES.LAST_CHANGE,
                it -> LocalDateTime.now());

        txContext.update(TRAFARET_POSITION_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID.in(
                        changes.stream().map(it -> it.getModel().getId()).collect(toSet())))
                .execute();
    }

    @Override
    public Class<BidModifierTrafaretPositionAdjustment> getAdjustmentClass() {
        return BidModifierTrafaretPositionAdjustment.class;
    }

    @Override
    public List<BidModifierTrafaretPositionAdjustment> getAdjustments(BidModifierTrafaretPosition modifier) {
        return modifier.getTrafaretPositionAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(TRAFARET_POSITION_MULTIPLIER_VALUES)
                .where(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierTrafaretPositionAdjustment> addedAdjustments,
                                     List<BidModifierTrafaretPositionAdjustment> insertedMultipliers) {
        Map<TrafaretPosition, Long> idsByKey =
                insertedMultipliers.stream().collect(toMap(this::getKey, BidModifierTrafaretPositionAdjustment::getId));
        return addedAdjustments.stream().map(adjustment ->
                idsByKey.get(adjustment.getTrafaretPosition()))
                .collect(toList());
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierTrafaretPosition modifier,
                                              BidModifierTrafaretPositionAdjustment adjustment,
                                              @Nullable Integer oldPercent) {
        return new LogTrafaretPositionMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.trafaret_position_multiplier,
                oldPercent, adjustment.getPercent(), TrafaretPosition.toSource(adjustment.getTrafaretPosition()));
    }

    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierTrafaretPosition> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = TRAFARET_POSITION_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field =
                TRAFARET_POSITION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierTrafaretPositionAdjustment>> changes,
                               List<BidModifierTrafaretPosition> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<TrafaretPositionMultiplierValuesRecord, BidModifierTrafaretPositionAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID,
                        changes);

        updateBuilder.processProperty(BidModifierTrafaretPositionAdjustment.PERCENT,
                TRAFARET_POSITION_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(TRAFARET_POSITION_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(TRAFARET_POSITION_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(TRAFARET_POSITION_MULTIPLIER_VALUES.TRAFARET_POSITION_MULTIPLIER_VALUE_ID.in(
                        changes.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierTrafaretPosition> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getTrafaretPositionAdjustments()
                .forEach(trafaretPositionAdjustment -> trafaretPositionAdjustment.withLastChange(now)));
    }
}
