package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.multivalue;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectForUpdateStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.UpdatableRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.container.bidmodifiers.LogWeatherMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherLiteral;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbschema.ppc.tables.records.WeatherMultiplierValuesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment.PERCENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustmentMultiple.LAST_CHANGE;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.WEATHER_ADJUSTMENT_FIELDS;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.WEATHER_ADJUSTMENT_MAPPER;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.WEATHER_MAPPER;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.tables.WeatherMultiplierValues.WEATHER_MULTIPLIER_VALUES;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class BidModifierWeatherTypeSupport extends AbstractBidModifierMultipleValuesTypeSupport<
        BidModifierWeather, BidModifierWeatherAdjustment, BidModifierWeatherTypeSupport.WeatherKey> {
    @Autowired
    public BidModifierWeatherTypeSupport(ShardHelper shardHelper,
                                         LogBidModifiersService logBidModifiersService) {
        super(shardHelper, logBidModifiersService);
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.WEATHER_MULTIPLIER;
    }

    @Override
    public Class<BidModifierWeather> getBidModifierClass() {
        return BidModifierWeather.class;
    }

    @Override
    public boolean areEqual(BidModifierWeather a, BidModifierWeather b) {
        return Objects.equals(a.getId(), b.getId()) &&
                Objects.equals(a.getCampaignId(), b.getCampaignId()) &&
                Objects.equals(a.getAdGroupId(), b.getAdGroupId()) &&
                Objects.equals(a.getEnabled(), b.getEnabled()) &&
                Objects.equals(new HashSet<>(a.getWeatherAdjustments()),
                        new HashSet<>(b.getWeatherAdjustments()));
    }

    @Override
    public BidModifierWeather createEmptyBidModifierFromRecord(Record record) {
        return WEATHER_MAPPER.fromDb(record);
    }

    @Override
    public List<BidModifierWeather> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public Map<Long, List<BidModifierWeatherAdjustment>> getAdjustmentsByIds(DSLContext dslContext,
                                                                             Collection<Long> ids) {
        Result<Record> records = dslContext.select(WEATHER_ADJUSTMENT_FIELDS)
                .from(WEATHER_MULTIPLIER_VALUES)
                .where(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID.in(ids)).fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        return EntryStream.of(recordsByParentId.asMap())
                .mapValues(list -> mapList(list, WEATHER_ADJUSTMENT_MAPPER::fromDb))
                .toMap();
    }

    @Override
    public void setAdjustments(BidModifierWeather modifier, List<BidModifierWeatherAdjustment> adjustments) {
        modifier.setWeatherAdjustments(adjustments);
    }

    @Override
    public void fillAdjustments(DSLContext dslContext, Collection<BidModifierWeather> bidModifiers,
                                boolean updLock) {
        SelectFinalStep<Record> selectStep = dslContext
                .select(WEATHER_ADJUSTMENT_FIELDS)
                .from(WEATHER_MULTIPLIER_VALUES)
                .where(WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.in(
                        bidModifiers.stream().map(BidModifier::getId).collect(toSet())
                ));
        if (updLock) {
            selectStep = ((SelectForUpdateStep) selectStep).forUpdate();
        }
        Result<Record> records = selectStep.fetch();
        Multimap<Long, Record> recordsByParentId = Multimaps.index(records,
                record -> record.get(WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, Long.class));
        Map<Long, BidModifierWeather> bidModifiersById = Maps.uniqueIndex(bidModifiers, BidModifier::getId);
        EntryStream.of(recordsByParentId.asMap())
                .forKeyValue((bidModifierId, recordsList) -> {
                    List<BidModifierWeatherAdjustment> adjustments =
                            mapList(recordsList, WEATHER_ADJUSTMENT_MAPPER::fromDb);
                    bidModifiersById.get(bidModifierId).withWeatherAdjustments(adjustments);
                });
    }

    @Override
    protected BidModifierWeatherTypeSupport.WeatherKey getKey(BidModifierWeatherAdjustment adjustment) {
        return new BidModifierWeatherTypeSupport.WeatherKey(adjustment.getExpression());
    }

    @Override
    protected void insertAdjustments(Multimap<Long, BidModifierWeatherAdjustment> adjustments,
                                     DSLContext txContext) {
        InsertHelper<WeatherMultiplierValuesRecord>
                insertHelper = new InsertHelper<>(txContext, WEATHER_MULTIPLIER_VALUES);
        adjustments.forEach((modifierId, adjustment) -> {
            insertHelper.add(WEATHER_ADJUSTMENT_MAPPER, adjustment);
            insertHelper.set(WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID, modifierId);
            insertHelper.newRecord();
        });
        insertHelper.executeIfRecordsAdded();
    }

    @Override
    protected void updateAdjustments(Collection<AppliedChanges<BidModifierWeatherAdjustment>> changes,
                                     DSLContext txContext) {
        JooqUpdateBuilder<WeatherMultiplierValuesRecord, BidModifierWeatherAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT, WEATHER_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);
        updateBuilder.processProperty(LAST_CHANGE, WEATHER_MULTIPLIER_VALUES.LAST_CHANGE, it -> LocalDateTime.now());

        txContext.update(WEATHER_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .where(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID.in(
                        listToSet(changes, c -> c.getModel().getId())))
                .execute();
    }

    @Override
    public Class<BidModifierWeatherAdjustment> getAdjustmentClass() {
        return BidModifierWeatherAdjustment.class;
    }

    @Override
    public List<BidModifierWeatherAdjustment> getAdjustments(BidModifierWeather modifier) {
        return modifier.getWeatherAdjustments();
    }

    @Override
    protected void deleteAdjustments(Collection<Long> multiplierIds, DSLContext txContext) {
        txContext.deleteFrom(WEATHER_MULTIPLIER_VALUES)
                .where(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID.in(multiplierIds))
                .execute();
    }

    @Override
    protected List<Long> getAddedIds(List<BidModifierWeatherAdjustment> addedAdjustments,
                                     List<BidModifierWeatherAdjustment> insertedMultipliers) {
        Map<BidModifierWeatherTypeSupport.WeatherKey, Long> idsByKey =
                insertedMultipliers.stream().collect(toMap(this::getKey, BidModifierWeatherAdjustment::getId));
        return mapList(addedAdjustments, adjustment ->
                idsByKey.get(new BidModifierWeatherTypeSupport.WeatherKey(adjustment.getExpression())));
    }

    @Override
    protected LogMultiplierInfo createLogItem(BidModifierWeather modifier,
                                              BidModifierWeatherAdjustment adjustment, @Nullable Integer oldPercent) {
        return new LogWeatherMultiplierInfo(adjustment.getId(), modifier.getId(),
                HierarchicalMultipliersType.weather_multiplier,
                oldPercent, adjustment.getPercent(), JsonUtils.toJson(adjustment.getExpression()));
    }

    static class WeatherKey {
        private final Set<Set<BidModifierWeatherLiteral>> conditionJson;

        public WeatherKey(List<List<BidModifierWeatherLiteral>> conditionJson) {
            this.conditionJson = listToSet(conditionJson, HashSet::new);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            WeatherKey that = (WeatherKey) o;
            return Objects.equals(conditionJson, that.conditionJson);
        }

        @Override
        public int hashCode() {
            return Objects.hash(conditionJson);
        }
    }

    protected Set<Long> getEmptyHierarchicalMultipliersForUpdate(Collection<BidModifierWeather> bidModifiers,
                                                                 DSLContext dslContext) {
        Table table = WEATHER_MULTIPLIER_VALUES;
        TableField<? extends UpdatableRecord, Long> field = WEATHER_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID;
        return dslContext
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .leftJoin(table)
                .on(field.eq(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID))
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        listToSet(bidModifiers, BidModifier::getId)))
                .and(HIERARCHICAL_MULTIPLIERS.TYPE.eq(BidModifierType.toSource(getType())))
                .and(field.isNull())
                .forUpdate()
                .fetchSet(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierWeatherAdjustment>> changes,
                               List<BidModifierWeather> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<WeatherMultiplierValuesRecord, BidModifierWeatherAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID, changes);

        updateBuilder.processProperty(PERCENT,
                WEATHER_MULTIPLIER_VALUES.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(WEATHER_MULTIPLIER_VALUES)
                .set(updateBuilder.getValues())
                .set(WEATHER_MULTIPLIER_VALUES.LAST_CHANGE, LocalDateTime.now())
                .where(WEATHER_MULTIPLIER_VALUES.WEATHER_MULTIPLIER_VALUE_ID.in(
                        listToSet(changes, it -> it.getModel().getId())
                ))
                .execute();

        // Логируем изменения
        logUpdateChanges(operatorUid, changes, bidModifiers);
    }

    @Override
    public void prepareSystemFields(List<BidModifierWeather> bidModifiers) {
        super.prepareSystemFields(bidModifiers);
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> bidModifier.getWeatherAdjustments()
                .forEach(weatherAdjustment -> weatherAdjustment.withLastChange(now)));
    }
}
