package ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.singlevalue;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.log.container.bidmodifiers.LogBidModifierData;
import ru.yandex.direct.common.log.container.bidmodifiers.LogMultiplierInfo;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierSingleValueTypeSupport;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.dbschema.ppc.tables.records.HierarchicalMultipliersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Инкапсулирует в себе логику работы с корректировками, которые не имеют значений в дочерних таблицах.
 */
public abstract class AbstractBidModifierSingleValueTypeSupport<B extends BidModifier,
        TAdjustment extends BidModifierAdjustment> implements BidModifierSingleValueTypeSupport<B, TAdjustment> {
    private final ShardHelper shardHelper;
    protected final LogBidModifiersService logBidModifiersService;

    @Autowired
    public AbstractBidModifierSingleValueTypeSupport(ShardHelper shardHelper,
                                                     LogBidModifiersService logBidModifiersService) {
        this.shardHelper = shardHelper;
        this.logBidModifiersService = logBidModifiersService;
    }

    /**
     * Этот метод знает, как вытащить процент из конкретной корректировки.
     */
    protected abstract int extractPercent(B modifier);

    protected abstract void insertBidModifiers(List<B> bidModifiers, DSLContext dslContext);

    @Override
    public List<B> createEmptyBidModifiersFromRecords(Collection<Record> records) {
        return records.stream().map(this::createEmptyBidModifierFromRecord).collect(toList());
    }

    @Override
    public Map<BidModifierKey, AddedBidModifierInfo> add(DSLContext txContext, List<B> modifiers, long operatorUid) {
        // Генерируем необходимое количество новых ID
        List<Long> ids = BidModifierRepository.generateIds(shardHelper, modifiers.size());

        // Подготавливаем записи к вставке
        List<B> toInsert = StreamEx.zip(modifiers, ids, Pair::of).map(it -> {
            B modifier = it.getLeft();
            Long id = it.getRight();
            modifier.withId(id);
            return modifier;
        }).toList();

        // Вставляем в таблицу
        if (!toInsert.isEmpty()) {
            insertBidModifiers(toInsert, txContext);
        }

        // Логируем изменения
        List<LogBidModifierData> logDataRecords =
                toInsert.stream().map(this::convertModifierToLogItem).collect(toList());
        logBidModifiersService.logBidModifiers(logDataRecords, operatorUid);

        // Возвращаем результат
        return StreamEx.zip(modifiers, ids, Pair::of)
                .toMap(it -> new BidModifierKey(it.getLeft()),
                        it -> AddedBidModifierInfo.added(singletonList(it.getRight())));
    }

    @Override
    public Set<CampaignIdAndAdGroupIdPair> addOrReplace(DSLContext txContext, List<B> modifiers,
                                                        Map<BidModifierKey, BidModifier> lockedExistingModifiers,
                                                        long operatorUid) {
        Map<BidModifierKey, B> modifierByKey = modifiers.stream().collect(toMap(BidModifierKey::new, t -> t));

        Set<BidModifierKey> targetKeys = modifierByKey.keySet();
        Set<BidModifierKey> existingKeys = lockedExistingModifiers.keySet().stream()
                .filter(it -> it.getType().equals(getType())).collect(toSet());

        List<B> toInsert = Sets.difference(targetKeys, existingKeys)
                .stream().map(modifierByKey::get).collect(toList());

        List<B> toUpdate = Sets.intersection(targetKeys, existingKeys)
                .stream()
                .filter(key -> {
                    B oldModifier = (B) lockedExistingModifiers.get(key);
                    B newModifier = modifierByKey.get(key);
                    return extractPercent(oldModifier) != extractPercent(newModifier) ||
                            !Objects.equals(oldModifier.getEnabled(), newModifier.getEnabled());
                })
                .map(modifierByKey::get).collect(toList());

        //noinspection unchecked
        List<B> toDelete = (List<B>) Sets.difference(existingKeys, targetKeys)
                .stream().map(lockedExistingModifiers::get).collect(toList());

        delete(txContext, operatorUid, toDelete);
        add(txContext, toInsert, operatorUid);
        update(txContext, toUpdate, lockedExistingModifiers, operatorUid);

        return Stream.of(toInsert, toUpdate, toDelete)
                .flatMap(Collection::stream)
                .map(b -> new CampaignIdAndAdGroupIdPair()
                        .withCampaignId(b.getCampaignId())
                        .withAdGroupId(b.getAdGroupId()))
                .collect(toSet());
    }

    private void update(DSLContext txContext, List<B> modifiers,
                        Map<BidModifierKey, BidModifier> lockedExistingModifiers, long operatorUid) {
        List<AppliedChanges<TAdjustment>> appliedChangesList = new ArrayList<>();
        List<B> oldModifiers = new ArrayList<>();
        for (B modifier : modifiers) {
            //noinspection unchecked
            B oldModifier = (B) lockedExistingModifiers.get(new BidModifierKey(modifier));
            ModelChanges<TAdjustment> modelChanges = new ModelChanges<>(oldModifier.getId(), getAdjustmentClass());
            modelChanges.process(getAdjustments(modifier).get(0).getPercent(), TAdjustment.PERCENT);
            AppliedChanges<TAdjustment> appliedChanges = modelChanges.applyTo(
                    getAdjustments(oldModifier).get(0));
            appliedChangesList.add(appliedChanges);
            oldModifiers.add(oldModifier);
        }

        updatePercentsCore(operatorUid, appliedChangesList, oldModifiers, txContext);
    }

    private LogBidModifierData convertModifierToLogItem(B modifier) {
        return convertModifierToLogItem(modifier, extractPercent(modifier));
    }

    private LogBidModifierData convertModifierToLogItem(B modifier, @Nullable Integer oldPercent) {
        return new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                .withInserted(
                        singletonList(
                                new LogMultiplierInfo(modifier.getId(), modifier.getId(),
                                        BidModifierType.toSource(getType()),
                                        oldPercent, extractPercent(modifier))));
    }

    @Override
    public void updatePercents(ClientId clientId, long operatorUid, List<AppliedChanges<TAdjustment>> appliedChanges,
                               List<B> bidModifiers, DSLContext dslContext) {
        updatePercentsCore(operatorUid, appliedChanges, bidModifiers, dslContext);
    }

    @Override
    public void prepareSystemFields(List<B> bidModifiers) {
        LocalDateTime now = LocalDateTime.now();
        bidModifiers.forEach(bidModifier -> {
            bidModifier
                    .withEnabled(nvl(bidModifier.getEnabled(), true))
                    .withLastChange(now);
        });
    }

    protected void updatePercentsCore(long operatorUid, List<AppliedChanges<TAdjustment>> appliedChanges,
                                      List<B> bidModifiers, DSLContext dslContext) {
        JooqUpdateBuilder<HierarchicalMultipliersRecord, TAdjustment> updateBuilder =
                new JooqUpdateBuilder<>(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, appliedChanges);

        updateBuilder.processProperty(TAdjustment.PERCENT, HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT, Integer::longValue);

        dslContext.update(HIERARCHICAL_MULTIPLIERS)
                .set(updateBuilder.getValues())
                .set(HIERARCHICAL_MULTIPLIERS.LAST_CHANGE, LocalDateTime.now())
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        appliedChanges.stream()
                                .map(it -> it.getModel().getId())
                                .collect(toSet())
                ))
                .execute();

        logUpdateChanges(operatorUid, appliedChanges, bidModifiers);
    }

    private void logUpdateChanges(long operatorUid, List<AppliedChanges<TAdjustment>> changes, List<B> bidModifiers) {
        Map<Long, Integer> oldPercentById = changes.stream()
                .collect(toMap(t -> t.getModel().getId(), t -> t.getOldValue(TAdjustment.PERCENT)));

        List<LogBidModifierData> logDataItems = bidModifiers.stream().map(modifier -> {
            Long id = modifier.getId();
            Integer oldPercent = oldPercentById.get(id);
            return convertModifierToLogItem(modifier, oldPercent);
        }).collect(toList());

        logBidModifiersService.logBidModifiers(logDataItems, operatorUid);
    }

    @Override
    public void delete(DSLContext txContext, ClientId clientId, long operatorUid, Collection<B> modifiers) {
        delete(txContext, operatorUid, modifiers);
    }

    private void delete(DSLContext txContext, long operatorUid, Collection<B> modifiers) {
        // Удаляем
        txContext.deleteFrom(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(
                        modifiers.stream().map(BidModifier::getId).collect(toSet())
                ))
                .execute();

        // Пишем в лог
        List<LogBidModifierData> logItems = modifiers.stream().map(modifier ->
                        new LogBidModifierData(modifier.getCampaignId(), modifier.getAdGroupId())
                                .withDeletedSet(modifier.getId()))
                .collect(toList());

        logBidModifiersService.logBidModifiers(logItems, operatorUid);
    }
}
