package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupportDispatcher;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.atLeastCampaignIdOrAdGroupIdRequired;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.possibleOnlyOneCampaignIdOrAdGroupIdField;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Component
@ParametersAreNonnullByDefault
public class AddBidModifiersValidationService {
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher;

    @Autowired
    public AddBidModifiersValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                            BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.validationTypeSupportDispatcher = validationTypeSupportDispatcher;
    }

    /**
     * Выполняет валидацию набор корректировок. Далее если планируется добавлять к уже имеющимся в базе
     * наборам, нужно выполнить второй этап валидации, а если планируется заменить все имеющиеся корректировки
     * новыми, то можно приступать к выполнению сохранения в БД.
     *
     * @param modifiers               Наборы корректировок
     * @param allCampaignIds          Все id кампаний, которые были использованы в наборах
     * @param campaignIdsByAdGroupIds Для определения id кампании по id группы
     * @param campaignsType           Для определения типа кампании по id
     * @param operatorUid             uid оператора
     * @param clientId                id клиента
     */
    public ValidationResult<List<BidModifier>, Defect> preValidate(List<BidModifier> modifiers,
                                                                   Set<Long> allCampaignIds,
                                                                   Map<Long, Long> campaignIdsByAdGroupIds,
                                                                   Map<Long, CampaignType> campaignsType,
                                                                   Map<Long, AdGroup> adGroupsWithType,
                                                                   long operatorUid, ClientId clientId,
                                                                   CachingFeaturesProvider featuresProvider) {
        ListValidationBuilder<BidModifier, Defect> builder = ListValidationBuilder.of(modifiers);

        CampaignSubObjectAccessValidator campaignAccessValidator =
                campaignSubObjectAccessCheckerFactory
                        .newCampaignChecker(operatorUid, clientId, allCampaignIds)
                        .createValidator(CampaignAccessType.READ_WRITE);
        CampaignSubObjectAccessValidator adGroupAccessValidator =
                campaignSubObjectAccessCheckerFactory
                        .newAdGroupChecker(operatorUid, clientId, campaignIdsByAdGroupIds.keySet())
                        .createAdGroupValidator(CampaignAccessType.READ_WRITE);

        builder.checkEachBy(it ->
                validateSingleItem(it, campaignIdsByAdGroupIds, campaignsType, adGroupsWithType,
                        campaignAccessValidator, adGroupAccessValidator, clientId, featuresProvider)
        ).checkEach(unique(BidModifierAddItemUniqueKey::new),
                new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_ADJUSTMENT));

        return builder.getResult();
    }

    /**
     * Выполняет совместную валидацию новых+старых наборов корректировок.
     *
     * @param preValidationResult Уже провалидированный новый набор
     * @param existingModifiers   Корректировки, которые уже присутствуют в базе
     * @param campaignTypes       Для определения типа кампании по id
     * @param clientId            id клиента
     */
    void validateAdd(ValidationResult<List<BidModifier>, Defect> preValidationResult,
                     Map<BidModifierKey, BidModifier> existingModifiers,
                     Map<Long, CampaignType> campaignTypes,
                     Map<Long, AdGroup> adGroupsWithType,
                     ClientId clientId,
                     CachingFeaturesProvider featuresProvider) {
        ListValidationBuilder<BidModifier, Defect> lvb = new ListValidationBuilder<>(preValidationResult);

        Map<BidModifierKey, BidModifier> allValidAddingBidModifier = StreamEx.of(getValidItems(preValidationResult))
                .mapToEntry(BidModifierKey::new, Function.identity())
                .toMap();

        List<BidModifierValidationTypeSupport<BidModifier>> validationTypeSupports =
                listToSet(allValidAddingBidModifier.values(), BidModifier::getType).stream()
                        .map(validationTypeSupportDispatcher::getValidationTypeSupportByType)
                        .collect(Collectors.toList());

        validationTypeSupports.forEach(validationTypeSupport ->
                lvb.checkSublistBy(
                        modifiers -> validationTypeSupport.validateAddStep2(
                                clientId, modifiers, existingModifiers, campaignTypes,
                                adGroupsWithType, featuresProvider, allValidAddingBidModifier),
                        When.isValidAnd(When.valueIs(validationTypeSupport::supports)))
        );
    }

    private ValidationResult<BidModifier, Defect> validateSingleItem(BidModifier item,
                                                                     Map<Long, Long> campaignIdsByAdGroupIds,
                                                                     Map<Long, CampaignType> campaignsType,
                                                                     Map<Long, AdGroup> adGroupsWithType,
                                                                     CampaignSubObjectAccessValidator campaignAccessValidator,
                                                                     CampaignSubObjectAccessValidator adGroupAccessValidator,
                                                                     ClientId clientId,
                                                                     CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifier> ivb = ModelItemValidationBuilder.of(item);

        // Основные общие проверки
        ivb.check(atLeastCampaignIdOrAdGroupIdRequired());
        ivb.check(possibleOnlyOneCampaignIdOrAdGroupIdField(), When.isValid());

        ivb.item(BidModifier.CAMPAIGN_ID)
                .check(validId(), When.notNull())
                .checkBy(campaignAccessValidator, When.isValidAnd(When.notNull()));
        ivb.item(BidModifier.AD_GROUP_ID)
                .check(validId(), When.notNull())
                .checkBy(adGroupAccessValidator, When.isValidAnd(When.notNull()));

        // Типоспецифичные проверки
        ivb.checkBy(it -> {
            long campaignId = Optional.ofNullable(it.getCampaignId())
                    .orElseGet(() -> campaignIdsByAdGroupIds.get(it.getAdGroupId()));
            return validationTypeSupportDispatcher.validateAddStep1(it,
                    campaignsType.get(campaignId),
                    it.getAdGroupId() != null ? adGroupsWithType.get(it.getAdGroupId()) : null,
                    clientId,
                    featuresProvider);
        }, When.isValid());

        return ivb.getResult();
    }

    /**
     * Для проверки на уникальность наборов корректировок.
     */
    public class BidModifierAddItemUniqueKey {
        private final Long campaignId;
        private final Long adGroupId;
        private final BidModifierType type;

        BidModifierAddItemUniqueKey(BidModifier item) {
            this.campaignId = item.getCampaignId();
            this.adGroupId = item.getAdGroupId();
            this.type = validationTypeSupportDispatcher.getTypeOf(item);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            BidModifierAddItemUniqueKey that = (BidModifierAddItemUniqueKey) o;
            return Objects.equals(campaignId, that.campaignId) &&
                    Objects.equals(adGroupId, that.adGroupId) &&
                    type == that.type;
        }

        @Override
        public int hashCode() {
            return Objects.hash(campaignId, adGroupId, type);
        }
    }
}
