package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.container.AddedBidModifierInfo;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignAccessCheckRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService.getExternalId;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds.GeneralDefects.CONFLICT_APPLYING_CHANGES;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Операция добавления корректировок (к уже имеющимся наборам).
 * Предназначена для использования из API, поэтому partial=true.
 */
public class BidModifierAddOperation extends AbstractAddOperation<BidModifier, List<Long>> {
    private final CampaignAccessibiltyChecker<? extends CampaignForAccessCheck> campaignAccessibiltyChecker;
    private final CampaignRepository campaignRepository;
    private final CampaignAccessCheckRepository campaignAccessCheckRepository;
    private final BidModifierRepository bidModifierRepository;
    private final BidModifierService bidModifierService;
    private final AdGroupRepository adGroupRepository;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;
    private final AddBidModifiersValidationService validationService;
    private final DslContextProvider dslContextProvider;

    private final ClientId clientId;
    private final long operatorUid;
    private final int shard;

    // Состояние, необходимое на разных этапах операции
    private Map<Long, Long> campaignIdsByAdGroupIds;
    private Map<Long, CampaignType> campaignsType;
    private Map<Long, AdGroup> adGroupsWithType;
    private Map<BidModifierKey, BidModifier> existingModifiers;
    private CachingFeaturesProvider featuresProvider;

    public BidModifierAddOperation(long operatorUid, ClientId clientId, int shard,
                                   CampaignRepository campaignRepository,
                                   CampaignAccessCheckRepository campaignAccessCheckRepository,
                                   AdGroupRepository adGroupRepository,
                                   BidModifierRepository bidModifierRepository,
                                   BidModifierService bidModifierService,
                                   BidModifierTypeSupportDispatcher typeSupportDispatcher,
                                   AddBidModifiersValidationService validationService,
                                   DslContextProvider dslContextProvider,
                                   FeatureService featureService,
                                   CampaignAccessibiltyChecker<? extends CampaignForAccessCheck>
                                           campaignAccessibiltyChecker,
                                   List<BidModifier> models) {
        super(Applicability.PARTIAL, models);

        checkState(models.stream().allMatch(it -> it.getType() != null));

        this.campaignAccessibiltyChecker = campaignAccessibiltyChecker;
        this.campaignRepository = campaignRepository;
        this.campaignAccessCheckRepository = campaignAccessCheckRepository;
        this.bidModifierRepository = bidModifierRepository;
        this.adGroupRepository = adGroupRepository;
        this.bidModifierService = bidModifierService;
        this.typeSupportDispatcher = typeSupportDispatcher;
        this.validationService = validationService;
        this.dslContextProvider = dslContextProvider;

        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;

        // Кешируем фичи клиента во время выполнения операции
        this.featuresProvider = new CachingFeaturesProvider(featureService);
    }

    @Override
    protected ValidationResult<List<BidModifier>, Defect> preValidate(List<BidModifier> modifiers) {
        // Собираем вспомогательные данные
        Set<Long> campaignIds =
                modifiers.stream().map(BidModifier::getCampaignId).filter(Objects::nonNull).collect(toSet());
        Set<Long> adGroupIds =
                modifiers.stream().map(BidModifier::getAdGroupId).filter(Objects::nonNull).collect(toSet());

        campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, clientId, adGroupIds);
        Set<Long> allCampaignIds = Sets.union(campaignIds, new HashSet<>(campaignIdsByAdGroupIds.values()));

        campaignsType = EntryStream.of(campaignAccessCheckRepository.getCampaignsForAccessCheckByCampaignIds(
                shard, campaignAccessibiltyChecker.toAllowableCampaignsRepositoryAdapter(clientId), allCampaignIds))
                .mapValues(CampaignForAccessCheck::getType)
                .toMap();

        adGroupsWithType = StreamEx.of(adGroupRepository.getAdGroups(shard, adGroupIds))
                .toMap(AdGroup::getId, identity());
        // Выполняем первичные проверки над наборами от пользователя
        return validationService.preValidate(modifiers, allCampaignIds, campaignIdsByAdGroupIds,
                campaignsType, adGroupsWithType, operatorUid, clientId, featuresProvider);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<BidModifier> modelsPreValidatedStep) {
        // Проставим всем кому сможем campaignId для упрощения дальнейшего кода
        modelsPreValidatedStep.getModels().forEach(it -> {
            if (it.getCampaignId() == null) {
                it.withCampaignId(campaignIdsByAdGroupIds.get(it.getAdGroupId()));
            }
        });
    }

    @Override
    protected void validate(ValidationResult<List<BidModifier>, Defect> preValidationResult) {
        // Загружаем имеющиеся корректировки из базы (берём только валидные элементы)
        // Список validModifiers не будет пустым, иначе preValidateInternal() уже вернул бы brokenMassAction().
        List<BidModifier> validModifiers = getValidItems(preValidationResult);

        checkState(validModifiers.stream().allMatch(b -> b.getCampaignId() != null),
                "Всем валидным модификаторам уже должен быть проставлен campaignId на стадии onPreValidated");

        // Сохраняем загруженные наборы для дальнейшего использования в execute()
        existingModifiers = bidModifierRepository.getBidModifiersByKeys(
                shard, listToSet(validModifiers, BidModifierKey::new));

        // Проверяем вместе старые+новые (кроме уже невалидных и тех, которым в базе ничего не соответствует)
        validationService.validateAdd(
                preValidationResult, existingModifiers, campaignsType, adGroupsWithType, clientId, featuresProvider);
    }

    @Override
    protected void beforeExecution(Map<Integer, BidModifier> validModelsMapToApply) {
        typeSupportDispatcher.prepareSystemFields(validModelsMapToApply.values());
    }

    @Override
    protected Map<Integer, List<Long>> execute(Map<Integer, BidModifier> validModelsMapToApply) {
        Collection<BidModifier> bidModifiers = validModelsMapToApply.values();

        checkState(bidModifiers.stream().allMatch(b -> b.getCampaignId() != null),
                "Всем валидным модификаторам уже должен быть проставлен campaignId на стадии onPreValidated");

        // Добавляем валидные наборы корректировок
        Map<BidModifierKey, AddedBidModifierInfo> addedInfoMap = addModifiers(shard, bidModifiers, existingModifiers,
                clientId, operatorUid);

        // Для успешно добавленных возвращается список добавленных id, а для тех, кто не был добавлен из-за
        // того, что в базе данные успели поменяться - пустой список
        return EntryStream.of(validModelsMapToApply)
                .mapValues(bidModifier -> {
                    BidModifierKey key = typeSupportDispatcher.makeKey(bidModifier, campaignIdsByAdGroupIds);
                    AddedBidModifierInfo addedInfo = addedInfoMap.get(key);
                    if (!addedInfo.isAdded()) {
                        return Collections.<Long>emptyList();
                    }
                    return addedInfo.getIds().stream()
                            .map(id -> getExternalId(id, bidModifier.getType()))
                            .collect(toList());
                }).toMap();
    }

    /**
     * Выполняет добавление указанных наборов корректировок к существующим (по сути merge новых и существующих).
     * Побочные эффекты: в modifiers могут быть установлены campaignId там, где их не было.
     */
    private Map<BidModifierKey, AddedBidModifierInfo> addModifiers(int shard, Collection<BidModifier> modifiers,
                                                                   Map<BidModifierKey, BidModifier> existingModifiers,
                                                                   ClientId clientId, long operatorUid) {
        // campaignId и type должны быть проставлены вызывающим кодом
        checkState(modifiers.stream().allMatch(it -> it.getCampaignId() != null && it.getType() != null));

        return dslContextProvider.ppc(shard).transactionResult(configuration -> {
            DSLContext txContext = configuration.dsl();

            // Добавляем к существующим наборам (или создаём новые)
            Map<BidModifierKey, AddedBidModifierInfo> addedInfoMap =
                    bidModifierRepository.addModifiers(txContext, modifiers, existingModifiers, clientId, operatorUid);

            // Наборы, которые действительно были сохранены
            List<BidModifierKey> affectedModifiers = EntryStream.of(addedInfoMap)
                    .filterValues(AddedBidModifierInfo::isAdded)
                    .keys()
                    .collect(toList());

            // Переотправляем в БК изменённые наборы
            bidModifierService.setBsSyncedForChangedCampaignsAndAdGroups(txContext, affectedModifiers);

            return addedInfoMap;
        });
    }

    /**
     * Метод переопределён, чтобы добавлять дефект к объектам, которые не получилось добавить
     * из-за того, что данные в базе обновились между prepare() и apply().
     *
     * @param resultMap              Map добавленных объектов
     * @param validationResult       Объединенный результат валидации объектов
     * @param canceledElementIndexes Индексы объектов для которых операция отменилась
     */
    @Override
    protected MassResult<List<Long>> createMassResult(Map<Integer, List<Long>> resultMap,
                                                      ValidationResult<List<BidModifier>, Defect> validationResult,
                                                      Set<Integer> canceledElementIndexes) {
        resultMap.entrySet().stream().filter(e -> e.getValue().isEmpty())
                .forEach(e -> {
                    Integer index = e.getKey();
                    validationResult.getOrCreateSubValidationResult(PathHelper.index(index), getModels().get(index))
                            .addError(new Defect<>(CONFLICT_APPLYING_CHANGES));
                });
        return super.createMassResult(resultMap, validationResult, canceledElementIndexes);
    }
}
