package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService.getRealIdsGroupedByType;
import static ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService.getRealType;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNoRights;
import static ru.yandex.direct.core.validation.defects.Defects.badStatusCampaignArchived;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Операция удаления корректировок.
 */
public class BidModifierDeleteOperation extends AbstractOperationWithId {

    private final CampaignRepository campaignRepository;
    private final BidModifierService bidModifierService;
    private final RbacService rbacService;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;

    private final long operatorUid;
    private final ClientId clientId;
    private final int shard;

    // Состояние, необходимое на обоих этапах операции
    private List<BidModifier> bidModifiers;

    BidModifierDeleteOperation(int shard, ClientId clientId, long operatorUid, CampaignRepository campaignRepository,
                               BidModifierService bidModifierService, RbacService rbacService,
                               BidModifierTypeSupportDispatcher typeSupportDispatcher,
                               List<Long> modelIds) {
        super(Applicability.PARTIAL, modelIds);
        this.campaignRepository = campaignRepository;
        this.bidModifierService = bidModifierService;
        this.rbacService = rbacService;
        this.typeSupportDispatcher = typeSupportDispatcher;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.shard = shard;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {

        // Предварительная валидация на положительные идентификаторы, префиксы и отсутствие дублей
        ListValidationBuilder<Long, Defect> listBuilder = ListValidationBuilder.of(ids);
        listBuilder
                .checkEach(validId())
                .checkEach(validExternalId(), When.isValid())
                .checkEach(unique(), new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_ADJUSTMENT));

        ValidationResult<List<Long>, Defect> vr = listBuilder.getResult();
        List<Long> validIds = getValidItems(vr);

        if (validIds.isEmpty()) {
            return vr;
        }

        // Получаем существующие корректировки по переданным идентификаторам
        Multimap<BidModifierType, Long> idsByType = getRealIdsGroupedByType(validIds);
        List<BidModifier> bidModifiers = bidModifierService.getByIds(
                clientId, idsByType, operatorUid);

        // По ненайденным возвращаем ошибку "Объект не найден"
        Set<Long> notFoundIds = Sets.difference(new HashSet<>(validIds),
                bidModifierService.getExternalIdsFlattened(bidModifiers));
        listBuilder
                .checkEach(notInSet(notFoundIds), new Defect<>(DefectIds.OBJECT_NOT_FOUND), When.isValid());

        // Получаем архивные кампании
        // По корректировкам из архивных компаний возвращаем ошибку "Запрещено изменять заархивированную кампанию"
        Set<Long> campaignIds = bidModifiers.stream().map(BidModifier::getCampaignId).collect(toSet());
        Set<Long> archivedCampaignIds = campaignRepository.getArchivedCampaigns(shard, campaignIds);
        Set<Long> idsOfArchivedModifiers = bidModifierService.getExternalIdsFlattened(
                bidModifiers.stream().filter(it -> archivedCampaignIds.contains(it.getCampaignId())).collect(toList()));
        listBuilder.checkEach(notInSet(idsOfArchivedModifiers), badStatusCampaignArchived(), When.isValid());

        // Получаем корректировки, которые относятся к кампаниям без права на запись
        // По этим корректировкам возвращаем ошибку "Нет прав на запись"
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, campaignIds);
        Set<Long> idsOfNonWriteableModifiers = bidModifierService.getExternalIdsFlattened(
                bidModifiers.stream().filter(it -> !writableCampaignIds.contains(it.getCampaignId()))
                        .collect(toList()));
        listBuilder.checkEach(notInSet(idsOfNonWriteableModifiers), campaignNoRights(), When.isValid());

        // Сохраняем все подходящие наборы
        this.bidModifiers = bidModifiers.stream()
                .filter(modifier -> !archivedCampaignIds.contains(modifier.getCampaignId()))
                .filter(modifier -> writableCampaignIds.contains(modifier.getCampaignId()))
                .collect(toList());

        Preconditions.checkState(
                bidModifierService.getExternalIdsFlattened(this.bidModifiers).size() == getValidItems(vr).size());

        return vr;
    }

    @Override
    protected void execute(List<Long> ids) {
        // Удаляем найденные корректировки в БД
        bidModifierService.deleteAdjustments(shard, clientId, operatorUid, bidModifiers);
    }

    public static Constraint<Long, Defect> validExternalId() {
        return fromPredicate(id -> getRealType(id) != null, new Defect<>(DefectIds.OBJECT_NOT_FOUND));
    }
}
