package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.feature.FeatureName;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.CONTENT_DURATION_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.DEMOGRAPHICS_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.INVENTORY_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.PRISMA_INCOME_GRADE_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.RETARGETING_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.RETARGETING_FILTER_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.TRAFARET_POSITION_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.WEATHER_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.isCpmCampaignCondition;

public class BidModifierLimitsAdvanced {
    public static final int DEFAULT_MAX = 1300;
    private static final int DEFAULT_MIN = 0;
    //Пока только по фиче на мобиильные ОС, потом расширится на все десктопы, см. DIRECT-113159
    private static final int MAX_DESKTOP_ON_FEATURE = 400;
    private static final int NO_LIMIT = -1;

    private static final BidModifierLimits BID_MODIFIER_LIMITS_FORBIDDEN = new BidModifierLimits.Builder()
            .withPercentMin(DEFAULT_MIN)
            .withPercentMax(DEFAULT_MAX)
            .withEnabled(false)
            .withMaxConditions(0)
            .build();

    public interface BidModifierLimitCalculator {
        BidModifierLimits calculate(
                CampaignType campaignType,
                @Nullable AdGroup adGroupWithType,
                ClientId clientId,
                FeaturesProvider featuresProvider
        );
    }

    public BidModifierLimitCalculator constant(BidModifierLimits limits) {
        return (campaignType, adGroupWithType, clientId, featuresProvider) -> limits;
    }

    private static boolean adGroupSupportWeatherMultiplierCondition(AdGroupType adGroupType) {
        return (adGroupType == AdGroupType.CPM_BANNER
                || adGroupType == AdGroupType.CPM_VIDEO
                || adGroupType == AdGroupType.CPM_OUTDOOR
                || adGroupType == AdGroupType.BASE
                || adGroupType == AdGroupType.PERFORMANCE
                || adGroupType == AdGroupType.CPM_AUDIO
        );
    }

    public static boolean isCpmBannerAdGroupWithKeywords(AdGroup adGroup) {
        return adGroup instanceof CpmBannerAdGroup
                && ((CpmBannerAdGroup) adGroup).getCriterionType() == CriterionType.KEYWORD;
    }

    private static final Map<BidModifierType, BidModifierLimitCalculator> ALL_CALCULATORS =
            ImmutableMap.<BidModifierType,
                    BidModifierLimitCalculator>builder().put(BidModifierType.AB_SEGMENT_MULTIPLIER,
                    (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                        // Корректировка разрешена только на кампаниях
                        boolean enabled = adGroupWithType == null && campaignType != CampaignType.CPM_YNDX_FRONTPAGE;
                        return new BidModifierLimits.Builder()
                                .withPercentMin(DEFAULT_MIN)
                                .withPercentMax(DEFAULT_MAX)
                                .withEnabled(enabled)
                                .withMaxConditions(100)
                                .build();
                    })
                    .put(BidModifierType.BANNER_TYPE_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                // Корректировка разрешена только на кампаниях
                                // и только на CPM кампаниях
                                boolean enabled = adGroupWithType == null && isCpmCampaignCondition(campaignType);
                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.DEMOGRAPHY_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = false;
                                if (adGroupWithType != null) {
                                    AdGroupType adGroupType = adGroupWithType.getType();
                                    if (adGroupType == AdGroupType.BASE
                                            || adGroupType == AdGroupType.DYNAMIC
                                            || adGroupType == AdGroupType.MOBILE_CONTENT
                                            || adGroupType == AdGroupType.PERFORMANCE
                                            || adGroupType == AdGroupType.MCBANNER
                                            || isCpmBannerAdGroupWithKeywords(adGroupWithType)
                                            || adGroupType == AdGroupType.CPM_INDOOR
                                            || adGroupType == AdGroupType.CONTENT_PROMOTION
                                            || adGroupType == AdGroupType.CONTENT_PROMOTION_VIDEO
                                    ) {
                                        enabled = true;
                                    }
                                } else {
                                    if (campaignType == CampaignType.TEXT
                                            || campaignType == CampaignType.MOBILE_CONTENT
                                            || campaignType == CampaignType.PERFORMANCE
                                            || campaignType == CampaignType.DYNAMIC
                                            || campaignType == CampaignType.MCBANNER
                                            || campaignType == CampaignType.CONTENT_PROMOTION) {
                                        enabled = true;
                                    }
                                }

                                // По фиче DEMOGRAPHY_BID_MODIFIER_UNKNOWN_AGE_ALLOWED добавляется два варианта
                                // корректировок.
                                // Фронту важно знать какие это ограничения для дизейбления кнопки.
                                // Отрывать можно только при совместной работе с фронтом
                                int maxConditions = (featuresProvider.isEnabledForClientId(clientId,
                                        FeatureName.DEMOGRAPHY_BID_MODIFIER_UNKNOWN_AGE_ALLOWED))
                                        ? DEMOGRAPHICS_ADJUSTMENTS_LIMIT + 2
                                        : DEMOGRAPHICS_ADJUSTMENTS_LIMIT;

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withMaxConditions(maxConditions)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.DESKTOP_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> getDesktopLimits(
                                    campaignType, adGroupWithType, clientId, featuresProvider
                            ).build())
                    .put(BidModifierType.TABLET_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                var builder = getDesktopLimits(
                                        campaignType, adGroupWithType, clientId, featuresProvider
                                );

                                builder.withEnabled(
                                    builder.getEnabled() &&
                                    featuresProvider.isEnabledForClientId(
                                        clientId, FeatureName.TABLET_BIDMODIFER_ENABLED
                                    ) && !isCpmCampaignCondition(campaignType)
                                );

                                return builder.build();
                            })
                    .put(BidModifierType.DESKTOP_ONLY_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                var builder = getDesktopLimits(
                                        campaignType, adGroupWithType, clientId, featuresProvider
                                );

                                builder.withEnabled(
                                        builder.getEnabled() &&
                                        featuresProvider.isEnabledForClientId(
                                                clientId, FeatureName.TABLET_BIDMODIFER_ENABLED
                                        ) && !isCpmCampaignCondition(campaignType)
                                );

                                return builder.build();
                            })
                    .put(BidModifierType.SMARTTV_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = false;

                                if (adGroupWithType != null) {
                                    AdGroupType adGroupType = adGroupWithType.getType();
                                    enabled = adGroupType == AdGroupType.CPM_BANNER
                                            || adGroupType == AdGroupType.CPM_AUDIO
                                            || adGroupType == AdGroupType.CPM_VIDEO;
                                }

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.GEO_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled;
                                if (adGroupWithType != null || campaignType == CampaignType.CPM_YNDX_FRONTPAGE) {
                                    enabled = false;
                                } else {
                                    enabled = !isCpmCampaignCondition(campaignType);
                                }
                                return new BidModifierLimits.Builder()
                                        .withPercentMin(10)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withMaxConditions(NO_LIMIT)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.INVENTORY_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = (campaignType == CampaignType.CPM_BANNER ||
                                        campaignType == CampaignType.CPM_PRICE) && adGroupWithType == null;

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withMaxConditions(INVENTORY_ADJUSTMENTS_LIMIT)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.MOBILE_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled;
                                int percentMax = DEFAULT_MAX;
                                int percentMin = 0;

                                if (campaignType == CampaignType.MOBILE_CONTENT || campaignType == CampaignType.MCBANNER) {
                                    enabled = false;
                                } else if (campaignType == CampaignType.CPM_PRICE) {
                                    enabled = true;
                                    percentMax = 0;
                                } else {
                                    boolean mobileOsModifiersIsEnabled =
                                            BidModifierValidationHelper.TYPES_FOR_MOBILE_OS_BID_MODIFIER.contains(campaignType);

                                    if (adGroupWithType != null) {
                                        AdGroupType adGroupType = adGroupWithType.getType();
                                        enabled = adGroupType == AdGroupType.BASE
                                                || adGroupType == AdGroupType.DYNAMIC
                                                || adGroupType == AdGroupType.PERFORMANCE
                                                || adGroupType == AdGroupType.CPM_BANNER
                                                || adGroupType == AdGroupType.CPM_GEOPRODUCT
                                                || adGroupType == AdGroupType.CPM_GEO_PIN
                                                || adGroupType == AdGroupType.CONTENT_PROMOTION
                                                || adGroupType == AdGroupType.CONTENT_PROMOTION_VIDEO
                                                || adGroupType == AdGroupType.CPM_AUDIO
                                                || adGroupType == AdGroupType.CPM_VIDEO;
                                    } else {
                                        enabled = campaignType == CampaignType.TEXT
                                                || campaignType == CampaignType.PERFORMANCE
                                                || campaignType == CampaignType.DYNAMIC
                                                || campaignType == CampaignType.CONTENT_PROMOTION;
                                    }

                                    enabled = enabled || mobileOsModifiersIsEnabled;
                                }

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(percentMin)
                                        .withPercentMax(percentMax)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.PERFORMANCE_TGO_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = campaignType == CampaignType.PERFORMANCE;
                                return new BidModifierLimits.Builder()
                                        .withPercentMin(20)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.RETARGETING_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = false;
                                if (adGroupWithType != null) {
                                    AdGroupType adGroupType = adGroupWithType.getType();
                                    if (adGroupType == AdGroupType.BASE
                                            || adGroupType == AdGroupType.DYNAMIC
                                            || adGroupType == AdGroupType.MOBILE_CONTENT
                                            || adGroupType == AdGroupType.PERFORMANCE
                                            || adGroupType == AdGroupType.MCBANNER
                                            || adGroupType == AdGroupType.CONTENT_PROMOTION
                                            || adGroupType == AdGroupType.CONTENT_PROMOTION_VIDEO
                                            || isCpmBannerAdGroupWithKeywords(adGroupWithType)
                                    ) {
                                        enabled = true;
                                    }
                                } else {
                                    if (campaignType == CampaignType.TEXT
                                            || campaignType == CampaignType.MOBILE_CONTENT
                                            || campaignType == CampaignType.PERFORMANCE
                                            || campaignType == CampaignType.DYNAMIC
                                            || campaignType == CampaignType.MCBANNER
                                            || campaignType == CampaignType.CONTENT_PROMOTION) {
                                        enabled = true;
                                    }
                                }

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .withMaxConditions(RETARGETING_ADJUSTMENTS_LIMIT)
                                        .build();
                            })
                    .put(BidModifierType.RETARGETING_FILTER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) ->
                                new BidModifierLimits.Builder()
                                        .withPercentMin(0)
                                        .withPercentMax(0)
                                        .withEnabled(true)
                                        .withMaxConditions(RETARGETING_FILTER_LIMIT)
                                        .build()
                    )
                    .put(BidModifierType.EXPRESS_TRAFFIC_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                // Корректировки по пробкам разрешены только на группе CPM_OUTDOOR
                                boolean enabled = (campaignType == CampaignType.CPM_BANNER)
                                        && adGroupWithType != null
                                        && adGroupWithType.getType() == AdGroupType.CPM_OUTDOOR;
                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .withMaxConditions(20)
                                        .build();
                            })
                    .put(BidModifierType.VIDEO_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = campaignType == CampaignType.TEXT && (
                                        adGroupWithType == null || adGroupWithType.getType() == AdGroupType.BASE
                                );
                                return new BidModifierLimits.Builder()
                                        .withPercentMin(50)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.WEATHER_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled;
                                if (adGroupWithType == null) {
                                    enabled = false;
                                } else {
                                    enabled = (campaignType == CampaignType.PERFORMANCE
                                            || campaignType == CampaignType.CPM_BANNER
                                            || campaignType == CampaignType.CPM_DEALS
                                            || campaignType == CampaignType.TEXT)
                                            && adGroupSupportWeatherMultiplierCondition(adGroupWithType.getType());
                                }

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .withMaxConditions(WEATHER_ADJUSTMENTS_LIMIT)
                                        .build();
                            })
                    .put(BidModifierType.EXPRESS_CONTENT_DURATION_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = campaignType == CampaignType.CPM_BANNER
                                        && adGroupWithType != null
                                        && adGroupWithType.getType() == AdGroupType.CPM_VIDEO
                                        && featuresProvider.isEnabledForClientId(clientId,
                                        FeatureName.CONTENT_DURATION_BID_MODIFIER);

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(DEFAULT_MIN)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withEnabled(enabled)
                                        .withMaxConditions(CONTENT_DURATION_ADJUSTMENTS_LIMIT)
                                        .build();
                            })
                    .put(BidModifierType.TRAFARET_POSITION_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                boolean enabled = (campaignType == CampaignType.TEXT ||
                                        campaignType == CampaignType.MOBILE_CONTENT ||
                                        campaignType == CampaignType.DYNAMIC) && adGroupWithType == null;

                                return new BidModifierLimits.Builder()
                                        .withPercentMin(100)
                                        .withPercentMax(DEFAULT_MAX)
                                        .withMaxConditions(TRAFARET_POSITION_ADJUSTMENTS_LIMIT)
                                        .withEnabled(enabled)
                                        .build();
                            })
                    .put(BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER,
                            (campaignType, adGroupWithType, clientId, featuresProvider) -> {
                                var allowedCampaignTypes = Set.of(
                                        CampaignType.TEXT,
                                        CampaignType.DYNAMIC,
                                        CampaignType.PERFORMANCE,
                                        CampaignType.MOBILE_CONTENT
                                );
                                var allowedAdGroupTypes = Set.of(
                                        AdGroupType.BASE,
                                        AdGroupType.DYNAMIC,
                                        AdGroupType.PERFORMANCE,
                                        AdGroupType.MOBILE_CONTENT
                                );
                                var enabled = campaignType != null &&
                                        allowedCampaignTypes.contains(campaignType) &&
                                        (adGroupWithType == null || allowedAdGroupTypes.contains(adGroupWithType.getType())) &&
                                        featuresProvider.isEnabledForClientId(clientId,
                                                FeatureName.INCOME_GRADE_BID_MODIFIER_ALLOWED);

                                return new BidModifierLimits.Builder()
                                        .withPercentMax(DEFAULT_MAX)
                                        .withPercentMin(DEFAULT_MIN)
                                        .withMaxConditions(PRISMA_INCOME_GRADE_ADJUSTMENTS_LIMIT)
                                        .withEnabled(enabled)
                                        .build();
                            }
                    )
                    .build();

    private static BidModifierLimits.Builder getDesktopLimits(
            CampaignType campaignType,
            @Nullable AdGroup adGroupWithType,
            ClientId clientId,
            FeaturesProvider featuresProvider
    ) {
        if (campaignType == CampaignType.CPM_PRICE) {
            return new BidModifierLimits.Builder()
                    .withPercentMin(DEFAULT_MIN)
                    .withPercentMax(0)
                    .withEnabled(true);
        }

        boolean mobileOsModifiersIsEnabled =
                BidModifierValidationHelper.TYPES_FOR_MOBILE_OS_BID_MODIFIER.contains(campaignType);

        boolean enabled = false;


        if (adGroupWithType != null) {
            AdGroupType adGroupType = adGroupWithType.getType();
            enabled = adGroupType == AdGroupType.BASE
                    || adGroupType == AdGroupType.CPM_BANNER
                    || adGroupType == AdGroupType.PERFORMANCE
                    || adGroupType == AdGroupType.CPM_GEOPRODUCT
                    || adGroupType == AdGroupType.CONTENT_PROMOTION
                    || adGroupType == AdGroupType.CONTENT_PROMOTION_VIDEO
                    || adGroupType == AdGroupType.CPM_AUDIO
                    || adGroupType == AdGroupType.CPM_VIDEO;
        } else {
            enabled = campaignType == CampaignType.TEXT
                    || campaignType == CampaignType.CONTENT_PROMOTION
                    || campaignType == CampaignType.PERFORMANCE
                    || campaignType == CampaignType.DYNAMIC
                    || campaignType == CampaignType.MCBANNER;
        }

        enabled = enabled || mobileOsModifiersIsEnabled;

        return new BidModifierLimits.Builder()
                .withPercentMin(DEFAULT_MIN)
                .withPercentMax(DEFAULT_MAX)
                .withEnabled(enabled);
    }

    // Возможность переопределить поведение (только для тестов и только до тех пор, пока нет
    // ни одной expression-корректировки, которая может быть установлена на ТГО-группы)
    // TODO : удалить эти оверрайды после появления такой корректировки
    private static final Map<BidModifierType, BidModifierLimitCalculator> TEST_OVERRIDES = new HashMap<>();

    public static BidModifierLimits getLimits(BidModifierType bidModifierType,
                                              CampaignType campaignType,
                                              @Nullable AdGroup adGroupWithType,
                                              ClientId clientId,
                                              FeaturesProvider featuresProvider) {
        // Когда переделаем adGroupWithType в структуру с полями AdGroupType + CriterionType,
        // проверку можно будет убрать
        if (adGroupWithType != null) {
            checkNotNull(adGroupWithType.getType());
            if (adGroupWithType instanceof CpmBannerAdGroup) {
                checkNotNull(((CpmBannerAdGroup) adGroupWithType).getCriterionType());
            }
        }

        BidModifierLimitCalculator calculator = ALL_CALCULATORS.get(bidModifierType);
        // Позволяем в тестах переопределить поведение (временно, потом можно будет избавиться от этого)
        if (Environment.getCached().isSandbox() || Environment.getCached().isDevelopment()) {
            if (TEST_OVERRIDES.containsKey(bidModifierType)) {
                calculator = TEST_OVERRIDES.get(bidModifierType);
            }
        }
        // пока неизвестный тип корректировки, не разрешаем
        if (calculator == null) {
            return BID_MODIFIER_LIMITS_FORBIDDEN;
        }
        return calculator.calculate(campaignType, adGroupWithType, clientId, featuresProvider);
    }

    public static void addTestOverride(BidModifierType bidModifierType, BidModifierLimitCalculator calculator) {
        checkState(Environment.getCached().isSandbox() || Environment.getCached().isDevelopment());
        TEST_OVERRIDES.put(bidModifierType, calculator);
    }
}
