package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.bidmodifiers.IsEnabledChange;
import ru.yandex.direct.common.log.container.bidmodifiers.LogBidModifierData;
import ru.yandex.direct.common.log.service.LogBidModifiersService;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.container.MultipliersBounds;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.container.UntypedBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierMultipleValuesTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupportDispatcher;
import ru.yandex.direct.core.entity.campaign.repository.CampaignAccessCheckRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessibiltyChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.RequestCampaignAccessibilityCheckerProvider;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.AB_SEGMENT_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.BANNER_TYPE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DEMOGRAPHY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DESKTOP_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DESKTOP_ONLY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.EXPRESS_CONTENT_DURATION_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.EXPRESS_TRAFFIC_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.GEO_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.INVENTORY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.MOBILE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.PERFORMANCE_TGO_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.RETARGETING_FILTER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.RETARGETING_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.SMARTTV_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.TABLET_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.TRAFARET_POSITION_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.VIDEO_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.WEATHER_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_LEVELS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_TYPES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_TYPES_INTERNAL;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel.ADGROUP;
import static ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel.CAMPAIGN;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class BidModifierService implements EntityService<BidModifier, Long> {
    static final Integer ONE_HUNDRED = 100;
    public static final Set<BidModifierType> ALL_BID_MODIFIER_TYPES = EnumSet.allOf(BidModifierType.class);
    private final BidModifierRepository bidModifierRepository;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final CampaignRepository campaignRepository;
    private final CampaignAccessCheckRepository campaignAccessCheckRepository;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;
    private final BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher;
    private final AddBidModifiersValidationService addBidModifiersValidationService;
    private final ToggleBidModifiersValidationService toggleBidModifiersValidationService;
    private final LogBidModifiersService logBidModifiersService;
    private final DslContextProvider dslContextProvider;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider;
    private final FeatureService featureService;

    // Префиксы, которые дописываются к идентификаторам перед тем, как отдать их внешним потребителям
    private static final BiMap<String, BidModifierType> PREFIXES_TYPE =
            ImmutableBiMap.<String, BidModifierType>builder()
                    .put("10", MOBILE_MULTIPLIER)
                    .put("11", DEMOGRAPHY_MULTIPLIER)
                    .put("12", RETARGETING_MULTIPLIER)
                    .put("13", GEO_MULTIPLIER)
                    .put("14", VIDEO_MULTIPLIER)
                    .put("19", DESKTOP_MULTIPLIER)
                    .put("15", PERFORMANCE_TGO_MULTIPLIER)
                    .put("16", AB_SEGMENT_MULTIPLIER)
                    .put("17", BANNER_TYPE_MULTIPLIER)
                    .put("18", INVENTORY_MULTIPLIER)
                    .put("20", WEATHER_MULTIPLIER)
                    .put("21", EXPRESS_TRAFFIC_MULTIPLIER)
                    .put("22", EXPRESS_CONTENT_DURATION_MULTIPLIER)
                    .put("23", TRAFARET_POSITION_MULTIPLIER)
                    .put("24", PRISMA_INCOME_GRADE_MULTIPLIER)
                    .put("25", RETARGETING_FILTER)
                    .put("26", SMARTTV_MULTIPLIER)
                    .put("27", TABLET_MULTIPLIER)
                    .put("28", DESKTOP_ONLY_MULTIPLIER)
                    .build();

    public static final Map<BidModifierType, String> TYPE_PREFIXES = PREFIXES_TYPE.inverse();

    @Autowired
    public BidModifierService(BidModifierRepository bidModifierRepository,
                              AdGroupRepository adGroupRepository,
                              ShardHelper shardHelper,
                              RbacService rbacService,
                              CampaignRepository campaignRepository,
                              CampaignAccessCheckRepository campaignAccessCheckRepository,
                              BidModifierTypeSupportDispatcher typeSupportDispatcher,
                              BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher,
                              AddBidModifiersValidationService addBidModifiersValidationService,
                              ToggleBidModifiersValidationService toggleBidModifiersValidationService,
                              LogBidModifiersService logBidModifiersService,
                              DslContextProvider dslContextProvider,
                              CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                              RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider,
                              FeatureService featureService) {
        this.bidModifierRepository = bidModifierRepository;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.campaignRepository = campaignRepository;
        this.campaignAccessCheckRepository = campaignAccessCheckRepository;
        this.typeSupportDispatcher = typeSupportDispatcher;
        this.validationTypeSupportDispatcher = validationTypeSupportDispatcher;
        this.addBidModifiersValidationService = addBidModifiersValidationService;
        this.toggleBidModifiersValidationService = toggleBidModifiersValidationService;
        this.logBidModifiersService = logBidModifiersService;
        this.dslContextProvider = dslContextProvider;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.accessibilityCheckerProvider = accessibilityCheckerProvider;
        this.featureService = featureService;
    }

    /**
     * Получает наборы корректировок по идентификаторам.
     * Идентификаторы сгруппированы по типам, т.к. для разных типов корректировок идентификаторы
     * указывают на разные таблицы (см {@link #getExternalId(long, BidModifierType)} для подробной информации.
     */
    public List<BidModifier> getByIds(ClientId clientId, Multimap<BidModifierType, Long> idsByType, long operatorUid) {
        return getByIds(clientId, idsByType, emptySet(), emptySet(), ALL_TYPES_INTERNAL, ALL_LEVELS, operatorUid);
    }

    /**
     * Получает наборы корректировок по идентификаторам.
     * Идентификаторы сгруппированы по типам, т.к. для разных типов корректировок идентификаторы
     * указывают на разные таблицы (см {@link #getExternalId(long, BidModifierType)} для подробной информации.
     * <p>
     * Если adGroupIdsOnly не пустой, то добавляется фильтрация по campaignIdsOnly+adGroupIdsOnly
     * Если adGroupIdsOnly пустой, но campaignIds не пустой, то добавляется фильтрация только по campaignIdsOnly
     */
    public List<BidModifier> getByIds(ClientId clientId, Multimap<BidModifierType, Long> idsByType,
                                      Set<Long> campaignIdsOnly, Set<Long> adGroupIdsOnly,
                                      Set<BidModifierType> types, Set<BidModifierLevel> levels, long operatorUid) {
        checkArgument(!idsByType.isEmpty(), "idsByType shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // Получаем все наборы
        List<BidModifier> allModifiers =
                bidModifierRepository.getByIds(shard, idsByType, campaignIdsOnly, adGroupIdsOnly, types, levels);

        // Убираем те, которые оператор не должен видеть
        Set<Long> allCampaignIds = allModifiers.stream().map(BidModifier::getCampaignId).collect(Collectors.toSet());
        Set<Long> visibleCampaignIds = getVisibleCampaignIds(shard, clientId, operatorUid, allCampaignIds);

        return allModifiers.stream().filter(it -> visibleCampaignIds.contains(it.getCampaignId())).collect(toList());
    }

    /**
     * Получает наборы корректировок по идентификаторам групп (с проверкой доступа оператора и клиента к кампаниям).
     * Если campaignIdsOnly не пустой, то добавляется фильтрация по campaignIdsOnly.
     */
    public List<BidModifier> getByAdGroupIds(ClientId clientId, Set<Long> adGroupIds, Set<Long> campaignIdsOnly,
                                             Set<BidModifierType> types, Set<BidModifierLevel> levels,
                                             long operatorUid) {
        checkArgument(!adGroupIds.isEmpty(), "adGroupIds shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // Это соответствие нужно использовать при формировании SQL запроса, чтобы работал индекс (cid, pid)
        Map<Long, Long> campaignIdsByAdGroupIds =
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, clientId, adGroupIds);

        // Получаем те кампании, которые пользователь может просматривать
        Set<Long> allCampaignIds;
        if (campaignIdsOnly.isEmpty()) {
            allCampaignIds = new HashSet<>(campaignIdsByAdGroupIds.values());
        } else {
            allCampaignIds = Sets.intersection(new HashSet<>(campaignIdsByAdGroupIds.values()), campaignIdsOnly);
        }
        Set<Long> visibleCampaignIds = getVisibleCampaignIds(shard, clientId, operatorUid, allCampaignIds);

        Map<Long, Long> filteredCampaignIdsByAdGroupIds =
                campaignIdsByAdGroupIds.entrySet().stream()
                        .filter(e -> visibleCampaignIds.contains(e.getValue()))
                        .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));

        return bidModifierRepository.getByAdGroupIds(shard, filteredCampaignIdsByAdGroupIds, types, levels);
    }

    /**
     * Получает наборы корректировок по идентификаторам групп (без проверки доступа оператора и клиента к кампаниям).
     */
    public List<BidModifier> getByAdGroupIds(int shard, Set<Long> adGroupIds, Set<BidModifierType> types,
                                             Set<BidModifierLevel> levels) {
        checkArgument(!adGroupIds.isEmpty(), "adGroupIds shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);

        return bidModifierRepository.getByAdGroupIds(shard, campaignIdsByAdGroupIds, types, levels);
    }

    /**
     * @see #getByCampaignIds(DSLContext, Set, Set, Set)
     */
    public List<BidModifier> getByCampaignIds(int shard, Set<Long> campaignIds, Set<BidModifierType> types,
                                              Set<BidModifierLevel> levels) {
        return getByCampaignIds(dslContextProvider.ppc(shard), campaignIds, types, levels);
    }

    /**
     * Получает наборы корректировок по идентификаторам кампаний (без проверки доступа оператора и клиента к кампаниям).
     */
    public List<BidModifier> getByCampaignIds(DSLContext dslContext, Set<Long> campaignIds,
                                              Set<BidModifierType> types,
                                              Set<BidModifierLevel> levels) {
        checkArgument(!campaignIds.isEmpty(), "campaignIds shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        return bidModifierRepository.getByCampaignIds(dslContext, campaignIds, types, levels);
    }

    /**
     * Получает наборы корректировок по идентификаторам кампаний.
     */
    public List<BidModifier> getByCampaignIds(ClientId clientId, Collection<Long> campaignIds,
                                              Set<BidModifierType> types, Set<BidModifierLevel> levels,
                                              long operatorUid) {
        checkArgument(!campaignIds.isEmpty(), "campaignIds shouldn't be empty");
        checkArgument(!types.isEmpty(), "types shouldn't be empty");
        checkArgument(!levels.isEmpty(), "levels shouldn't be empty");

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // Получаем те кампании, которые пользователь может просматривать
        Set<Long> visibleCampaignIds = getVisibleCampaignIds(shard, clientId, operatorUid, campaignIds);

        return bidModifierRepository.getByCampaignIds(dslContextProvider.ppc(shard), visibleCampaignIds, types, levels);
    }

    public List<BidModifierRetargetingFilter> getRetargetingFilterModifier(ClientId clientId,
                                                                           @Nullable Set<Long> ids,
                                                                           @Nullable Set<Long> adGroupIds,
                                                                           @Nullable Set<Long> campaignIds,
                                                                           long operatorUid) {

        List<BidModifier> result = emptyList();
        if (ids != null && !ids.isEmpty()) {
            result = getByIds(
                    clientId,
                    Multimaps.index(ids, id -> RETARGETING_FILTER),
                    campaignIds, adGroupIds,
                    Set.of(RETARGETING_FILTER),
                    ALL_LEVELS,
                    operatorUid);
        } else if (adGroupIds != null && !adGroupIds.isEmpty()) {
            result = getByAdGroupIds(
                    clientId, adGroupIds, campaignIds, Set.of(RETARGETING_FILTER), ALL_LEVELS, operatorUid);
        } else if (campaignIds != null && !campaignIds.isEmpty()) {
            result = getByCampaignIds(clientId, campaignIds, Set.of(RETARGETING_FILTER), ALL_LEVELS, operatorUid);
        }
        return StreamEx.of(result)
                .map(bidModifier -> (BidModifierRetargetingFilter) bidModifier)
                .collect(toList());
    }

    /**
     * Возвращает множество id кампаний, которые доступны для просмотра указанному оператору с учётом
     * выбранного набора accessibleCampaignTypes (в контексте АПИ это будет ограниченный набор типов кампаний).
     */
    private Set<Long> getVisibleCampaignIds(int shard, ClientId clientId, long operatorUid,
                                            Collection<Long> campaignIds) {
        Set<Long> rbacVisibleCampaignIds = rbacService.getVisibleCampaigns(operatorUid, campaignIds);
        CampaignAccessibiltyChecker campaignAccessibiltyChecker = accessibilityCheckerProvider.get();
        return campaignAccessCheckRepository.getCampaignsForAccessCheckByCampaignIds(
                shard, campaignAccessibiltyChecker.toAllowableCampaignsRepositoryAdapter(clientId),
                rbacVisibleCampaignIds).keySet();
    }


    /**
     * Добавляет корректировки. Возвращает MassResult, в котором каждому элементу входного списка
     * соответствует либо набор ID добавленных корректировок, либо набор ошибок. Допускает создание выключенных
     * корректировок
     */
    public MassResult<List<Long>> add(List<BidModifier> bidModifiers, ClientId clientId, long operatorUid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return add(bidModifiers, shard, clientId, operatorUid);
    }

    /**
     * Добавляет корректировки. Возвращает MassResult, в котором каждому элементу входного списка
     * соответствует либо набор ID добавленных корректировок, либо набор ошибок. Допускает создание выключенных
     * корректировок
     */
    private MassResult<List<Long>> add(
            List<BidModifier> bidModifiers, int shard, ClientId clientId, long operatorUid) {
        checkState(bidModifiers.stream().allMatch(it -> it.getType() != null));

        return new BidModifierAddOperation(
                operatorUid, clientId, shard, campaignRepository, campaignAccessCheckRepository,
                adGroupRepository, bidModifierRepository,
                this, typeSupportDispatcher, addBidModifiersValidationService,
                dslContextProvider, featureService, accessibilityCheckerProvider.get(), bidModifiers
        ).prepareAndApply();
    }


    /**
     * Удаляет все корректировки с указанных кампаний и их групп
     */
    public MassResult<Long> deleteCampaignModifiers(ClientId clientId, long operatorUid, List<Long> campaignIds) {
        List<BidModifier> bidModifiers = getByCampaignIds(clientId, campaignIds,
                ALL_TYPES,
                Set.of(CAMPAIGN, ADGROUP), operatorUid);
        List<Long> externalModifierIds = List.copyOf(getExternalIdsFlattened(bidModifiers));
        return delete(externalModifierIds, clientId, operatorUid);
    }

    /**
     * Удаляет корректировки по внешним ID. Возвращает MassResult, в котором каждому элементу входного
     * списка соответствует либо набор ID удаленных корректировок, либо набор ошибок.
     */
    public MassResult<Long> delete(List<Long> ids, ClientId clientId, long operatorUid) {
        return createDeleteOperation(ids, clientId, operatorUid)
                .prepareAndApply();
    }

    public BidModifierDeleteOperation createDeleteOperation(List<Long> ids, ClientId clientId, long operatorUid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return new BidModifierDeleteOperation(shard, clientId, operatorUid, campaignRepository,
                this, rbacService, typeSupportDispatcher, ids
        );
    }

    /**
     * Включает/выключает корректировки. Возвращает MassResult, в котором каждому элементу входного списка
     * соответствует либо набор ID обработанных корректировок, либо набор ошибок.
     */
    public MassResult<UntypedBidModifier> toggle(List<UntypedBidModifier> items, ClientId clientId, long operatorUid) {

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<Long> adGroupIds = getAdGroupIds(items);
        Map<Long, Long> campaignIdsByAdGroupIds = adGroupIds.isEmpty() ? emptyMap() :
                adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);

        Set<Long> campaignIds = getCampaignIds(items);

        // Получаем существующие корректировки
        List<BidModifier> foundItems = campaignIdsByAdGroupIds.isEmpty() ? new ArrayList<>() :
                bidModifierRepository.getEmptyBidModifiersByAdGroupIds(shard, campaignIdsByAdGroupIds, ALL_TYPES,
                        ALL_LEVELS);

        foundItems.addAll(campaignIds.isEmpty() ? emptyList() :
                bidModifierRepository.getEmptyBidModifiersByCampaignIds(dslContextProvider.ppc(shard), campaignIds,
                        ALL_TYPES, EnumSet.of(CAMPAIGN)));

        // Создаем мапу: ключ из трех полей - корректировка
        Map<BidModifierKey, BidModifier> foundItemsMap = foundItems.stream()
                .collect(toMap(BidModifierKey::new, o -> o));

        // Заполняем поле id в переданных корректировках находя их в мапе ключу из трех полей
        for (BidModifier item : items) {
            @Nullable BidModifierKey key = makeKey(item, campaignIdsByAdGroupIds);
            @Nullable BidModifier foundItem = (key == null) ? null : foundItemsMap.get(key);
            if (foundItem != null) {
                item.setId(foundItem.getId());
            }
        }

        Set<Long> allCampaignIds = new HashSet<>(campaignIds);
        allCampaignIds.addAll(campaignIdsByAdGroupIds.values());

        //Выполняем валидацию
        ValidationResult<List<UntypedBidModifier>, Defect> vr = toggleBidModifiersValidationService
                .validate(items, adGroupIds, allCampaignIds, operatorUid, clientId);

        List<UntypedBidModifier> validItems = getValidItems(vr);

        if (validItems.isEmpty()) {
            return MassResult.brokenMassAction(items, vr);
        }

        Set<BidModifierKey> validItemKeys = validItems
                .stream()
                .map(item -> makeKey(item, campaignIdsByAdGroupIds))
                .collect(toSet());

        // Готовим изменения для репозитория
        List<AppliedChanges<BidModifier>> appliedChangesList = new ArrayList<>();

        for (BidModifier item : validItems) {
            ModelChanges<BidModifier> modelChanges =
                    new ModelChanges<>(item.getId(), BidModifier.class);
            modelChanges.process(item.getEnabled(), BidModifier.ENABLED);
            AppliedChanges<BidModifier> appliedChanges =
                    modelChanges.applyTo(foundItemsMap.get(makeKey(item, campaignIdsByAdGroupIds)));
            appliedChangesList.add(appliedChanges);
        }

        // В транзакции включаем/выключаем корректировки и синхронизируем кампании и группы
        dslContextProvider.ppc(shard).transaction(configuration -> {
            DSLContext txContext = configuration.dsl();
            bidModifierRepository.updateHierarchicalMultiplierEnabled(txContext, appliedChangesList);
            setBsSyncedForChangedCampaignsAndAdGroups(txContext, validItemKeys);
        });

        // Пишем в лог
        List<LogBidModifierData> logItems = validItems.stream().map(modifier ->
                        new LogBidModifierData(modifier.getCampaignId() == null
                                ? campaignIdsByAdGroupIds.get(modifier.getAdGroupId()) : modifier.getCampaignId(),
                                modifier.getAdGroupId())
                                .withEnabledChanges(new IsEnabledChange(modifier.getId(), modifier.getEnabled())))
                .collect(toList());

        logBidModifiersService.logBidModifiers(logItems, operatorUid);

        return MassResult.successfulMassAction(items, vr);
    }

    @Nullable
    private BidModifierKey makeKey(BidModifier modifier, Map<Long, Long> campaignIdsByAdGroupIds) {
        @Nullable Long campaignId = modifier.getCampaignId() == null
                ? campaignIdsByAdGroupIds.get(modifier.getAdGroupId())
                : modifier.getCampaignId();

        return campaignId == null ? null : new BidModifierKey(
                campaignId,
                modifier.getAdGroupId(),
                modifier.getType());
    }

    private Set<Long> getCampaignIds(List<UntypedBidModifier> items) {
        return items.stream()
                .filter(item -> item.getAdGroupId() == null)
                .map(BidModifier::getCampaignId)
                .collect(toSet());
    }

    private Set<Long> getAdGroupIds(List<UntypedBidModifier> items) {
        return items.stream()
                .filter(item -> item.getAdGroupId() != null)
                .map(BidModifier::getAdGroupId)
                .collect(toSet());
    }

    @Nullable
    public static BidModifierType getRealType(long id) {
        // Если число отрицательное или в нём меньше двух цифр, то это некорректный идентификатор
        if (id < 100) {
            return null;
        }
        return PREFIXES_TYPE.get(Long.toString(id).substring(0, 2));
    }

    public static Long getRealId(long id) {
        if (id < 100) {
            return null;
        }
        return Long.parseLong(Long.toString(id).substring(2));
    }

    public static Multimap<BidModifierType, Long> getRealIdsGroupedByType(Collection<Long> externalIds) {
        return Multimaps.transformValues(
                Multimaps.index(filterList(externalIds, id -> getRealType(id) != null),
                        BidModifierService::getRealType),
                BidModifierService::getRealId);
    }

    /**
     * Возвращает "внешнее" представление идентификатора корректировки.
     * Сейчас эта штука работает так: на таблице hierarchical_multipliers и дочерних по отношению к ней
     * таблицах *_multiplier_values генерируются сквозные идентификаторы. Они возвращаются ответом пользователю в АПИ.
     * Потом пользователь может передавать их в ручки set/toggle/delete. И для того, чтобы быстро понимать,
     * с какой таблицей работать, были введены префиксы.
     * Пример: в базе мобильная корректировка хранится с id=2884794, а пользователь получит её как 102884794.
     */
    public static long getExternalId(long realId, BidModifierType type) {
        return Long.parseLong(TYPE_PREFIXES.get(type) + realId);
    }

    /**
     * Выполняет установку наборов корректировок на указанные кампании и группы объявлений.
     * Новые корректировки заменят старые. Если новых корректировок для какой-то группы/кампании указано не было,
     * но группа/кампания присутствует в objectsToModify, то с неё все корректировки будут удалены.
     *
     * @return набор объектов (кампаний или групп), в которых что-то было изменено
     */
    public Set<CampaignIdAndAdGroupIdPair> replaceModifiers(ClientId clientId, long operatorUid, int shard,
                                                            List<BidModifier> modifiers,
                                                            Set<CampaignIdAndAdGroupIdPair> objectsToModify) {
        // campaignId, enabled и type должны быть проставлены вызывающим кодом
        checkState(modifiers.stream().allMatch(it -> it.getCampaignId() != null && it.getType() != null));

        if (!objectsToModify.isEmpty()) {
            typeSupportDispatcher.prepareSystemFields(modifiers);
            return dslContextProvider.ppc(shard).transactionResult(configuration -> {
                DSLContext txContext = configuration.dsl();

                Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups = bidModifierRepository
                        .replaceModifiers(txContext, modifiers, objectsToModify, clientId, operatorUid);

                // Сбрасываем статус синхронизации с БК
                resetSyncStatus(txContext, changedCampaignsAndAdGroups);
                return changedCampaignsAndAdGroups;
            });
        }
        return Collections.emptySet();
    }

    /**
     * Выполняет установку наборов корректировок на указанные кампании.
     * Новые корректировки заменят старые. Если новых корректировок для какой-то кампании указано не было,
     * но кампания присутствует в affectedCampaignIds, то с неё все корректировки будут удалены.
     * Статус синхронизации с БК у кампаний сбрасывается вызывающим кодом
     */
    public void replaceCampaignModifiers(DSLContext txContext, ClientId clientId, long operatorUid,
                                         List<BidModifier> modifiers, Set<Long> affectedCampaignIds) {
        // campaignId, enabled и type должны быть проставлены вызывающим кодом
        checkState(modifiers.stream().allMatch(it -> it.getCampaignId() != null && it.getType() != null));

        if (!affectedCampaignIds.isEmpty()) {
            Set<CampaignIdAndAdGroupIdPair> affectedObjects = mapSet(affectedCampaignIds,
                    id -> new CampaignIdAndAdGroupIdPair().withCampaignId(id));
            typeSupportDispatcher.prepareSystemFields(modifiers);
            bidModifierRepository
                    .replaceModifiers(txContext, modifiers, affectedObjects, clientId, operatorUid);
        }
    }

    private void resetSyncStatus(DSLContext txContext, Set<CampaignIdAndAdGroupIdPair> changedCampaignsAndAdGroups) {
        Set<Long> campaignIdsForSync = changedCampaignsAndAdGroups.stream()
                .filter(affectedObject -> affectedObject.getAdGroupId() == null)
                .map(CampaignIdAndAdGroupIdPair::getCampaignId)
                .collect(toSet());
        Set<Long> adGroupIdsForSync = changedCampaignsAndAdGroups.stream()
                .filter(affectedObject -> affectedObject.getAdGroupId() != null)
                .map(CampaignIdAndAdGroupIdPair::getAdGroupId)
                .collect(toSet());
        setBsSyncedForChangedCampaignsAndAdGroups(txContext, campaignIdsForSync, adGroupIdsForSync);
    }

    /**
     * Выполняет установку наборов корректировок на указанные кампании и группы объявлений.
     * Новые корректировки заменят старые. Если новых корректировок для какой-то группы/кампании указано не было,
     * но группа/кампания присутствует в objectsToModify, то с неё все корректировки будут удалены.
     *
     * @return набор объектов (кампаний или групп), в которых что-то было изменено
     */
    public Set<CampaignIdAndAdGroupIdPair> replaceModifiers(ClientId clientId, long operatorUid,
                                                            List<BidModifier> modifiers,
                                                            Set<CampaignIdAndAdGroupIdPair> objectsToModify) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return replaceModifiers(clientId, operatorUid, shard, modifiers, objectsToModify);
    }

    /**
     * Удаляет корректировки из наборов и наборы целиком, если в них не осталось больше корректировок (в базе).
     *
     * @param bidModifiers Наборы корректировок для удаления. Удаляются все adjustment'ы, указанные в них
     */
    public void deleteAdjustments(int shard, ClientId clientId, long operatorUid,
                                  Collection<BidModifier> bidModifiers) {
        if (bidModifiers.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard).transaction(configuration -> {
            DSLContext txContext = configuration.dsl();

            // Выполняем удаление
            bidModifierRepository.deleteAdjustments(txContext, clientId, operatorUid, bidModifiers);

            List<BidModifierKey> bidModifierKeys = bidModifiers.stream().map(BidModifierKey::new).collect(toList());

            // Сбрасываем статус синхронизации с БК
            setBsSyncedForChangedCampaignsAndAdGroups(txContext, bidModifierKeys);
        });
    }

    /**
     * Сбрасывает статус синхронизации у кампаний и групп, которым принадлежали изменённые наборы корректировок.
     * Должен быть вызван в контексте транзакции.
     */
    void setBsSyncedForChangedCampaignsAndAdGroups(DSLContext txContext, Collection<BidModifierKey> bidModifiers) {
        List<Long> campaignIdsForSync = bidModifiers.stream()
                .filter(hm -> hm.getAdGroupId() == null)
                .map(BidModifierKey::getCampaignId)
                .collect(toList());

        List<Long> adGroupIdsForSync = bidModifiers.stream()
                .map(BidModifierKey::getAdGroupId)
                .filter(Objects::nonNull)
                .collect(toList());

        setBsSyncedForChangedCampaignsAndAdGroups(txContext, campaignIdsForSync, adGroupIdsForSync);
    }

    private void setBsSyncedForChangedCampaignsAndAdGroups(DSLContext txContext,
                                                           Collection<Long> campaignIdsForSync,
                                                           Collection<Long> adGroupIdsForSync) {
        if (!campaignIdsForSync.isEmpty()) {
            campaignRepository.updateStatusBsSynced(txContext, campaignIdsForSync, StatusBsSynced.NO);
        }

        if (!adGroupIdsForSync.isEmpty()) {
            adGroupRepository.updateStatusBsSynced(txContext.configuration(), adGroupIdsForSync, StatusBsSynced.NO);
        }
    }

    /**
     * Устанавливает новое значение процентов для указанных корректировок.
     */
    public MassResult<Long> set(List<ModelChanges<BidModifierAdjustment>> modelChanges, ClientId clientId,
                                long operatorUid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return new BidModifierSetOperation(shard, clientId, operatorUid, this, campaignRepository,
                adGroupRepository, typeSupportDispatcher, validationTypeSupportDispatcher, featureService,
                campaignSubObjectAccessCheckerFactory, modelChanges).prepareAndApply();
    }

    /**
     * Обновляет значения коэффициентов на корректировках и переотправляет изменённые кампании/группы в БК.
     *
     * @param clientId                 id клиента. Нужен для определения страны, в которой работает клиент
     *                                 (для транслокальности в геокорректировках)
     * @param applicableAppliedChanges Применяемые изменения
     * @param bidModifiers             Наборы корректировок, которые затрагиваются этими изменениями. Нужны для того,
     *                                 чтобы для геокорректировок не ходить лишний раз в базу перед тем, как забрать.
     *                                 Сюда не обязательно передавать наборы, соответствующие только валидным элементам,
     *                                 они не будут изменяться, а только служить информацией для typeSupport'ов
     */
    public void updatePercents(int shard, ClientId clientId, long operatorUid,
                               List<AppliedChanges<BidModifierAdjustment>> applicableAppliedChanges,
                               List<BidModifier> bidModifiers) {
        dslContextProvider.ppc(shard).transaction(configuration -> {
            DSLContext txContext = configuration.dsl();

            // Обновляем значения процентов
            bidModifierRepository.updatePercents(
                    txContext, clientId, operatorUid, applicableAppliedChanges, bidModifiers);

            // Переотправляем в БК
            List<BidModifierKey> bidModifierKeys = bidModifiers.stream().map(BidModifierKey::new).collect(toList());
            setBsSyncedForChangedCampaignsAndAdGroups(txContext, bidModifierKeys);
        });
    }

    /**
     * Считает максимальную и минимальную возможные корректировки ставок для групп.
     * <p>
     * Для каждой группы считает корректировку, которая может получиться, при выполнении в один момент
     * всех возможных повышающих корректировок -- верхняя граница корректировок
     * <p>
     * Для каждой группы считает корректировку, которая может получиться, при выполнении в один момент
     * всех возможных понижающих корректировок -- нижняя граница корректировок
     * <p>
     * Пример: если указана моб. корректировка -50%, соцдем +200% и ретаргетинг +300%, -- верхняя граница 1200% от
     * исходной ставки
     * нижняя граница 50% от исходной ставки
     * <p>
     * Если повышающих корректировок нет, в качестве верхней границы берется наименее понижающая.
     * <p>
     * Если выставлена корректировка одного типа и на группу и на кампанию группы, берется корректировка группы
     *
     * @param clientId              идентификатор клиента
     * @param operatorUid           идентификатор оператора
     * @param adGroupIdToCampaignId id группы к id кампании
     * @return id группы к границам корректировок
     */
    public Map<Long, MultipliersBounds> calculateMultipliersBoundsForAdGroups(ClientId clientId, long operatorUid,
                                                                              Map<Long, Long> adGroupIdToCampaignId) {
        Set<Long> adGroupIds = adGroupIdToCampaignId.keySet();
        if (adGroupIds.isEmpty()) {
            return Collections.emptyMap();
        }
        List<BidModifier> adGroupsBidModifiers =
                getByAdGroupIds(clientId, adGroupIds, Collections.emptySet(), ALL_TYPES,
                        Collections.singleton(ADGROUP), operatorUid);
        Map<Long, List<BidModifier>> adGroupsBidModifiersByAdGroupIds = StreamEx.of(adGroupsBidModifiers)
                .groupingBy(BidModifier::getAdGroupId);

        Set<Long> campaignIds = new HashSet<>(adGroupIdToCampaignId.values());
        List<BidModifier> campaignsBidModifiers = getByCampaignIds(clientId, campaignIds, ALL_TYPES,
                Collections.singleton(BidModifierLevel.CAMPAIGN), operatorUid);
        Map<Long, List<BidModifier>> campaignsBidModifiersByCampaignIds = StreamEx.of(campaignsBidModifiers)
                .groupingBy(BidModifier::getCampaignId);

        return EntryStream.of(adGroupsBidModifiersByAdGroupIds)
                .mapToValue((adGroupId, adGroupBidModifiers) -> {
                    Long campaignId = adGroupIdToCampaignId.get(adGroupId);
                    List<BidModifier> campaignBidModifiers =
                            campaignsBidModifiersByCampaignIds.getOrDefault(campaignId, emptyList());
                    return calculateMultipliersBoundsForAdGroup(adGroupBidModifiers, campaignBidModifiers);
                }).toMap();
    }

    private MultipliersBounds calculateMultipliersBoundsForAdGroup(List<BidModifier> adGroupModifiers,
                                                                   List<BidModifier> campaignModifiers) {
        Map<BidModifierType, BidModifier> adGroupModifiersByTypes = listToMap(adGroupModifiers, BidModifier::getType);

        Map<BidModifierType, BidModifier> campaignModifiersByTypes = listToMap(campaignModifiers, BidModifier::getType);

        List<List<BidModifierAdjustment>> applicableBidModifierAdjustments =
                StreamEx.of(adGroupModifiersByTypes.keySet())
                        .append(campaignModifiersByTypes.keySet())
                        .distinct()
                        .map(type -> adGroupModifiersByTypes.get(type) != null ? adGroupModifiersByTypes.get(type)
                                : campaignModifiersByTypes.get(type))
                        .map(bidModifier -> typeSupportDispatcher.getTypeSupport(bidModifier.getType())
                                .getAdjustments(bidModifier))
                        .toList();


        int upperBound = calculateUpperBound(applicableBidModifierAdjustments);
        int lowerBound = calculateLowerBound(applicableBidModifierAdjustments);

        return new MultipliersBounds()
                .withLower(lowerBound)
                .withUpper(upperBound);

    }

    private static int calculateUpperBound(List<List<BidModifierAdjustment>> applicableBidModifierAdjustments) {
        Function<List<BidModifierAdjustment>, BidModifierAdjustment> maxBidModifierAdjustment =
                bidModifier -> StreamEx.of(bidModifier)
                        .maxByInt(BidModifierAdjustment::getPercent)
                        .get();

        List<Integer> percents = StreamEx.of(applicableBidModifierAdjustments)
                .map(maxBidModifierAdjustment)
                .map(BidModifierAdjustment::getPercent)
                .toList();

        Integer maxPercent = Collections.max(percents);

        boolean hasIncreasingBidModifier = maxPercent > ONE_HUNDRED;
        if (!hasIncreasingBidModifier) {
            return maxPercent;
        }

        return StreamEx.of(percents)
                .filter(percent -> percent >= ONE_HUNDRED)
                .map(Integer::doubleValue)
                .foldLeft((currentPercent, nextPercent) -> currentPercent * nextPercent / ONE_HUNDRED)
                .orElse(ONE_HUNDRED.doubleValue())
                .intValue();

    }

    private static int calculateLowerBound(List<List<BidModifierAdjustment>> applicableBidModifierAdjustments) {
        Function<List<BidModifierAdjustment>, BidModifierAdjustment> minBidModifierAdjustment =
                bidModifier -> StreamEx.of(bidModifier)
                        .minByInt(BidModifierAdjustment::getPercent)
                        .get();

        return StreamEx.of(applicableBidModifierAdjustments)
                .map(minBidModifierAdjustment)
                .map(BidModifierAdjustment::getPercent)
                .filter(percent -> percent < ONE_HUNDRED)
                .map(Integer::doubleValue)
                .foldLeft((currentPercent, nextPercent) -> currentPercent * nextPercent / ONE_HUNDRED)
                .orElse(ONE_HUNDRED.doubleValue())
                .intValue();
    }

    public Set<Long> getExternalIdsFlattened(Collection<BidModifier> bidModifiers) {
        Multimap<BidModifierType, BidModifier> byType = Multimaps.index(bidModifiers, BidModifier::getType);

        StreamEx<Long> multiValuesIdsStream = EntryStream.of(byType.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .filterKeys(BidModifierTypeSupport::isMultiValues)
                .mapKeys(typeSupport -> (BidModifierMultipleValuesTypeSupport) typeSupport)
                .flatMapKeyValue((typeSupport, modifiers) ->
                        modifiers.stream()
                                .map((Function<BidModifier, List<BidModifierAdjustment>>) typeSupport::getAdjustments)
                                .flatMap(Collection::stream)
                                .map(BidModifierAdjustment::getId)
                                .map(id -> BidModifierService.getExternalId(id, typeSupport.getType())));

        StreamEx<Long> singleValueIdsStream = EntryStream.of(byType.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .filterKeys(typeSupport -> !typeSupport.isMultiValues())
                .flatMapKeyValue((typeSupport, modifiers) ->
                        modifiers.stream()
                                .map(BidModifier::getId)
                                .map(id -> BidModifierService.getExternalId(id, typeSupport.getType())));

        return Stream.concat(multiValuesIdsStream, singleValueIdsStream).collect(toSet());
    }


    @Override
    public List<BidModifier> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, BidModifierKey> keysByIds = bidModifierRepository.getBidModifierKeysByIds(shard, ids);
        Map<BidModifierKey, BidModifier> bidModifiersByKeys
                = bidModifierRepository.getBidModifiersByKeys(shard, keysByIds.values());
        return StreamEx.of(ids).map(keysByIds::get)
                .nonNull()
                .map(bidModifiersByKeys::get)
                .toList();
    }

    @Override
    public MassResult<List<Long>> add(
            ClientId clientId, Long operatorUid, List<BidModifier> entities, Applicability applicability) {
        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("BidModifiers add operation does not support full applicability.");
        }
        return this.add(entities, clientId, operatorUid);
    }

    public MassResult<List<Long>> copy(CopyOperationContainer copyContainer, List<BidModifier> entities,
                                       Applicability applicability) {
        if (applicability == Applicability.FULL) {
            throw new UnsupportedOperationException("BidModifiers add operation does not support full applicability.");
        }
        return this.add(entities, copyContainer.getShardTo(), copyContainer.getClientIdTo(), copyContainer.getOperatorUid());
    }
}
