package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.UntypedBidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupportDispatcher;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.util.GuavaCollectors.toListMultimap;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

/**
 * Операция установки новых значений коэффициентов для указанных корректировок.
 * Используется только из API.
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/bidmodifiers/set-docpage/
 */
public class BidModifierSetOperation extends SimpleAbstractUpdateOperation<BidModifierAdjustment, Long> {
    private final BidModifierService bidModifierService;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final BidModifierTypeSupportDispatcher typeSupportDispatcher;
    private final BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher;
    private final FeatureService featureService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final int shard;
    private final ClientId clientId;
    private final long operatorUid;

    // Полная информация по загруженным из базы корректировкам
    private List<BidModifier> bidModifiers;

    @SuppressWarnings("WeakerAccess")
    public BidModifierSetOperation(int shard, ClientId clientId, long operatorUid,
                                   BidModifierService bidModifierService,
                                   CampaignRepository campaignRepository,
                                   AdGroupRepository adGroupRepository,
                                   BidModifierTypeSupportDispatcher typeSupportDispatcher,
                                   BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher,
                                   FeatureService featureService,
                                   CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                   List<ModelChanges<BidModifierAdjustment>> modelChanges) {
        super(Applicability.PARTIAL, modelChanges, id -> new UntypedBidModifierAdjustment().withId(id));
        this.bidModifierService = bidModifierService;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.typeSupportDispatcher = typeSupportDispatcher;
        this.validationTypeSupportDispatcher = validationTypeSupportDispatcher;
        this.featureService = featureService;
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;

        // Проверить, что операция инициализирована поддерживаемыми изменениями
        modelChanges.forEach(mc -> checkArgument(
                mc.getChangedPropsNames().equals(singleton(BidModifierAdjustment.PERCENT)),
                "ModelChanges with unsupported changes is passed to BidModifierSetOperation"));
    }

    /**
     * Проводит валидацию списка ModelChanges перед загрузкой моделей (первый шаг валидации).
     */
    @Override
    protected ValidationResult<List<ModelChanges<BidModifierAdjustment>>, Defect> validateModelChanges(
            List<ModelChanges<BidModifierAdjustment>> modelChanges) {
        // Соответствие типа корректировки и id для того, чтобы далее можно было их получить из базы
        Multimap<BidModifierType, Long> idsByType = modelChanges.stream().collect(toListMultimap(
                mc -> typeSupportDispatcher.getTypeByAdjustmentClass(mc.getModelType()),
                ModelChanges::getId));

        // Получаем из базы
        // Нужно это сделать именно на этапе валидации, т.к. если вызвать этот метод при загрузке моделей,
        // то может оказаться, что часть корректировок не существует, и это нарушит инварианты AbstractUpdateOperation
        bidModifiers = bidModifierService.getByIds(clientId, idsByType, operatorUid);

        ListValidationBuilder<ModelChanges<BidModifierAdjustment>, Defect> vb =
                ListValidationBuilder.of(modelChanges);

        vb.checkEach(unique(ModelChanges::getId),
                new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_SINGLE_ADJUSTMENT));

        Map<Long, Long> campaignIdByAdjustmentId = getCampaignIdByAdjustmentIdMap(bidModifiers);

        Map<Long, CampaignType> adjustmentIdToCampaignType = getAdjustmentIdToCampaignType(bidModifiers);
        Map<Long, AdGroup> adjustmentIdToAdGroup = getAdjustmentIdToAdGroup(bidModifiers);

        CampaignSubObjectAccessConstraint accessConstraint = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, mapList(bidModifiers, BidModifier::getCampaignId))
                .createBidModifierConstraint(CampaignAccessType.READ_WRITE, campaignIdByAdjustmentId);

        vb.checkEach((Constraint<ModelChanges<BidModifierAdjustment>, Defect>)
                mc -> accessConstraint.apply(mc.getId()), When.isValid());
        CachingFeaturesProvider featuresProvider = new CachingFeaturesProvider(featureService);

        //лютый костыль, дополняющий проверку не только на значение процента, но и на связь процент/тип ОС для
        //мобильных корректировок. пришел из DIRECT-114200, должен уйти с DIRECT-114437
        List<ModelChanges<BidModifierAdjustment>> mobileAdjustmentModelChanges = StreamEx.of(modelChanges)
                .filter(modelChangesItem -> modelChangesItem.getModelType() == BidModifierMobileAdjustment.class)
                .distinct(ModelChanges::getId)
                .toList();

        Map<Long, BidModifierMobile> mobileModifiersMap = emptyMap();
        if (!isEmpty(mobileAdjustmentModelChanges)) {
            List<BidModifier> mobileModifiers = bidModifierService.get(clientId, operatorUid,
                    mapList(mobileAdjustmentModelChanges, ModelChanges::getId));
            mobileModifiersMap = listToMap(mobileModifiers, BidModifier::getId,
                    modifier -> (BidModifierMobile) modifier);
        }

        vb.checkEachBy(
                percentValidator(clientId, adjustmentIdToCampaignType, adjustmentIdToAdGroup, featuresProvider,
                        mobileModifiersMap),
                When.isValid()
        );
        return vb.getResult();
    }

    private Validator<ModelChanges<BidModifierAdjustment>, Defect> percentValidator(
            ClientId clientId,
            Map<Long, CampaignType> adjustmentIdToCampaignType,
            Map<Long, AdGroup> adjustmentIdToAdGroup,
            CachingFeaturesProvider featuresProvider,
            Map<Long, BidModifierMobile> mobileModifiersMap) {
        return mc -> {
            BidModifierType type = typeSupportDispatcher.getTypeByAdjustmentClass(mc.getModelType());
            BidModifierValidationTypeSupport<BidModifier> validationTypeSupport =
                    validationTypeSupportDispatcher.getValidationTypeSupportByType(type);

            Long adjustmentId = mc.getId();
            CampaignType campaignType = adjustmentIdToCampaignType.get(adjustmentId);
            AdGroup adGroup = adjustmentIdToAdGroup.get(adjustmentId);

            Integer percent = mc.getChangedProp(BidModifierAdjustment.PERCENT);
            ValidationResult<Integer, Defect> vr = validationTypeSupport
                    .validatePercent(percent, campaignType, adGroup, clientId, featuresProvider,
                            mobileModifiersMap.get(mc.getId()));
            if (vr.hasErrors()) {
                return ValidationResult.failed(mc, vr.getErrors().get(0));
            } else {
                return ValidationResult.success(mc);
            }
        };
    }

    private Map<Long, CampaignType> getAdjustmentIdToCampaignType(List<BidModifier> bidModifiers) {
        Map<Long, Long> modifierIdToCampaignId = StreamEx.of(bidModifiers)
                .mapToEntry(BidModifier::getCampaignId)
                .flatMapKeys(m -> getAdjustmentIds(m).stream())
                .nonNullKeys()
                .nonNullValues()
                .distinctKeys()
                .toMap();
        Map<Long, CampaignWithType> campaignWithTypeMap =
                campaignRepository.getCampaignsWithTypeByCampaignIds(shard, clientId, modifierIdToCampaignId.values());
        return EntryStream.of(modifierIdToCampaignId)
                .mapValues(campaignWithTypeMap::get)
                .nonNullValues()
                .mapValues(CampaignWithType::getType)
                .toMap();
    }

    private Map<Long, AdGroup> getAdjustmentIdToAdGroup(List<BidModifier> bidModifiers) {
        Map<Long, Long> modifierIdToAdGroupId = StreamEx.of(bidModifiers)
                .mapToEntry(BidModifier::getAdGroupId)
                .flatMapKeys(m -> getAdjustmentIds(m).stream())
                .nonNullKeys()
                .nonNullValues()
                .distinctKeys()
                .toMap();
        Map<Long, AdGroup> adGroupMap =
                StreamEx.of(adGroupRepository.getAdGroups(shard, modifierIdToAdGroupId.values()))
                        .toMap(AdGroup::getId, identity());
        return EntryStream.of(modifierIdToAdGroupId)
                .mapValues(adGroupMap::get)
                .nonNullValues()
                .toMap();
    }

    /**
     * Получает соответствие {id корректировки -> id кампании} для проверки доступов
     */
    private Map<Long, Long> getCampaignIdByAdjustmentIdMap(List<BidModifier> bidModifiers) {
        Map<Long, Set<Long>> adjustmentIdsByCampaignId =
                EntryStream.of(StreamEx.of(bidModifiers).groupingBy(BidModifier::getCampaignId))
                        .mapValues(modifiers -> modifiers.stream().map(this::getAdjustmentIds).collect(toList()))
                        .mapValues(ids -> ids.stream().flatMap(Collection::stream).collect(toSet()))
                        .toMap();

        return EntryStream.of(adjustmentIdsByCampaignId)
                .flatMapValues(Collection::stream)
                .invert()
                .toMap();
    }

    private List<Long> getAdjustmentIds(BidModifier modifier) {
        BidModifierTypeSupport<BidModifier, BidModifierAdjustment> typeSupport =
                typeSupportDispatcher.getTypeSupport(modifier.getType());
        List<BidModifierAdjustment> adjustments = typeSupport.getAdjustments(modifier);
        return adjustments.stream().map(BidModifierAdjustment::getId).collect(toList());
    }

    /**
     * Получает модели для тех ModelChanges, для которых успешно прошел первый этап валидации,
     * для последующего применения к ним изменений и получения объектов {@link AppliedChanges}.
     */
    @Override
    protected Collection<BidModifierAdjustment> getModels(Collection<Long> ids) {
        HashSet<Long> idsSet = new HashSet<>(ids);
        return flattenAdjustments(bidModifiers).stream()
                .filter(it -> idsSet.contains(it.getId()))
                .collect(toList());
    }

    private List<BidModifierAdjustment> flattenAdjustments(Collection<BidModifier> bidModifiers) {
        Multimap<BidModifierType, BidModifier> byType = Multimaps.index(bidModifiers, BidModifier::getType);

        return EntryStream.of(byType.asMap())
                .mapKeys(typeSupportDispatcher::getTypeSupport)
                .flatMapKeyValue((typeSupport, modifiers) ->
                        modifiers.stream()
                                .map(typeSupport::getAdjustments)
                                .flatMap(Collection::stream))
                .collect(toList());
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BidModifierAdjustment>> appliedChanges) {
        // Определяем, какие из загруженных наборов будут действительно изменены
        // (берём только валидные)
        List<BidModifier> affectedBidModifiers = getAffectedBidModifiers(bidModifiers, appliedChanges);

        // Сохраняем в БД
        bidModifierService.updatePercents(shard, clientId, operatorUid, appliedChanges, affectedBidModifiers);

        // Вернуть результат нужно во внешнем формате, с префиксами
        return appliedChanges.stream().map(ac -> {
            BidModifierType type = typeSupportDispatcher.getTypeByAdjustmentClass(ac.getModel().getClass());
            return BidModifierService.getExternalId(ac.getModel().getId(), type);
        }).collect(toList());
    }

    private List<BidModifier> getAffectedBidModifiers(List<BidModifier> bidModifiers,
                                                      List<AppliedChanges<BidModifierAdjustment>> validChanges) {
        Set<Long> validAdjustmentsIds = validChanges.stream().map(AppliedChanges::getModel)
                .map(BidModifierAdjustment::getId)
                .collect(toSet());

        return bidModifiers.stream()
                .filter(modifier -> containsAnyOfAdjustments(modifier, validAdjustmentsIds))
                .collect(toList());
    }

    private boolean containsAnyOfAdjustments(BidModifier modifier, Set<Long> adjustmentIds) {
        BidModifierTypeSupport<BidModifier, BidModifierAdjustment> typeSupport =
                typeSupportDispatcher.getTypeSupport(modifier.getType());

        return typeSupport.getAdjustments(modifier).stream()
                .map(BidModifierAdjustment::getId)
                .anyMatch(adjustmentIds::contains);
    }
}
