package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationDesktopOnlyTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationDesktopTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationMobileTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationSmartTVTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTabletTypeSupport;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationTypeSupportDispatcher;
import ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.deviceBidModifiersAllZeros;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.util.ValidationUtils.transferIssuesFromValidationToValidation;

/**
 * Работа с комплексными моделями наборов корректировок - преимущественно для кода, вызываемого из интерфейса.
 */
@Service
@ParametersAreNonnullByDefault
public class ComplexBidModifierService {
    private final BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher;
    private final FeatureService featureService;

    @Autowired
    public ComplexBidModifierService(BidModifierValidationTypeSupportDispatcher validationTypeSupportDispatcher,
                                     FeatureService featureService) {
        this.validationTypeSupportDispatcher = validationTypeSupportDispatcher;
        this.featureService = featureService;
    }

    public Pair<List<BidModifier>, Multimap<Integer, Integer>> convertFromComplexModelsForAdGroups(
            List<ComplexBidModifier> complexBidModifiers) {

        return convertFromComplexModels(complexBidModifiers, true);
    }

    public Pair<List<BidModifier>, Multimap<Integer, Integer>> convertFromComplexModelsForCampaigns(
            List<ComplexBidModifier> complexBidModifiers) {

        return convertFromComplexModels(complexBidModifiers, false);
    }

    private Pair<List<BidModifier>, Multimap<Integer, Integer>> convertFromComplexModels(
            List<ComplexBidModifier> complexBidModifiers, boolean forAdGroups) {
        // При сохранении наборов на группах объявлений в них не должно быть гео-корректировок и АБ-сегментов
        // Гео-корректировки разрешены только на кампаниях
        checkState(!forAdGroups || complexBidModifiers.stream()
                .allMatch(it -> it.getGeoModifier() == null && it.getAbSegmentModifier() == null));

        List<BidModifier> bidModifiers = new ArrayList<>();
        Multimap<Integer, Integer> complexToFlatIndexMap = ArrayListMultimap.create();

        for (int i = 0; i < complexBidModifiers.size(); i++) {
            ComplexBidModifier complexBidModifier = complexBidModifiers.get(i);

            if (complexBidModifier.getMobileModifier() != null) {
                bidModifiers.add(complexBidModifier.getMobileModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getDesktopModifier() != null) {
                bidModifiers.add(complexBidModifier.getDesktopModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getTabletModifier() != null) {
                bidModifiers.add(complexBidModifier.getTabletModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getDesktopOnlyModifier() != null) {
                bidModifiers.add(complexBidModifier.getDesktopOnlyModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getSmartTVModifier() != null) {
                bidModifiers.add(complexBidModifier.getSmartTVModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getVideoModifier() != null) {
                bidModifiers.add(complexBidModifier.getVideoModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getPerformanceTgoModifier() != null) {
                bidModifiers.add(complexBidModifier.getPerformanceTgoModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getDemographyModifier() != null) {
                bidModifiers.add(complexBidModifier.getDemographyModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getWeatherModifier() != null) {
                bidModifiers.add(complexBidModifier.getWeatherModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getRetargetingModifier() != null) {
                bidModifiers.add(complexBidModifier.getRetargetingModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getGeoModifier() != null) {
                bidModifiers.add(complexBidModifier.getGeoModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getAbSegmentModifier() != null) {
                bidModifiers.add(complexBidModifier.getAbSegmentModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getBannerTypeModifier() != null) {
                bidModifiers.add(complexBidModifier.getBannerTypeModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getInventoryModifier() != null) {
                bidModifiers.add(complexBidModifier.getInventoryModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getExpressionModifiers() != null &&
                    !complexBidModifier.getExpressionModifiers().isEmpty()
            ) {
                // порядок добавления корректировок в список bidModifier будет иметь значение
                // при переносе результатов валидации из плоского списка в комплексные модели
                // см transferValidationResultFlatToComplex
                for (var expressModifier : complexBidModifier.getExpressionModifiers()) {
                    bidModifiers.add(expressModifier);
                    complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
                }
            }
            if (complexBidModifier.getTrafaretPositionModifier() != null) {
                bidModifiers.add(complexBidModifier.getTrafaretPositionModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
            if (complexBidModifier.getRetargetingFilterModifier() != null) {
                bidModifiers.add(complexBidModifier.getRetargetingFilterModifier());
                complexToFlatIndexMap.put(i, bidModifiers.size() - 1);
            }
        }

        // У всех элементов должен быть установлен type
        checkState(bidModifiers.stream().allMatch(it -> it.getType() != null));

        return Pair.of(bidModifiers, complexToFlatIndexMap);
    }

    /**
     * Валидирует плоский список корректировок для операции "заменить всё целиком".
     *
     * @param bidModifiers         Плоский список корректировок
     * @param campaignTypeProvider Функция, которая по индексу в списке возвращает тип кампании для этой корректировки
     */
    public ValidationResult<List<BidModifier>, Defect> validateBidModifiersFlat(
            List<BidModifier> bidModifiers, Multimap<Integer, Integer> complexToFlatIndexMap,
            Function<Integer, CampaignType> campaignTypeProvider,
            Function<Integer, AdGroup> adGroupWithTypeProvider,
            ClientId clientId) {
        ListValidationBuilder<BidModifier, Defect> vb = ListValidationBuilder.of(bidModifiers);

        Map<Integer, Integer> flatToComplexIndexMap = EntryStream.of(complexToFlatIndexMap.asMap())
                .flatMapValues(Collection::stream)
                .collect(toMap(Map.Entry::getValue, Map.Entry::getKey));

        // Валидируем плоский список
        vb.checkEachBy((idx, bidModifier) ->
                validationTypeSupportDispatcher.validateAddStep1(
                        bidModifier,
                        campaignTypeProvider.apply(idx),
                        adGroupWithTypeProvider.apply(idx),
                        clientId,
                        new CachingFeaturesProvider(featureService)
                )
        );

        // Проверяем корректировки на устройство
        checkDeviceCorrectionNotAllZeroes(
                bidModifiers,
                complexToFlatIndexMap,
                vb,
                flatToComplexIndexMap,
                campaignTypeProvider,
                featureService.isEnabledForClientId(clientId, FeatureName.SMARTTV_BID_MODIFIER_ENABLED));

        // проверяем корректировки на инвентарь
        checkInventoryAndBanerCorrectionNotAllZeroes(bidModifiers, complexToFlatIndexMap, vb, flatToComplexIndexMap);

        return vb.getResult();
    }

    private void checkDeviceCorrectionNotAllZeroes(List<BidModifier> bidModifiers,
                                                   Multimap<Integer, Integer> complexToFlatIndexMap,
                                                   ListValidationBuilder<BidModifier, Defect> vb,
                                                   Map<Integer, Integer> flatToComplexIndexMap,
                                                   Function<Integer, CampaignType> campaignTypeProvider,
                                                   boolean smartTvFeatureIsEnabled) {
        vb.checkEachBy((idx, bidModifier) -> {

            boolean checkForSmartTv = smartTvFeatureIsEnabled && (
                    campaignTypeProvider.apply(idx) == CampaignType.CPM_BANNER ||
                            campaignTypeProvider.apply(idx) == CampaignType.CPM_DEALS);

            Integer complexIndex = flatToComplexIndexMap.get(idx);
            Collection<BidModifier> adjacentModifiers = StreamEx.of(complexToFlatIndexMap.get(complexIndex))
                    .map(bidModifiers::get)
                    .toList();

            if (bidModifier instanceof BidModifierMobile) {
                return checkNotAllZeroForMobile(bidModifier, adjacentModifiers, checkForSmartTv);
            }
            if (bidModifier instanceof BidModifierDesktop) {
                return checkNotAllZeroForDesktop(bidModifier, adjacentModifiers, checkForSmartTv);
            }
            if (bidModifier instanceof BidModifierSmartTV) {
                return checkNotAllZeroForSmartTv(bidModifier, adjacentModifiers);
            }
            if (bidModifier instanceof BidModifierDesktopOnly) {
                return checkNotAllZeroForDesktopOnly(bidModifier, adjacentModifiers, checkForSmartTv);
            }
            if (bidModifier instanceof BidModifierTablet) {
                return checkNotAllZeroForTablet(bidModifier, adjacentModifiers, checkForSmartTv);
            }

            return ValidationResult.success(bidModifier);
        }, When.valueIs(m -> m instanceof BidModifierDesktop ||
                m instanceof BidModifierMobile ||
                m instanceof BidModifierSmartTV ||
                m instanceof BidModifierDesktopOnly ||
                m instanceof BidModifierTablet));
    }

    @NotNull
    private ValidationResult<BidModifier, Defect> checkNotAllZeroForMobile(
            BidModifier bidModifier,
            Collection<BidModifier> adjacentModifiers,
            boolean checkForSmartTv) {

        if (!BidModifierValidationMobileTypeSupport.getZeroValuePredicate().test(bidModifier)) {
            return ValidationResult.success(bidModifier);
        }

        for (var rules : BidModifierValidationMobileTypeSupport.getAdjacentRules(checkForSmartTv)) {
            var result = DeviceModifiersConflictChecker.checkPredicates(adjacentModifiers, rules);

            if (result == DeviceModifiersConflictChecker.CheckedPredicates.ALL_FAILED)  {
                return ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros());
            }
        }

        return ValidationResult.success(bidModifier);
    }

    @NotNull
    private ValidationResult<BidModifier, Defect> checkNotAllZeroForDesktop(
            BidModifier bidModifier,
            Collection<BidModifier> adjacentModifiers,
            boolean checkForSmartTv) {

        if (!BidModifierValidationDesktopTypeSupport.getZeroValuePredicate().test(bidModifier)) {
            return ValidationResult.success(bidModifier);
        }

        for (var rules : BidModifierValidationDesktopTypeSupport.getAdjacentRules(checkForSmartTv)) {
            var result = DeviceModifiersConflictChecker.checkPredicates(adjacentModifiers, rules);

            if (result == DeviceModifiersConflictChecker.CheckedPredicates.ALL_FAILED)  {
                return ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros());
            }
        }

        return ValidationResult.success(bidModifier);
    }

    @NotNull
    private ValidationResult<BidModifier, Defect> checkNotAllZeroForDesktopOnly(
            BidModifier bidModifier,
            Collection<BidModifier> adjacentModifiers,
            boolean checkForSmartTv) {

        if (!BidModifierValidationDesktopOnlyTypeSupport.getZeroValuePredicate().test(bidModifier)) {
            return ValidationResult.success(bidModifier);
        }

        for (var rules : BidModifierValidationDesktopOnlyTypeSupport.getAdjacentRules(checkForSmartTv)) {
            var result = DeviceModifiersConflictChecker.checkPredicates(adjacentModifiers, rules);

            if (result == DeviceModifiersConflictChecker.CheckedPredicates.ALL_FAILED)  {
                return ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros());
            }
        }

        return ValidationResult.success(bidModifier);
    }

    @NotNull
    private ValidationResult<BidModifier, Defect> checkNotAllZeroForTablet(
            BidModifier bidModifier,
            Collection<BidModifier> adjacentModifiers,
            boolean checkForSmartTv) {

        if (!BidModifierValidationTabletTypeSupport.getZeroValuePredicate().test(bidModifier)) {
            return ValidationResult.success(bidModifier);
        }

        for (var rules : BidModifierValidationTabletTypeSupport.getAdjacentRules(checkForSmartTv)) {
            var result = DeviceModifiersConflictChecker.checkPredicates(adjacentModifiers, rules);

            if (result == DeviceModifiersConflictChecker.CheckedPredicates.ALL_FAILED)  {
                return ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros());
            }
        }

        return ValidationResult.success(bidModifier);
    }

    @NotNull
    private ValidationResult<BidModifier, Defect> checkNotAllZeroForSmartTv(
            BidModifier bidModifier,
            Collection<BidModifier> adjacentModifiers) {

        if (!BidModifierValidationSmartTVTypeSupport.getZeroValuePredicate().test(bidModifier)) {
            return ValidationResult.success(bidModifier);
        }

        for (var rules : BidModifierValidationSmartTVTypeSupport.getAdjacentRules()) {
            var result = DeviceModifiersConflictChecker.checkPredicates(adjacentModifiers, rules);

            if (result == DeviceModifiersConflictChecker.CheckedPredicates.ALL_FAILED)  {
                return ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros());
            }
        }

        return ValidationResult.success(bidModifier);
    }

    private void checkInventoryAndBanerCorrectionNotAllZeroes(List<BidModifier> bidModifiers,
                                                              Multimap<Integer, Integer> complexToFlatIndexMap,
                                                              ListValidationBuilder<BidModifier, Defect> vb,
                                                              Map<Integer, Integer> flatToComplexIndexMap) {
        vb.checkEachBy((idx, bidModifier) -> {
            Integer complexIndex = flatToComplexIndexMap.get(idx);
            Collection<BidModifier> adjacentModifiers = StreamEx.of(complexToFlatIndexMap.get(complexIndex))
                    .map(bidModifiers::get)
                    .toList();
            if (bidModifier instanceof BidModifierBannerType) {
                BidModifierBannerType bidModifierBannerType = (BidModifierBannerType) bidModifier;
                if (!BidModifierValidationHelper.bannerTypePercentEqualsZero(bidModifierBannerType)) {
                    return ValidationResult.success(bidModifier);
                }
                boolean hasZeroInventory = StreamEx.of(adjacentModifiers)
                        .select(BidModifierInventory.class)
                        .findAny(BidModifierValidationHelper::isAllInventoryPercentEqualsZero)
                        .isPresent();
                return hasZeroInventory ?
                        ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros())
                        : ValidationResult.success(bidModifier);
            }
            if (bidModifier instanceof BidModifierInventory) {
                BidModifierInventory bidModifierInventory = (BidModifierInventory) bidModifier;
                if (!BidModifierValidationHelper.isAllInventoryPercentEqualsZero(bidModifierInventory)) {
                    return ValidationResult.success(bidModifier);
                }
                boolean hasZeroBannerType = StreamEx.of(adjacentModifiers)
                        .select(BidModifierBannerType.class)
                        .findAny(BidModifierValidationHelper::bannerTypePercentEqualsZero)
                        .isPresent();
                return hasZeroBannerType ?
                        ValidationResult.failed(bidModifier, deviceBidModifiersAllZeros())
                        : ValidationResult.success(bidModifier);
            }
            return ValidationResult.success(bidModifier);
        }, When.valueIs(m -> m instanceof BidModifierInventory || m instanceof BidModifierBannerType));
    }

    /**
     * Переносит "плоский" результат валидации на результат валидации комплексной модели.
     *
     * @param destListValidationResult     Приёмник
     * @param bidModifiersValidationResult Источник
     * @param complexToFlatIndexMap        Соответствие между индексами комплексных и плоских моделей
     */
    public void transferValidationResultFlatToComplex(
            ValidationResult<List<ComplexBidModifier>, Defect> destListValidationResult,
            ValidationResult<List<BidModifier>, Defect> bidModifiersValidationResult,
            Multimap<Integer, Integer> complexToFlatIndexMap) {
        List<ComplexBidModifier> complexBidModifiers = destListValidationResult.getValue();
        for (int complexIndex = 0; complexIndex < complexBidModifiers.size(); complexIndex++) {
            Collection<Integer> flatIndexes = complexToFlatIndexMap.get(complexIndex);
            if (flatIndexes.isEmpty()) {
                continue;
            }

            ComplexBidModifier complexBidModifier = complexBidModifiers.get(complexIndex);
            ValidationResult<ComplexBidModifier, Defect> destComplexValidationResult =
                    destListValidationResult.getOrCreateSubValidationResult(index(complexIndex), complexBidModifier);

            List<BidModifierExpression> expressModifierList = new ArrayList<>();
            List<ValidationResult<BidModifier, Defect>> expressModifierValidationResults = new ArrayList<>();
            for (int flatIndex : flatIndexes) {
                //noinspection unchecked
                ValidationResult<BidModifier, Defect> sourceValidationResult =
                        (ValidationResult<BidModifier, Defect>)
                                bidModifiersValidationResult.getSubResults().get(index(flatIndex));
                if (sourceValidationResult == null) {
                    continue;
                }

                BidModifier bidModifier = sourceValidationResult.getValue();
                if (bidModifier instanceof BidModifierExpression) {
                    // восстанавливаем список expressionModifiers опираясь на тот факт, что в плоском списке
                    // корректировки шли в том же порядке, как и в комплексной модели
                    // генерацию плоского списка смотри в convertFromComplexModels
                    expressModifierList.add((BidModifierExpression) bidModifier);
                    expressModifierValidationResults.add(sourceValidationResult);

                    // перенос результатов универсальных корректировок будет в отдельном цикле
                    continue;
                }
                String complexFieldName = getComplexModifierPropertyName(bidModifier);

                ValidationResult<BidModifier, Defect> destValidationResult =
                        destComplexValidationResult
                                .getOrCreateSubValidationResult(field(complexFieldName), bidModifier);
                transferIssuesFromValidationToValidation(sourceValidationResult, destValidationResult);
            }

            if (!expressModifierList.isEmpty()) {
                var expressModifierListResult = destComplexValidationResult
                        .getOrCreateSubValidationResult(ComplexBidModifier.EXPRESSION_MODIFIERS, expressModifierList);
                for (var i = 0; i < expressModifierList.size(); i++) {
                    BidModifierExpression bidModifier = expressModifierList.get(i);
                    var destValidationResult = expressModifierListResult
                            .getOrCreateSubValidationResult(index(i), bidModifier);
                    transferIssuesFromValidationToValidation(
                            expressModifierValidationResults.get(i),
                            destValidationResult
                    );
                }
            }
        }
    }

    private static String getComplexModifierPropertyName(BidModifier bidModifier) {
        final ModelProperty property;
        switch (bidModifier.getType()) {
            case MOBILE_MULTIPLIER:
                property = ComplexBidModifier.MOBILE_MODIFIER;
                break;

            case DESKTOP_MULTIPLIER:
                property = ComplexBidModifier.DESKTOP_MODIFIER;
                break;

            case TABLET_MULTIPLIER:
                property = ComplexBidModifier.TABLET_MODIFIER;
                break;

            case DESKTOP_ONLY_MULTIPLIER:
                property = ComplexBidModifier.DESKTOP_ONLY_MODIFIER;
                break;

            case SMARTTV_MULTIPLIER:
                property = ComplexBidModifier.SMART_T_V_MODIFIER;
                break;

            case VIDEO_MULTIPLIER:
                property = ComplexBidModifier.VIDEO_MODIFIER;
                break;

            case PERFORMANCE_TGO_MULTIPLIER:
                property = ComplexBidModifier.PERFORMANCE_TGO_MODIFIER;
                break;

            case DEMOGRAPHY_MULTIPLIER:
                property = ComplexBidModifier.DEMOGRAPHY_MODIFIER;
                break;

            case RETARGETING_MULTIPLIER:
                property = ComplexBidModifier.RETARGETING_MODIFIER;
                break;

            case GEO_MULTIPLIER:
                property = ComplexBidModifier.GEO_MODIFIER;
                break;

            case AB_SEGMENT_MULTIPLIER:
                property = ComplexBidModifier.AB_SEGMENT_MODIFIER;
                break;

            case BANNER_TYPE_MULTIPLIER:
                property = ComplexBidModifier.BANNER_TYPE_MODIFIER;
                break;

            case INVENTORY_MULTIPLIER:
                property = ComplexBidModifier.INVENTORY_MODIFIER;
                break;

            case WEATHER_MULTIPLIER:
                property = ComplexBidModifier.WEATHER_MODIFIER;
                break;

            case TRAFARET_POSITION_MULTIPLIER:
                property = ComplexBidModifier.TRAFARET_POSITION_MODIFIER;
                break;

            case RETARGETING_FILTER:
                property = ComplexBidModifier.RETARGETING_FILTER_MODIFIER;
                break;

            default:
                throw new IllegalStateException("Unknown type " + bidModifier.getType());
        }
        return property.name();
    }
}
