package ru.yandex.direct.core.entity.bidmodifiers.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.container.UntypedBidModifier;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.GEO_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.adjustmentSetNotFound;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.geoBidModifiersNotSupportedOnAdGroups;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.atLeastCampaignIdOrAdGroupIdRequired;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.possibleOnlyOneCampaignIdOrAdGroupIdField;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Component
@ParametersAreNonnullByDefault
public class ToggleBidModifiersValidationService {
    private static final List<String> CAMPAIGN_ADGROUP_FIELDS = ImmutableList.of("CampaignId", "AdGroupId");

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public ToggleBidModifiersValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    public ValidationResult<List<UntypedBidModifier>, Defect> validate(List<UntypedBidModifier> items,
                                                                       Set<Long> adGroupIds, Collection<Long> campaignIds, long operatorUid, ClientId clientId) {
        CampaignSubObjectAccessValidator campaignAccessValidator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds)
                .createValidator(CampaignAccessType.READ_WRITE);
        CampaignSubObjectAccessValidator adGroupAccessValidator = campaignSubObjectAccessCheckerFactory
                .newAdGroupChecker(operatorUid, clientId, adGroupIds)
                .createAdGroupValidator(CampaignAccessType.READ_WRITE);

        ListValidationBuilder<UntypedBidModifier, Defect> listBuilder = ListValidationBuilder.of(items);
        listBuilder.checkEachBy(item -> validateSingleItem(item, campaignAccessValidator, adGroupAccessValidator));
        return listBuilder.getResult();
    }

    private ValidationResult<UntypedBidModifier, Defect> validateSingleItem(UntypedBidModifier item,
                                                                            CampaignSubObjectAccessValidator campaignAccessValidator,
                                                                            CampaignSubObjectAccessValidator adGroupAccessValidator) {

        ModelItemValidationBuilder<UntypedBidModifier> ivb = ModelItemValidationBuilder.of(item);

        ivb.check(atLeastCampaignIdOrAdGroupIdRequired());
        ivb.check(possibleOnlyOneCampaignIdOrAdGroupIdField(), When.isValid());

        ivb.item(BidModifier.CAMPAIGN_ID)
                .check(validId(), When.isValidAnd(When.notNull()))
                .checkBy(campaignAccessValidator, When.isValidAnd(When.notNull()));

        ivb.item(BidModifier.AD_GROUP_ID)
                .check(validId(), When.isValidAnd(When.notNull()))
                .checkBy(adGroupAccessValidator, When.isValidAnd(When.notNull()));

        // На группах геокорректировки не поддерживаются
        ivb.check(fromPredicate(o -> o.getAdGroupId() == null || o.getType() != GEO_MULTIPLIER,
                geoBidModifiersNotSupportedOnAdGroups()), When.isValid());

        ivb.check(Constraint.fromPredicate(o -> o.getId() != null,
                adjustmentSetNotFound()), When.isValid());

        return ivb.getResult();
    }
}
