package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionParameter;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.expression.ParameterInfo;
import ru.yandex.direct.core.entity.bidmodifiers.expression.ParameterType;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimits;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression.EXPRESSION_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment.CONDITION;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;

/**
 * <pre>
 * Универсальные наборы корректировок, представляемые формулами, сохраняются в виде наборов adjustment'ов.
 * В каждом adjustment'e есть КНФ-формула из условий вида "параметр оператор константа", например, "level EQ 10".
 * То есть, отдельный adjustment определяет формулу вида
 * (level <= 10) && (city EQ "Москва" || city EQ "Рязань")
 * В нашем json это будет выглядеть примерно так:
 *   [
 *     [
 *       {"parameter": "level", "operation": "LE", "valueInteger": 10}
 *     ],
 *     [
 *       {"parameter": "city", "operation": "EQ", valueString: "Москва"},
 *       {"parameter": "city", "operation": "EQ", valueString: "Рязань"}
 *     ]
 *   ]
 * А все adjustment'ы (отдельные корректировки) обычно соединяются по "ИЛИ".
 * Для каждого adjustment'a можно установить значение корректировки (сколько процентов взять от исходной ставки).
 * При этом для любого набора значений параметров должен отработать максимум одна корректировка (adjustment).
 * Поэтому при валидации мы проверяем, что такого пересечения нет (аналогично демографическим корректировкам).
 * </pre>
 */
public abstract class AbstractBidModifierExpressionValidationTypeSupport implements BidModifierValidationTypeSupport<BidModifierExpression> {

    /**
     * Для конкретного типа корректировки возвращает набор параметров с информацией об ограничениях,
     * накладываемых на значения этих параметров в формулах.
     */
    protected Map<BidModifierExpressionParameter, ParameterInfo> getParametersInfo() {
        return Constants.ALL_PARAMETERS.get(getType());
    }

    @Override
    public ValidationResult<BidModifierExpression, Defect> validateAddStep1(BidModifierExpression modifier,
                                                                            CampaignType campaignType,
                                                                            AdGroup adGroupWithType,
                                                                            ClientId clientId,
                                                                            CachingFeaturesProvider featuresProvider) {
        var vb = ModelItemValidationBuilder.of(modifier);

        // Общие проверки содержимого
        vb.item(EXPRESSION_ADJUSTMENTS).checkBy(adjustments ->
                        validateAdjustmentsCommon(adjustments, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider,
                                BidModifiersDefectIds.Number.TOO_MANY_EXPRESSION_CONDITIONS),
                When.isValid());

        // Сначала проверки структуры корректировок по отдельности
        vb.list(EXPRESSION_ADJUSTMENTS).check(notEmptyCollection());
        vb.list(EXPRESSION_ADJUSTMENTS).checkEachBy(this::validateSingleAdjustment);

        // Проверки корректировок в целом: не должно быть пересечений по значениям true
        // (не должно быть таких наборов значений параметров, при которых применяется более одной корректировки)
        vb.list(EXPRESSION_ADJUSTMENTS)
                .check(fromPredicate(this::hasNoIntersection,
                        BidModifiersDefects.expressionConditionsIntersection()),
                        When.isValid()
                );

        return vb.getResult();
    }

    private ValidationResult<BidModifierExpressionAdjustment, Defect> validateSingleAdjustment(
            int index, BidModifierExpressionAdjustment adjustment) {
        var vb = ModelItemValidationBuilder.of(adjustment);

        // Выражение не может быть пустым
        vb.list(CONDITION).check(notEmptyCollection());

        vb.list(CONDITION).checkEachBy((idx, subCondition) -> {
            ListValidationBuilder<BidModifierExpressionLiteral, Defect> lb = ListValidationBuilder.of(subCondition);

            // Подвыражение не может быть пустым
            lb.check(notEmptyCollection());

            // Проверка отдельного элементарного выражения
            lb.checkEach(fromPredicate(
                    literal -> isLiteralWellFormed(literal, getParametersInfo()),
                    // Пока один дефект на все проблемы с условием
                    BidModifiersDefects.invalidExpressionLiteral())
            );

            return lb.getResult();
        });

        // Если выражение принимает всегда одно и то же значение -- считаем это некорректной корректировкой
        vb.list(CONDITION)
                .check(fromPredicate(condition -> !isConstant(condition),
                        BidModifiersDefects.expressionAdjustmentIsConstantForAnyValues()), When.isValid());

        return vb.getResult();
    }

    /**
     * Проверяет корректность элементарного выражения с учётом ограничений на параметры.
     */
    boolean isLiteralWellFormed(BidModifierExpressionLiteral literal,
                                Map<BidModifierExpressionParameter, ParameterInfo> allParameterInfos) {
        if ((literal.getValueInteger() == null) == (literal.getValueString() == null)) {
            // Должно быть определено только одно из значений: либо integer, либо string
            return false;
        }

        // Параметр должен быть указан в parameterInfos для этого типа корректировки
        if (!allParameterInfos.containsKey(literal.getParameter())) {
            return false;
        }

        // Тип параметра и значение должны быть согласованы
        var parameterInfo = allParameterInfos.get(literal.getParameter());
        switch (parameterInfo.getType()) {
            case INTEGER: {
                if (literal.getValueInteger() == null) {
                    return false;
                }
                if (parameterInfo.getMin() != null && literal.getValueInteger() < parameterInfo.getMin()) {
                    return false;
                }
                if (parameterInfo.getMax() != null && literal.getValueInteger() > parameterInfo.getMax()) {
                    return false;
                }
                break;
            }
            case STRING: {
                if (literal.getValueString() == null) {
                    return false;
                }
                if (parameterInfo.getAllowedValues() != null) {
                    if (!parameterInfo.getAllowedValues().contains(literal.getValueString())) {
                        return false;
                    }
                }
                break;
            }
            case ENUM: {
                if (literal.getValueString() == null) {
                    return false;
                }
                if (!parameterInfo.getAllowedValues().contains(literal.getValueString())) {
                    return false;
                }
                break;
            }
            default:
                throw new AssertionError("Unknown type");
        }

        // Операция тоже должна быть из списка допустимых
        if (!parameterInfo.getAllowedOperators().contains(literal.getOperation())) {
            return false;
        }

        return true;
    }

    /**
     * Проверяет, не является ли переданное условие константным,
     * то есть принимающим одно и то же значение на всём множестве входных значений параметров.
     */
    private boolean isConstant(List<List<BidModifierExpressionLiteral>> condition) {
        try (TraceProfile profile = Trace.current().profile("bidmodifiers:expression_is_constant")) {
            Map<BidModifierExpressionParameter, Set<Object>> values = new HashMap<>();

            // Находим все возможные значения параметров, на которых формула может менять значение
            addPossibleValues(values, condition);

            // Дополнительно для каждого параметра типа STRING добавляем значение, которое не равно
            // ни одной из указанных в формуле констант. Для ENUM и INTEGER этого делать не требуется,
            // а вот для STRING -- необходимо проверить, как формула отреагирует на такое значение
            addUnexistingStringValues(values);

            Set<Boolean> results = new HashSet<>();
            testAllCombinations(values, valuesCombination -> {
                boolean result = evaluateCondition(condition, valuesCombination);
                results.add(result);
            });
            checkState(!results.isEmpty());

            return results.size() == 1;
        }
    }

    private boolean hasNoIntersection(List<BidModifierExpressionAdjustment> adjustments) {
        try (TraceProfile profile = Trace.current().profile("bidmodifiers:expression_has_no_intersection")) {
            Map<BidModifierExpressionParameter, Set<Object>> values = new HashMap<>();

            // Находим все возможные значения параметров, на которых формулы могут менять значение
            for (var adjustment : adjustments) {
                addPossibleValues(values, adjustment.getCondition());
            }

            // Дополнительно для каждого параметра типа STRING добавляем значение, которое не равно
            // ни одной из указанных в формуле констант. Для ENUM и INTEGER этого делать не требуется,
            // а вот для STRING -- необходимо проверить, как формула отреагирует на такое значение
            addUnexistingStringValues(values);

            List<Map<BidModifierExpressionParameter, Object>> failCombinations = new ArrayList<>();
            testAllCombinations(values, valuesCombination -> {
                // Из набора корректировок максимум одна должна вернуть true
                boolean foundTrue = false;
                for (var adjustment : adjustments) {
                    var condition = adjustment.getCondition();
                    boolean result = evaluateCondition(condition, valuesCombination);
                    if (result) {
                        if (foundTrue) {
                            if (failCombinations.isEmpty()) {
                                failCombinations.add(new HashMap<>(valuesCombination));
                            }
                        } else {
                            foundTrue = true;
                        }
                    }
                }
            });
            return failCombinations.isEmpty();
        }
    }

    private void addPossibleValues(Map<BidModifierExpressionParameter, Set<Object>> values,
                                   List<List<BidModifierExpressionLiteral>> condition) {
        var allParameterInfos = getParametersInfo();
        for (List<BidModifierExpressionLiteral> subCondition : condition) {
            for (BidModifierExpressionLiteral literal : subCondition) {
                var parameter = literal.getParameter();
                var parameterInfo = allParameterInfos.get(parameter);
                //
                values.putIfAbsent(parameter, new HashSet<>());
                Set<Object> paramValues = values.get(parameter);
                switch (parameterInfo.getType()) {
                    case INTEGER: {
                        // Каждое значение разбивает отрезок допустимых значений на три: меньше, равно и больше.
                        // Поэтому для проверки всех случаев нужно добавить по одному значению из каждого подмножества.
                        paramValues.add(literal.getValueInteger());
                        if (parameterInfo.getMin() == null || literal.getValueInteger() - 1 >= parameterInfo.getMin()) {
                            paramValues.add(literal.getValueInteger() - 1);
                        }
                        if (parameterInfo.getMax() == null || literal.getValueInteger() + 1 <= parameterInfo.getMax()) {
                            paramValues.add(literal.getValueInteger() + 1);
                        }
                        break;
                    }
                    case ENUM: {
                        checkNotNull(parameterInfo.getAllowedValues());
                        paramValues.addAll(parameterInfo.getAllowedValues());
                        break;
                    }
                    case STRING: {
                        paramValues.add(literal.getValueString());
                        break;
                    }
                    default: {
                        throw new AssertionError("Not supported");
                    }
                }
            }
        }
    }

    private void addUnexistingStringValues(Map<BidModifierExpressionParameter, Set<Object>> values) {
        // Дополнительно для каждого параметра типа STRING добавляем значение, которое не равно
        // ни одной из указанных в формуле констант. Для ENUM и INTEGER этого делать не требуется,
        // а вот для STRING -- необходимо проверить, как формула отреагирует на такое значение
        var allParameterInfos = getParametersInfo();
        for (BidModifierExpressionParameter parameter : values.keySet()) {
            var parameterInfo = allParameterInfos.get(parameter);
            if (parameterInfo.getType() == ParameterType.STRING) {
                Set<Object> set = values.get(parameter);
                set.add(generateUnexistingString((Set) set));
            }
        }
    }

    private boolean evaluateCondition(List<List<BidModifierExpressionLiteral>> condition,
                                      Map<BidModifierExpressionParameter, Object> valuesCombination) {
        boolean result = true;
        for (List<BidModifierExpressionLiteral> subCondition : condition) {
            boolean subResult = false;
            for (BidModifierExpressionLiteral literal : subCondition) {
                subResult = subResult || evaluateLiteral(literal, valuesCombination);
                if (subResult) {
                    break;
                }
            }
            result = result && subResult;
            if (!result) {
                break;
            }
        }
        return result;
    }

    /**
     * Вычисляет результат элементарного выражения
     */
    private boolean evaluateLiteral(BidModifierExpressionLiteral literal,
                                    Map<BidModifierExpressionParameter, Object> valuesCombination) {
        var parameterInfo = getParametersInfo().get(literal.getParameter());
        switch (parameterInfo.getType()) {
            case INTEGER: {
                return evaluateIntegerLiteral(literal, valuesCombination);
            }
            case ENUM:
            case STRING: {
                return evaluateNonIntegerLiteral(literal, valuesCombination);
            }
            default: {
                throw new AssertionError("Unknown type");
            }
        }
    }

    private boolean evaluateIntegerLiteral(BidModifierExpressionLiteral literal,
                                           Map<BidModifierExpressionParameter, Object> valuesCombination) {
        int value = (Integer) valuesCombination.get(literal.getParameter());
        int constant = literal.getValueInteger();
        switch (literal.getOperation()) {
            case EQ:
            case MATCH_GOAL_CONTEXT: {
                return value == constant;
            }
            case LE: {
                return value <= constant;
            }
            case GE: {
                return value >= constant;
            }
            case LT: {
                return value < constant;
            }
            case GT: {
                return value > constant;
            }
            default: {
                throw new AssertionError("Unknown operation");
            }
        }
    }

    private boolean evaluateNonIntegerLiteral(BidModifierExpressionLiteral literal,
                                              Map<BidModifierExpressionParameter, Object> valuesCombination) {
        String value = (String) valuesCombination.get(literal.getParameter());
        String constant = literal.getValueString();
        switch (literal.getOperation()) {
            case EQ: {
                return Objects.equals(value, constant);
            }
            case LE: {
                throw new IllegalStateException("LE is not supported for strings and enums");
            }
            case GE: {
                throw new IllegalStateException("GE is not supported for strings and enums");
            }
            case LT: {
                throw new IllegalStateException("LT is not supported for strings and enums");
            }
            case GT: {
                throw new IllegalStateException("GT is not supported for strings and enums");
            }
            case MATCH_GOAL_CONTEXT: {
                throw new IllegalStateException("MATCH_GOAL_CONTEXT is not supported for strings and enums");
            }
            default: {
                throw new AssertionError("Unknown operation");
            }
        }
    }

    /**
     * Перебирает все комбинации значений параметров и найденные комбинации подаёт на вход consumer'у.
     */
    private void testAllCombinations(Map<BidModifierExpressionParameter, Set<Object>> values,
                                     Consumer<Map<BidModifierExpressionParameter, Object>> consumer) {
        List<BidModifierExpressionParameter> allParameters = new ArrayList<>(values.keySet());
        List<List<Object>> allParameterValues = allParameters.stream()
                .map(parameter -> new ArrayList<>(values.get(parameter)))
                .collect(Collectors.toList());
        findCombination(allParameters, allParameterValues, 0, new HashMap<>(), consumer);
    }

    /**
     * Просто рекурсивная функция для перебора комбинаций.
     */
    private void findCombination(List<BidModifierExpressionParameter> allParameters,
                                 List<List<Object>> allParameterValues,
                                 int parameterIndex,
                                 Map<BidModifierExpressionParameter, Object> valuesMap,
                                 Consumer<Map<BidModifierExpressionParameter, Object>> consumer) {
        if (parameterIndex >= allParameters.size()) {
            // Получен полный набор параметров
            consumer.accept(valuesMap);
            return;
        }
        for (Object value : allParameterValues.get(parameterIndex)) {
            valuesMap.put(allParameters.get(parameterIndex), value);
            findCombination(allParameters, allParameterValues, parameterIndex + 1, valuesMap, consumer);
            valuesMap.remove(allParameters.get(parameterIndex));
        }
    }

    /**
     * Генерирует строку, которая отсутствует во множестве set.
     */
    private String generateUnexistingString(Set<String> set) {
        String s = "UNEXISTING";

        for (int i = 1; set.contains(s); i++) {
            s = String.format("UNEXISTING_%d", i);
        }

        return s;
    }

    @Override
    public ValidationResult<BidModifierExpression, Defect> validateAddStep2(BidModifierExpression modifier,
                                                                            BidModifierExpression existingModifier,
                                                                            CampaignType campaignType,
                                                                            @Nullable AdGroup adGroupWithType,
                                                                            ClientId clientId,
                                                                            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierExpression> vb = ModelItemValidationBuilder.of(modifier);

        List<BidModifierExpressionAdjustment> existingAdjustments = existingModifier.getExpressionAdjustments();

        // Общие проверки содержимого
        vb.item(EXPRESSION_ADJUSTMENTS).check(fromPredicate(adjustments -> {
            List<BidModifierExpressionAdjustment> allAdjustments = new ArrayList<>(adjustments);
            allAdjustments.addAll(existingAdjustments);

            BidModifierLimits limits = BidModifierLimitsAdvanced.getLimits(
                    getType(), campaignType, adGroupWithType, clientId, featuresProvider);
            return limits.maxConditions == null || allAdjustments.size() <= limits.maxConditions;
        }, new Defect<>(BidModifiersDefectIds.Number.TOO_MANY_EXPRESSION_CONDITIONS)));

        // Проверки объединённого набора корректировок в целом: не должно быть пересечений по значениям true
        // (не должно быть таких наборов значений параметров, при которых применяется более одной корректировки)
        vb.list(EXPRESSION_ADJUSTMENTS)
                .check(fromPredicate(adjustments -> {
                            List<BidModifierExpressionAdjustment> allAdjustments = new ArrayList<>(adjustments);
                            allAdjustments.addAll(existingAdjustments);
                            return hasNoIntersection(allAdjustments);
                        },
                        BidModifiersDefects.expressionConditionsIntersection()
                ));

        return vb.getResult();
    }
}
