package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting;
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting.RETARGETING_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.duplicateRetargetingCondition;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.unsupportedRetargetingType;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

public abstract class AbstractBidModifierValidationRetargetingTypeSupport
        implements BidModifierValidationTypeSupport<AbstractBidModifierRetargeting> {

    protected final ShardHelper shardHelper;
    protected final RetargetingConditionRepository retargetingConditionRepository;

    protected
    AbstractBidModifierValidationRetargetingTypeSupport(ShardHelper shardHelper,
                                                        RetargetingConditionRepository retargetingConditionRepository) {
        this.shardHelper = shardHelper;
        this.retargetingConditionRepository = retargetingConditionRepository;
    }

    @Override
    public ValidationResult<AbstractBidModifierRetargeting, Defect> validateAddStep1(
            AbstractBidModifierRetargeting modifier, CampaignType campaignType, AdGroup adGroupWithType,
            ClientId clientId, CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<AbstractBidModifierRetargeting> vb = ModelItemValidationBuilder.of(modifier);

        // Сначала проверки структурности
        // Нельзя повторно использовать один и тот же RetargetingConditionId в разных adjustment'ах
        vb.list(RETARGETING_ADJUSTMENTS)
                .checkEachBy(this::validateRetargetingAdjustmentSchema, When.notNull());
        // TODO : попробовать использовать unique
        vb.check(fromPredicate(o -> StreamEx.of(o.getRetargetingAdjustments()).map(
                        AbstractBidModifierRetargetingAdjustment::getRetargetingConditionId).distinct(2).count() == 0,
                duplicateRetargetingCondition()));

        // Общие проверки
        vb.item(RETARGETING_ADJUSTMENTS).checkBy(adjustments ->
                        BidModifierValidationHelper.validateAdjustmentsCommon(
                                adjustments, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider,
                                BidModifiersDefectIds.Number.TOO_MANY_RETARGETING_CONDITIONS),
                When.isTrue(!vb.getResult().hasAnyErrors()));

        // Затем типоспецифичные
        vb.item(RETARGETING_ADJUSTMENTS).checkBy(adjustments ->
                        validateRetargetingAdjustments(adjustments, clientId),
                When.isTrue(!vb.getResult().hasAnyErrors()));

        return vb.getResult();
    }

    @Override
    public ValidationResult<AbstractBidModifierRetargeting, Defect> validateAddStep2(AbstractBidModifierRetargeting modifier,
                                                                                     AbstractBidModifierRetargeting existingModifier,
                                                                                     CampaignType campaignType,
                                                                                     @Nullable AdGroup adGroupWithType,
                                                                                     ClientId clientId,
                                                                                     CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<AbstractBidModifierRetargeting> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(RETARGETING_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<AbstractBidModifierRetargetingAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<AbstractBidModifierRetargetingAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getRetargetingAdjustments().stream())
                            .collect(toList());

                    // Общие проверки совместного списка старых+новых
                    ValidationResult<List<AbstractBidModifierRetargetingAdjustment>, Defect>
                            commonChecksValidationResult =
                            validateAdjustmentsCommon(allToValidate, getType(), campaignType,
                                    adGroupWithType, clientId, featuresProvider,
                                    BidModifiersDefectIds.Number.TOO_MANY_RETARGETING_CONDITIONS);
                    if (commonChecksValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(commonChecksValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<AbstractBidModifierRetargetingAdjustment>, Defect>
                            specificValidationResult =
                            validateRetargetingAdjustments(allToValidate, clientId);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    private ValidationResult<AbstractBidModifierRetargetingAdjustment, Defect> validateRetargetingAdjustmentSchema(
            AbstractBidModifierRetargetingAdjustment adjustment) {
        ModelItemValidationBuilder<AbstractBidModifierRetargetingAdjustment> vb =
                ModelItemValidationBuilder.of(adjustment);
        vb.item(BidModifierRetargetingAdjustment.RETARGETING_CONDITION_ID).check(validId());
        return vb.getResult();
    }

    private ValidationResult<List<AbstractBidModifierRetargetingAdjustment>, Defect> validateRetargetingAdjustments(
            List<AbstractBidModifierRetargetingAdjustment> retargetingAdjustments, ClientId clientId) {
        ListValidationBuilder<AbstractBidModifierRetargetingAdjustment, Defect> builder =
                ListValidationBuilder.of(retargetingAdjustments);
        builder.checkBy(this::validateRetargetingsIntersection);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // Проверяем, что все условия ретаргетинга существуют и доступны клиенту
        Map<Long, RetargetingCondition> clientRetargetingConditionsMap =
                retargetingConditionRepository.getFromRetargetingConditionsTable(shard,
                        clientId,
                        retargetingAdjustments.stream()
                                .map(AbstractBidModifierRetargetingAdjustment::getRetargetingConditionId)
                                .collect(toSet())).stream().collect(toMap(RetargetingCondition::getId, identity()));
        builder.checkEachBy(adjustment -> {
            ItemValidationBuilder<AbstractBidModifierRetargetingAdjustment, Defect> vb =
                    ItemValidationBuilder.of(adjustment);
            long retCondId = adjustment.getRetargetingConditionId();
            RetargetingCondition retargetingCondition =
                    clientRetargetingConditionsMap.get(retCondId);
            vb.check(fromPredicate(it -> null != retargetingCondition && !retargetingCondition.getDeleted(),
                    RetargetingDefects.retargetingConditionNotFoundDetailed(retCondId)));
            vb.check(fromPredicate(it -> retargetingCondition.getType() == ConditionType.metrika_goals
                            || retargetingCondition.getType() == ConditionType.shortcuts,
                    unsupportedRetargetingType()), When.isValid());
            return vb.getResult();
        });

        return builder.getResult();
    }

    /**
     * Проверяет то, что каждый регион в наборе встречается не более одного раза.
     */
    private ValidationResult<List<AbstractBidModifierRetargetingAdjustment>, Defect> validateRetargetingsIntersection(
            List<AbstractBidModifierRetargetingAdjustment> retargetingAdjustments) {
        ListValidationBuilder<AbstractBidModifierRetargetingAdjustment, Defect> builder =
                ListValidationBuilder.of(retargetingAdjustments);
        builder.check(fromPredicate(adjustments -> adjustments.stream()
                        .collect(groupingBy(
                                AbstractBidModifierRetargetingAdjustment::getRetargetingConditionId,
                                counting()))
                        .values().stream().allMatch(count -> count == 1),
                BidModifiersDefects.retargetingConditionAlreadyExists()));
        return builder.getResult();
    }
}
