package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegmentAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment.AB_SEGMENT_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.abSegmentBidModifiersNotSupportedOnAdGroups;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationABSegmentsTypeSupport implements BidModifierValidationTypeSupport<BidModifierABSegment> {
    private final RetargetingConditionService retargetingConditionService;

    @Override
    public BidModifierType getType() {
        return BidModifierType.AB_SEGMENT_MULTIPLIER;
    }

    @Autowired
    public BidModifierValidationABSegmentsTypeSupport(
            RetargetingConditionService retargetingConditionService) {
        this.retargetingConditionService = retargetingConditionService;
    }

    @Override
    public ValidationResult<BidModifierABSegment, Defect> validateAddStep1(
            BidModifierABSegment modifier, CampaignType campaignType, AdGroup adGroupWithType, ClientId clientId,
            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierABSegment> vb = ModelItemValidationBuilder.of(modifier);

        // Сначала проверяем, можно ли задавать эту корректировку для группы
        vb.item(BidModifier.AD_GROUP_ID)
                .check(isNull(), abSegmentBidModifiersNotSupportedOnAdGroups());

        // Проверки структурности
        // Нельзя повторно использовать один и тот же RetargetingConditionId в разных adjustment'ах
        vb.list(AB_SEGMENT_ADJUSTMENTS)
                .checkEachBy(this::validateABSegmentsAdjustmentSchema, When.notNull());

        // Общие проверки
        vb.list(AB_SEGMENT_ADJUSTMENTS).checkEachBy(adjustment ->
                BidModifierValidationHelper.validateAdjustmentCommon(
                        adjustment, getType(), campaignType, adGroupWithType, clientId, featuresProvider));

        // Затем типоспецифичные
        vb.item(AB_SEGMENT_ADJUSTMENTS).checkBy(adjustments ->
                        validateABSegmentsAdjustments(adjustments, clientId),
                When.isValid());

        return vb.getResult();
    }

    @Override
    public ValidationResult<BidModifierABSegment, Defect> validateAddStep2(BidModifierABSegment modifier,
                                                                           BidModifierABSegment existingModifier,
                                                                           CampaignType campaignType,
                                                                           @Nullable AdGroup adGroupWithType,
                                                                           ClientId clientId,
                                                                           CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierABSegment> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(AB_SEGMENT_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<BidModifierABSegmentAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<BidModifierABSegmentAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getAbSegmentAdjustments().stream())
                            .collect(toList());

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierABSegmentAdjustment>, Defect>
                            specificValidationResult =
                            validateABSegmentsAdjustments(allToValidate, clientId);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    private ValidationResult<BidModifierABSegmentAdjustment, Defect> validateABSegmentsAdjustmentSchema(
            BidModifierABSegmentAdjustment adjustment) {
        ModelItemValidationBuilder<BidModifierABSegmentAdjustment> vb =
                ModelItemValidationBuilder.of(adjustment);
        vb.item(BidModifierABSegmentAdjustment.SECTION_ID).check(validId());
        vb.item(BidModifierABSegmentAdjustment.SEGMENT_ID).check(validId());
        return vb.getResult();
    }

    private ValidationResult<List<BidModifierABSegmentAdjustment>, Defect> validateABSegmentsAdjustments(
            List<BidModifierABSegmentAdjustment> abSegmentsAdjustments, ClientId clientId) {
        ListValidationBuilder<BidModifierABSegmentAdjustment, Defect> builder =
                ListValidationBuilder.of(abSegmentsAdjustments);

        Map<Long, Goal> metrikaGoalsHash =
                listToMap(retargetingConditionService.getAvailableMetrikaGoalsForRetargeting(clientId,
                        GoalType.AB_SEGMENT),
                        Goal::getId);

        builder.checkBy(v -> validateSegmentUsage(metrikaGoalsHash, v));

        return builder.getResult();
    }

    private ValidationResult<List<BidModifierABSegmentAdjustment>, Defect> validateSegmentUsage(Map<Long, Goal> metrikaGoalsHash,
                                                                                                List<BidModifierABSegmentAdjustment> abSegmentsAdjustments) {
        ListValidationBuilder<BidModifierABSegmentAdjustment, Defect> builder =
                ListValidationBuilder.of(abSegmentsAdjustments);

        builder.checkEachBy(adjustment -> {
            ItemValidationBuilder<BidModifierABSegmentAdjustment, Defect> vb =
                    ItemValidationBuilder.of(adjustment);

            Long segmentId = adjustment.getSegmentId();
            Goal goal = metrikaGoalsHash.get(segmentId);

            vb.check(fromPredicate(it -> goal != null,
                    BidModifiersDefects.abSegmentNotFound(segmentId)
            ));
            vb.check(fromPredicate(it -> Objects.equals(goal.getSectionId(), adjustment.getSectionId()),
                    BidModifiersDefects.abSegmentSectionNotFound(segmentId)
            ), When.isValid());

            return vb.getResult();
        });

        // Если количество уникальных segmentId не равно количеству корректировок - значит какие-то из них
        // повторялись, то есть пересечение
        Set<Long> uniqSegmentIds = listToSet(abSegmentsAdjustments, BidModifierABSegmentAdjustment::getSegmentId);
        builder.check(fromPredicate(it -> uniqSegmentIds.size() == abSegmentsAdjustments.size(),
                BidModifiersDefects.abSegmentIntersection()));

        return builder.getResult();

    }
}
