package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType.BANNER_TYPE_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.INVENTORY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.isAllInventoryPercentEqualsZero;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.setDeviceBidModifierAllZerosDefect;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.validateModifierDoNotRewriteExisting;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationBannerTypeSupport implements BidModifierValidationTypeSupport<BidModifierBannerType> {
    private final DeviceModifiersConflictChecker deviceModifiersConflictChecker;

    public BidModifierValidationBannerTypeSupport(DeviceModifiersConflictChecker deviceModifiersConflictChecker) {
        this.deviceModifiersConflictChecker = deviceModifiersConflictChecker;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.BANNER_TYPE_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierBannerType, Defect> validateAddStep1(BidModifierBannerType modifier,
                                                                            CampaignType campaignType,
                                                                            AdGroup adGroupWithType,
                                                                            ClientId clientId,
                                                                            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierBannerType> vb = ModelItemValidationBuilder.of(modifier);
        vb.item(BANNER_TYPE_ADJUSTMENTS)
                .checkBy(adjustments -> validateAdjustmentsCommon(adjustments, getType(), campaignType,
                        adGroupWithType, clientId, featuresProvider, null));
        vb.list(BANNER_TYPE_ADJUSTMENTS)
                .checkEach(unique(BidModifierBannerTypeAdjustment::getBannerType),
                        new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_ADJUSTMENT), When.isValid());
        return vb.getResult();
    }

    @Override
    public ValidationResult<List<BidModifierBannerType>, Defect> validateAddStep2(
            ClientId clientId,
            List<BidModifierBannerType> modifiersToValidate,
            Map<BidModifierKey, BidModifierBannerType> existingModifiers,
            Map<Long, CampaignType> campaignTypes,
            Map<Long, AdGroup> adGroupsWithType,
            CachingFeaturesProvider featuresProvider,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {
        // Запрещено добавлять одновременно корректировки типа баннера и инвентаря с нулевым множителем
        // т.к. их совместное действие перекроет весь трафик
        Set<BidModifierKey> keysWithConflict =
                deviceModifiersConflictChecker.findConflictingModifiers(
                        clientId, modifiersToValidate, allValidBidModifiersInOperation,
                        BidModifierValidationHelper::bannerTypePercentEqualsZero,
                        Map.of(INVENTORY_MULTIPLIER, m -> isAllInventoryPercentEqualsZero((BidModifierInventory) m)));

        ListValidationBuilder<BidModifierBannerType, Defect> lvb = ListValidationBuilder.of(modifiersToValidate);
        lvb.checkEachBy(modifier ->
                validateModifierDoNotRewriteExisting(modifier, existingModifiers, BANNER_TYPE_ADJUSTMENTS));
        lvb.checkEach(setDeviceBidModifierAllZerosDefect(keysWithConflict));
        return lvb.getResult();
    }
}
