package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics.DEMOGRAPHICS_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_AGES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.ALL_GENDERS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.OLD_AGE_45_EXPANSION;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.demographicsConditionsIntersection;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationDemographicsTypeSupport implements BidModifierValidationTypeSupport<BidModifierDemographics> {
    @Override
    public BidModifierType getType() {
        return BidModifierType.DEMOGRAPHY_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierDemographics, Defect> validateAddStep1(
            BidModifierDemographics modifier,
            CampaignType campaignType, @Nullable AdGroup adGroupWithType, ClientId clientId,
            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierDemographics> vb = ModelItemValidationBuilder.of(modifier);

        // Сначала проверки структуры
        vb.list(DEMOGRAPHICS_ADJUSTMENTS)
                .checkEachBy(this::validateDemographicsAdjustmentSchema);

        // Общие проверки содержимого
        vb.item(DEMOGRAPHICS_ADJUSTMENTS).checkBy(adjustments ->
                        validateAdjustmentsCommon(adjustments, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider,
                                BidModifiersDefectIds.Number.TOO_MANY_DEMOGRAPHY_CONDITIONS),
                When.isValid());

        // Затем типоспецифичные
        vb.item(DEMOGRAPHICS_ADJUSTMENTS)
                .checkBy(this::validateDemographicsAdjustments, When.isValid());

        return vb.getResult();
    }

    @Override
    public ValidationResult<BidModifierDemographics, Defect> validateAddStep2(
            BidModifierDemographics modifier,
            BidModifierDemographics existingModifier, CampaignType campaignType,
            @Nullable AdGroup adGroupWithType, ClientId clientId, CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierDemographics> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(DEMOGRAPHICS_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<BidModifierDemographicsAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<BidModifierDemographicsAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getDemographicsAdjustments().stream())
                            .collect(toList());

                    // Общие проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierDemographicsAdjustment>, Defect>
                            commonChecksValidationResult =
                            validateAdjustmentsCommon(allToValidate, getType(), campaignType,
                                    adGroupWithType, clientId, featuresProvider,
                                    BidModifiersDefectIds.Number.TOO_MANY_DEMOGRAPHY_CONDITIONS);
                    if (commonChecksValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(commonChecksValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierDemographicsAdjustment>, Defect>
                            specificValidationResult =
                            validateDemographicsAdjustments(allToValidate);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    private ValidationResult<BidModifierDemographicsAdjustment, Defect> validateDemographicsAdjustmentSchema(
            BidModifierDemographicsAdjustment adjustment) {
        return ItemValidationBuilder.<BidModifierDemographicsAdjustment, Defect>
                of(adjustment).check(
                fromPredicate(
                        adj -> adj.getAge() != null || adj.getGender() != null,
                        BidModifiersDefects.requiredAtLeastOneOfAgeOrGender())
        ).getResult();
    }

    private ValidationResult<List<BidModifierDemographicsAdjustment>, Defect> validateDemographicsAdjustments(
            List<BidModifierDemographicsAdjustment> adjustments) {
        ListValidationBuilder<BidModifierDemographicsAdjustment, Defect> listValidationBuilder =
                ListValidationBuilder.of(adjustments);

        // Проверяем, не пересекаются ли условия корректировок в наборе
        Map<Pair<GenderType, AgeType>, Long> effectiveGenderAgeUsages = adjustments.stream()
                .map(adjustment -> Pair.of(adjustment.getGender(), adjustment.getAge()))
                .flatMap(pair -> {
                    // Вычисляем набор комбинаций, которые покрывает текущая пара пол+возраст
                    GenderType genderType = pair.getLeft();
                    AgeType ageType = pair.getRight();
                    Set<GenderType> genders = genderType != null ? singleton(genderType) : ALL_GENDERS;
                    Set<AgeType> ages = expandAgeType(ageType);
                    Set<Pair<GenderType, AgeType>> pairs = new HashSet<>();
                    for (GenderType gender : genders) {
                        for (AgeType age : ages) {
                            pairs.add(Pair.of(gender, age));
                        }
                    }
                    return pairs.stream();
                    // Считаем их количество
                }).collect(groupingBy(identity(), counting()));

        // Если хотя бы 1 комбинация используется дважды -- ругаемся
        if (effectiveGenderAgeUsages.values().stream().anyMatch(count -> count > 1)) {
            listValidationBuilder.getResult().addError(demographicsConditionsIntersection());
        }

        return listValidationBuilder.getResult();
    }

    private Set<AgeType> expandAgeType(@Nullable AgeType ageType) {
        if (ageType == null) {
            return ALL_AGES;
        }
        // Если передан старый диапазон "45+", для валидации раскрываем его как ["45-54", "55+"]
        if (ageType == AgeType._45_) {
            return OLD_AGE_45_EXPANSION;
        }
        return singleton(ageType);
    }
}
