package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DESKTOP_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DESKTOP_ONLY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.TABLET_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentCommon;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationSmartTVTypeSupport.isSmartTvEnabled;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.setConflictingModifiersDefect;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.setDeviceBidModifierAllZerosDefect;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.validateModifierDoNotRewriteExisting;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationDesktopTypeSupport implements BidModifierValidationTypeSupport<BidModifierDesktop> {
    private final DeviceModifiersConflictChecker deviceModifiersConflictChecker;

    public static Predicate<BidModifier> getZeroValuePredicate() {
        return m -> ((BidModifierDesktop) m).getDesktopAdjustment().getPercent().equals(0);
    }

    public static void fillZeroValuePredicate(Map<BidModifierType, Predicate<BidModifier>> typeRules) {
        typeRules.put(DESKTOP_MULTIPLIER, getZeroValuePredicate());
    }

    public static Set<Map<BidModifierType, Predicate<BidModifier>>> getAdjacentRules(
            boolean checkForSmartTv
    ) {
        var rules = new HashMap<BidModifierType, Predicate<BidModifier>>();
        BidModifierValidationMobileTypeSupport.fillZeroValuePredicate(rules);
        if (checkForSmartTv) {
            BidModifierValidationSmartTVTypeSupport.fillZeroValuePredicate(rules);
        }

        return Set.of(rules);
    }

    public BidModifierValidationDesktopTypeSupport(DeviceModifiersConflictChecker deviceModifiersConflictChecker) {
        this.deviceModifiersConflictChecker = deviceModifiersConflictChecker;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.DESKTOP_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierDesktop, Defect> validateAddStep1(BidModifierDesktop modifier,
                                                                         CampaignType campaignType,
                                                                         @Nullable AdGroup adGroupWithType, ClientId clientId,
                                                                         CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierDesktop> vb = ModelItemValidationBuilder.of(modifier);
        vb.item(BidModifierDesktop.DESKTOP_ADJUSTMENT)
                .checkBy(adjustment -> validateAdjustmentCommon(adjustment, getType(), campaignType,
                        adGroupWithType, clientId, featuresProvider));
        return vb.getResult();
    }

    @Override
    public ValidationResult<List<BidModifierDesktop>, Defect> validateAddStep2(
            ClientId clientId,
            List<BidModifierDesktop> modifiersToValidate,
            Map<BidModifierKey, BidModifierDesktop> existingModifiers,
            Map<Long, CampaignType> campaignTypes, Map<Long, AdGroup> adGroupsWithType,
            CachingFeaturesProvider featuresProvider,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {

        ListValidationBuilder<BidModifierDesktop, Defect> lvb = ListValidationBuilder.of(modifiersToValidate);
        lvb.checkEachBy(modifier -> validateModifierDoNotRewriteExisting(modifier, existingModifiers,
                BidModifierDesktop.DESKTOP_ADJUSTMENT));

        var typeConflicts = checkNoDesktopOnlyOrTablet(
                clientId, modifiersToValidate, allValidBidModifiersInOperation);

        lvb.checkEach(setConflictingModifiersDefect(typeConflicts));

        var mediaModifiers = modifiersToValidate.stream()
                .filter(m -> isSmartTvEnabled(campaignTypes.get(m.getCampaignId())))
                .collect(Collectors.toList());

        var regularModifiers = modifiersToValidate.stream()
                .filter(m -> !isSmartTvEnabled(campaignTypes.get(m.getCampaignId())))
                .collect(Collectors.toList());

        var isSmartTvEnabled = featuresProvider.isEnabledForClientId(
                clientId, FeatureName.SMARTTV_BID_MODIFIER_ENABLED);

        Set<BidModifierKey> zeroValueConflicts = Sets.union(
                checkNoZeroValue(
                        clientId, regularModifiers, allValidBidModifiersInOperation, false),
                checkNoZeroValue(
                        clientId, mediaModifiers, allValidBidModifiersInOperation, isSmartTvEnabled)
        );

        lvb.checkEach(setDeviceBidModifierAllZerosDefect(zeroValueConflicts));

        return lvb.getResult();
    }

    private Set<BidModifierKey> checkNoZeroValue(
            ClientId clientId,
            List<BidModifierDesktop> modifiersToValidate,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation,
            boolean checkForSmartTv
    ) {
        Set<BidModifierKey> result = new HashSet<>();

        for (var rulesSet : getAdjacentRules(checkForSmartTv)) {
            result.addAll(deviceModifiersConflictChecker.findConflictingModifiers(
                    clientId, modifiersToValidate, allValidBidModifiersInOperation,
                    k -> getZeroValuePredicate().test(k),
                    rulesSet
            ));
        }

        return result;
    }

    private Set<BidModifierKey> checkNoDesktopOnlyOrTablet(
            ClientId clientId,
            List<BidModifierDesktop> modifiersToValidate,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {

        Map<BidModifierType, Predicate<BidModifier>> oppositeTypeRules = new HashMap<>();
        oppositeTypeRules.put(DESKTOP_ONLY_MULTIPLIER, m -> true);
        oppositeTypeRules.put(TABLET_MULTIPLIER, m -> true);

        return deviceModifiersConflictChecker.findConflictingModifiers(
                clientId, modifiersToValidate, allValidBidModifiersInOperation,
                m -> true,
                oppositeTypeRules,
                true
        );
    }

}
