package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo.REGIONAL_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.geoBidModifiersNotSupportedOnAdGroups;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.geoRegionsIntersection;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.nonexistentRegionIds;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationGeoTypeSupport implements BidModifierValidationTypeSupport<BidModifierGeo> {

    private final GeoTreeFactory geoTree;

    @Autowired
    public BidModifierValidationGeoTypeSupport(GeoTreeFactory geoTree) {
        this.geoTree = geoTree;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.GEO_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierGeo, Defect> validateAddStep1(BidModifierGeo modifier,
                                                                     CampaignType campaignType,
                                                                     @Nullable AdGroup adGroupWithType, ClientId clientId,
                                                                     CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierGeo> vb = ModelItemValidationBuilder.of(modifier);

        // Сначала проверяем, можно ли задавать эту корректировку для группы
        vb.check(fromPredicate(o -> o.getAdGroupId() == null,
                geoBidModifiersNotSupportedOnAdGroups()));

        // Общие проверки
        vb.item(REGIONAL_ADJUSTMENTS).checkBy(adjustments ->
                        BidModifierValidationHelper.validateAdjustmentsCommon(
                                adjustments, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider, null),
                When.isTrue(!vb.getResult().hasAnyErrors()));

        // Затем типоспецифичные
        vb.item(REGIONAL_ADJUSTMENTS).checkBy(this::validateGeoAdjustments,
                When.isTrue(!vb.getResult().hasAnyErrors()));

        return vb.getResult();
    }

    @Override
    public ValidationResult<BidModifierGeo, Defect> validateAddStep2(BidModifierGeo modifier,
                                                                     BidModifierGeo existingModifier,
                                                                     CampaignType campaignType,
                                                                     @Nullable AdGroup adGroupWithType,
                                                                     ClientId clientId,
                                                                     CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierGeo> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(REGIONAL_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<BidModifierRegionalAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<BidModifierRegionalAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getRegionalAdjustments().stream())
                            .collect(toList());

                    // Общие проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierRegionalAdjustment>, Defect>
                            commonChecksValidationResult =
                            validateAdjustmentsCommon(allToValidate, getType(), campaignType,
                                    adGroupWithType, clientId, featuresProvider, null);
                    if (commonChecksValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(commonChecksValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierRegionalAdjustment>, Defect>
                            specificValidationResult =
                            validateGeoAdjustments(allToValidate);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    /**
     * Специфичная для Geo валидация.
     * Проверяет то, что каждый регион в наборе встречается не более одного раза.
     * Проверяет то, что каждый регион существует.
     */
    private ValidationResult<List<BidModifierRegionalAdjustment>, Defect> validateGeoAdjustments(
            List<BidModifierRegionalAdjustment> adjustments) {
        ListValidationBuilder<BidModifierRegionalAdjustment, Defect> builder =
                ListValidationBuilder.of(adjustments);

        builder.check(allRegionsAreValid(getGeoTree()));

        builder.check(fromPredicate(list -> list.stream()
                // Скрытые регионы не должны принимать участия в валидации на пересечение
                .filter(it -> !it.getHidden())
                .collect(groupingBy(BidModifierRegionalAdjustment::getRegionId, counting()))
                .values().stream().allMatch(count -> count == 1), geoRegionsIntersection()), When.isValid());
        return builder.getResult();
    }

    private Constraint<List<BidModifierRegionalAdjustment>, Defect> allRegionsAreValid(GeoTree geoTree) {
        return list -> {
            List<Long> invalidIds = StreamEx.of(list)
                    .map(BidModifierRegionalAdjustment::getRegionId)
                    .filter(rid -> !geoTree.hasRegion(rid))
                    .toList();
            return invalidIds.isEmpty() ?
                    null : nonexistentRegionIds(String.join(",", mapList(invalidIds, Object::toString)));
        };
    }

    private GeoTree getGeoTree() {
        return geoTree.getGlobalGeoTree();
    }
}
