package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimits;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.params.NumberDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifiers.Constants.INVENTORY_ADJUSTMENTS_LIMIT;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.invalidPercentShouldBePositive;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.notSupportedMultiplier;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

public class BidModifierValidationHelper {
    private static final List<String> CAMPAIGN_ADGROUP_FIELDS = List.of("CampaignId", "AdGroupId");
    public static final List<CampaignType> TYPES_FOR_MOBILE_OS_BID_MODIFIER = List.of(CampaignType.TEXT,
            CampaignType.PERFORMANCE, CampaignType.CONTENT_PROMOTION, CampaignType.DYNAMIC, CampaignType.CPM_PRICE);

    private BidModifierValidationHelper() {
    }

    /**
     * Выполняет проверки, общие для всех модификаторов, но с настройками, которые зависят от
     * комбинации типа модификатора и типа кампании.
     *
     * @param tooManyConditionsDefect Дефект для случая, когда количество корректировок в наборе превышает лимит
     */
    public static <T extends BidModifierAdjustment> ValidationResult<List<T>, Defect> validateAdjustmentsCommon(
            List<T> adjustments, BidModifierType type, CampaignType campaignType,
            @Nullable AdGroup adGroupWithType, ClientId clientId, CachingFeaturesProvider featuresProvider,
            @Nullable DefectId<NumberDefectParams> tooManyConditionsDefect) {
        BidModifierLimits bidModifierLimits = BidModifierLimitsAdvanced.getLimits(
                type, campaignType, adGroupWithType, clientId, featuresProvider);
        //
        ListValidationBuilder<T, Defect> listValidationBuilder = ListValidationBuilder.of(adjustments);
        //
        boolean shouldValidateMaxConditions =
                bidModifierLimits.maxConditions != null && bidModifierLimits.maxConditions > 0;

        if (shouldValidateMaxConditions) {
            Preconditions.checkNotNull(tooManyConditionsDefect);

            listValidationBuilder.check(maxListSize(bidModifierLimits.maxConditions),
                    new Defect<>(
                            tooManyConditionsDefect,
                            new NumberDefectParams().withMax(bidModifierLimits.maxConditions)));
        }
        //
        listValidationBuilder.checkEachBy(o -> validateAdjustmentCommon(
                o, type, campaignType, adGroupWithType, clientId, featuresProvider));
        return listValidationBuilder.getResult();
    }

    /**
     * Выполняет проверки, общие для всех модификаторов, но с настройками, которые зависят от
     * комбинации типа модификатора и типа кампании. Вариант для отдельного элемента adjustment.
     */
    public static <T extends BidModifierAdjustment> ValidationResult<T, Defect> validateAdjustmentCommon(
            T adjustment, BidModifierType type, CampaignType campaignType,
            @Nullable AdGroup adGroupWithType, ClientId clientId, CachingFeaturesProvider featuresProvider) {
        BidModifierLimits bidModifierLimits = BidModifierLimitsAdvanced.getLimits(
                type, campaignType, adGroupWithType, clientId, featuresProvider);
        return validateAdjustmentCommon(adjustment, bidModifierLimits);
    }

    /**
     * Выполняет проверки, общие для всех модификаторов, но с настройками, которые зависят от
     * комбинации типа модификатора и типа кампании. Вариант для отдельного элемента adjustment.
     */
    public static <T extends BidModifierAdjustment>
    ValidationResult<T, Defect> validateAdjustmentCommon(T adjustment, BidModifierLimits bidModifierLimits) {

        ModelItemValidationBuilder<T> builder = ModelItemValidationBuilder.of(adjustment);

        builder.check(fromPredicate(t -> bidModifierLimits.enabled, notSupportedMultiplier()));
        builder.item(BidModifierAdjustment.PERCENT)
                .checkBy(percent -> percentValidator(percent, bidModifierLimits));
        return builder.getResult();
    }

    public static ValidationResult<Integer, Defect> percentValidator(Integer percent,
                                                                     BidModifierLimits bidModifierLimits) {
        ItemValidationBuilder<Integer, Defect> vb = ItemValidationBuilder.of(percent, Defect.class);
        vb.check(notNull())
                // Сообщение об ошибке не совсем точное, число должно быть неотрицательным, а не "положительным"
                .check(notLessThan(0), invalidPercentShouldBePositive())
                // А теперь проверяем уже min-max, но так, чтобы схожие ошибки  про PERCENT не возвращались одновременно
                // (и про то, что идентификатор должен быть положительным числом, и про min-max ограничения)
                .check(notLessThan(bidModifierLimits.percentMin), When.isValid())
                .check(notGreaterThan(bidModifierLimits.percentMax), When.isValid());
        return vb.getResult();
    }

    public static <T extends BidModifier> Constraint<T, Defect> atLeastCampaignIdOrAdGroupIdRequired() {
        return fromPredicate(o -> o.getCampaignId() != null || o.getAdGroupId() != null,
                BidModifiersDefects.requiredAtLeastOneOfFields(CAMPAIGN_ADGROUP_FIELDS));
    }

    public static <T extends BidModifier> Constraint<T, Defect> possibleOnlyOneCampaignIdOrAdGroupIdField() {
        return fromPredicate(o -> o.getCampaignId() == null || o.getAdGroupId() == null,
                BidModifiersDefects.possibleOnlyOneField(CAMPAIGN_ADGROUP_FIELDS));
    }

    public static boolean isCpmCampaignCondition(CampaignType campaignType) {
        return campaignType == CampaignType.CPM_BANNER
                || campaignType == CampaignType.CPM_DEALS
                || campaignType == CampaignType.CPM_PRICE;
    }

    public static boolean isAllInventoryPercentEqualsZero(BidModifierInventory bidModifierInventory) {
        List<BidModifierInventoryAdjustment> adjustments = bidModifierInventory.getInventoryAdjustments();
        if (adjustments.size() < INVENTORY_ADJUSTMENTS_LIMIT) {
            return false;
        }
        for (BidModifierInventoryAdjustment adjustment : adjustments) {
            if (!adjustment.getPercent().equals(0)) {
                return false;
            }
        }
        return true;
    }

    public static boolean bannerTypePercentEqualsZero(BidModifierBannerType m) {
        List<BidModifierBannerTypeAdjustment> adjustments = m.getBannerTypeAdjustments();
        for (BidModifierBannerTypeAdjustment adjustment : adjustments) {
            if (adjustment.getBannerType() == BannerType.CPM_VIDEO) {
                continue;
            }
            if (!adjustment.getPercent().equals(0)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Проверяет, что в комплексной модели нет повторяющихся универсальных корректировок
     * (разных наборов с одним и тем же типом)
     */
    public static boolean isExpressionModifiersUsedAtMostOnce(ComplexBidModifier complexBidModifier) {
        if (complexBidModifier.getExpressionModifiers() == null) {
            return true;
        }
        long distinctTypesCount = complexBidModifier.getExpressionModifiers().stream()
                .map(BidModifier::getType).distinct().count();
        return distinctTypesCount == complexBidModifier.getExpressionModifiers().size();
    }
}
