package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory.INVENTORY_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.BANNER_TYPE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds.Number.TOO_MANY_INVENTORY_CONDITIONS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.bannerTypePercentEqualsZero;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.setDeviceBidModifierAllZerosDefect;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.validateModifierDoNotRewriteExisting;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationInventoryTypeSupport implements BidModifierValidationTypeSupport<BidModifierInventory> {

    private final DeviceModifiersConflictChecker deviceModifiersConflictChecker;

    public BidModifierValidationInventoryTypeSupport(DeviceModifiersConflictChecker deviceModifiersConflictChecker) {
        this.deviceModifiersConflictChecker = deviceModifiersConflictChecker;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.INVENTORY_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierInventory, Defect> validateAddStep1(BidModifierInventory modifier,
                                                                           CampaignType campaignType,
                                                                           @Nullable AdGroup adGroupWithType,
                                                                           ClientId clientId,
                                                                           CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierInventory> vb = ModelItemValidationBuilder.of(modifier);
        vb.item(INVENTORY_ADJUSTMENTS).checkBy(adjustment ->
                BidModifierValidationHelper
                        .validateAdjustmentsCommon(adjustment, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider, TOO_MANY_INVENTORY_CONDITIONS));
        vb.list(INVENTORY_ADJUSTMENTS)
                .checkEach(unique(BidModifierInventoryAdjustment::getInventoryType),
                        new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_ADJUSTMENT), When.isValid());
        return vb.getResult();
    }

    @Override
    public ValidationResult<List<BidModifierInventory>, Defect> validateAddStep2(
            ClientId clientId,
            List<BidModifierInventory> modifiersToValidate, Map<BidModifierKey, BidModifierInventory> existingModifiers,
            Map<Long, CampaignType> campaignTypes, Map<Long, AdGroup> adGroupsWithType,
            CachingFeaturesProvider featuresProvider,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {
        // Запрещено добавлять одновременно корректировки типа баннера и инвентаря с нулевым множителем
        // т.к. их совместное действие перекроет весь трафик
        Set<BidModifierKey> keysWithConflict =
                deviceModifiersConflictChecker.findConflictingModifiers(
                        clientId, modifiersToValidate, allValidBidModifiersInOperation,
                        BidModifierValidationHelper::isAllInventoryPercentEqualsZero,
                        Map.of(BANNER_TYPE_MULTIPLIER, m -> bannerTypePercentEqualsZero((BidModifierBannerType) m))
                );

        ListValidationBuilder<BidModifierInventory, Defect> lvb = ListValidationBuilder.of(modifiersToValidate);
        lvb.checkEachBy(modifier -> validateModifierDoNotRewriteExisting(modifier, existingModifiers,
                INVENTORY_ADJUSTMENTS));
        lvb.checkEach(setDeviceBidModifierAllZerosDefect(keysWithConflict));
        return lvb.getResult();
    }
}
