package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimits;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile.MOBILE_ADJUSTMENT;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.MOBILE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationSmartTVTypeSupport.isSmartTvEnabled;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.setDeviceBidModifierAllZerosDefect;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.DeviceModifiersConflictChecker.validateModifierDoNotRewriteExisting;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationMobileTypeSupport implements BidModifierValidationTypeSupport<BidModifierMobile> {
    private final DeviceModifiersConflictChecker deviceModifiersConflictChecker;

    public static Predicate<BidModifier> getZeroValuePredicate() {
        return m -> ((BidModifierMobile) m).getMobileAdjustment().getPercent().equals(0)
                && ((BidModifierMobile) m).getMobileAdjustment().getOsType() == null;
    }

    public static void fillZeroValuePredicate(Map<BidModifierType, Predicate<BidModifier>> typeRules) {
        typeRules.put(MOBILE_MULTIPLIER, getZeroValuePredicate());
    }

    public static Set<Map<BidModifierType, Predicate<BidModifier>>> getAdjacentRules(
        boolean checkForSmartTv
    ) {
        var rules1 = new HashMap<BidModifierType, Predicate<BidModifier>>();
        BidModifierValidationDesktopTypeSupport.fillZeroValuePredicate(rules1);
        if (checkForSmartTv) {
            BidModifierValidationSmartTVTypeSupport.fillZeroValuePredicate(rules1);
        }

        var rules2 = new HashMap<BidModifierType, Predicate<BidModifier>>();
        BidModifierValidationDesktopOnlyTypeSupport.fillZeroValuePredicate(rules2);
        BidModifierValidationTabletTypeSupport.fillZeroValuePredicate(rules2);
        if (checkForSmartTv) {
            BidModifierValidationSmartTVTypeSupport.fillZeroValuePredicate(rules2);
        }

        return Set.of(rules1, rules2);
    }

    public BidModifierValidationMobileTypeSupport(DeviceModifiersConflictChecker deviceModifiersConflictChecker) {
        this.deviceModifiersConflictChecker = deviceModifiersConflictChecker;
    }

    @Override
    public BidModifierType getType() {
        return BidModifierType.MOBILE_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierMobile, Defect> validateAddStep1(BidModifierMobile modifier,
                                                                        CampaignType campaignType,
                                                                        @Nullable AdGroup adGroupWithType,
                                                                        ClientId clientId,
                                                                        CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierMobile> vb = ModelItemValidationBuilder.of(modifier);

        BidModifierLimits bidModifierLimitsForAdjustment = BidModifierLimitsAdvanced.getLimits(
                MOBILE_MULTIPLIER, campaignType, adGroupWithType, clientId, featuresProvider);

        vb.item(MOBILE_ADJUSTMENT)
                .checkBy(adjustment -> BidModifierValidationHelper.validateAdjustmentCommon(adjustment,
                        bidModifierLimitsForAdjustment));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<BidModifierMobile>, Defect> validateAddStep2(
            ClientId clientId,
            List<BidModifierMobile> modifiersToValidate, Map<BidModifierKey, BidModifierMobile> existingModifiers,
            Map<Long, CampaignType> campaignTypes,
            Map<Long, AdGroup> adGroupsWithType,
            CachingFeaturesProvider featuresProvider,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {

        ListValidationBuilder<BidModifierMobile, Defect> lvb = ListValidationBuilder.of(modifiersToValidate);

        lvb.checkEachBy(modifier -> validateModifierDoNotRewriteExisting(
                modifier, existingModifiers, MOBILE_ADJUSTMENT));

        var mediaModifiers = modifiersToValidate.stream()
                .filter(m -> isSmartTvEnabled(campaignTypes.get(m.getCampaignId())))
                .collect(Collectors.toList());

        var regularModifiers = modifiersToValidate.stream()
                .filter(m -> !isSmartTvEnabled(campaignTypes.get(m.getCampaignId())))
                .collect(Collectors.toList());

        var considerSmartTv = featuresProvider.isEnabledForClientId(clientId, FeatureName.SMARTTV_BID_MODIFIER_ENABLED);

        Set<BidModifierKey> zeroValueConflicts = new HashSet<>();

        zeroValueConflicts.addAll(checkNoZeroValue(
                clientId, mediaModifiers, allValidBidModifiersInOperation, considerSmartTv));
        zeroValueConflicts.addAll(checkNoZeroValue(
                clientId, regularModifiers, allValidBidModifiersInOperation, false));

        lvb.checkEach(setDeviceBidModifierAllZerosDefect(zeroValueConflicts));

        return lvb.getResult();
    }

    private Set<BidModifierKey> checkNoZeroValue(
            ClientId clientId,
            List<BidModifierMobile> modifiersToValidate,
            Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation,
            boolean checkForSmartTv
    ) {
        Set<BidModifierKey> result = new HashSet<>();

        for (var rulesSet : getAdjacentRules(checkForSmartTv)) {
            result.addAll(deviceModifiersConflictChecker.findConflictingModifiers(
                    clientId, modifiersToValidate, allValidBidModifiersInOperation,
                    k -> getZeroValuePredicate().test(k),
                    rulesSet
            ));
        }

        return result;
    }

}
