package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPositionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition.TRAFARET_POSITION_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds.Number.TOO_MANY_TRAFARET_POSITION_CONDITIONS;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.trafaretPositionsIntersection;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationTrafaretPositionTypeSupport implements BidModifierValidationTypeSupport<BidModifierTrafaretPosition> {

    @Override
    public BidModifierType getType() {
        return BidModifierType.TRAFARET_POSITION_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierTrafaretPosition, Defect> validateAddStep1(BidModifierTrafaretPosition modifier,
                                                                                  CampaignType campaignType,
                                                                                  @Nullable AdGroup adGroupWithType,
                                                                                  ClientId clientId,
                                                                                  CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierTrafaretPosition> vb = ModelItemValidationBuilder.of(modifier);
        vb.item(TRAFARET_POSITION_ADJUSTMENTS).checkBy(adjustment ->
                BidModifierValidationHelper
                        .validateAdjustmentsCommon(adjustment, getType(), campaignType,
                                adGroupWithType, clientId, featuresProvider, TOO_MANY_TRAFARET_POSITION_CONDITIONS));
        vb.list(TRAFARET_POSITION_ADJUSTMENTS)
                .checkEach(unique(BidModifierTrafaretPositionAdjustment::getTrafaretPosition),
                        new Defect<>(BidModifiersDefectIds.GeneralDefects.DUPLICATE_ADJUSTMENT), When.isValid());
        return vb.getResult();
    }

    @Override
    public ValidationResult<BidModifierTrafaretPosition, Defect> validateAddStep2(
            BidModifierTrafaretPosition modifier,
            BidModifierTrafaretPosition existingModifier,
            CampaignType campaignType,
            @Nullable AdGroup adGroupWithType,
            ClientId clientId,
            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierTrafaretPosition> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(TRAFARET_POSITION_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<BidModifierTrafaretPositionAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<BidModifierTrafaretPositionAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getTrafaretPositionAdjustments().stream())
                            .collect(toList());

                    // Общие проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierTrafaretPositionAdjustment>, Defect>
                            commonChecksValidationResult =
                            validateAdjustmentsCommon(allToValidate, getType(), campaignType,
                                    adGroupWithType, clientId, featuresProvider, TOO_MANY_TRAFARET_POSITION_CONDITIONS);
                    if (commonChecksValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(commonChecksValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierTrafaretPositionAdjustment>, Defect>
                            specificValidationResult =
                            validateTrafaretPositionAdjustments(allToValidate);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    /**
     * Специфичная для TrafaretPosition валидация.
     * Проверяет то, что каждая позиция в наборе встречается не более одного раза.
     */
    private ValidationResult<List<BidModifierTrafaretPositionAdjustment>, Defect> validateTrafaretPositionAdjustments(
            List<BidModifierTrafaretPositionAdjustment> adjustments) {
        ListValidationBuilder<BidModifierTrafaretPositionAdjustment, Defect> builder =
                ListValidationBuilder.of(adjustments);

        builder.check(fromPredicate(list -> list.stream()
                .collect(groupingBy(BidModifierTrafaretPositionAdjustment::getTrafaretPosition, counting()))
                .values().stream().allMatch(count -> count == 1), trafaretPositionsIntersection()), When.isValid());
        return builder.getResult();
    }
}
