package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimits;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierLimitsAdvanced;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Интерфейс для реализации типоспецифичной валидации корректировок.
 *
 * @param <B> тип корректировки
 */
public interface BidModifierValidationTypeSupport<B extends BidModifier> {
    int MIN_PERCENT_WITHOUT_DEVICE_MODIFIERS_ENABLED = 50;

    /**
     * @return Тип корректровок, с которой работает данная реализация
     */
    BidModifierType getType();

    default boolean supports(B modifier) {
        return modifier != null && modifier.getType() == getType();
    }

    /**
     * Выполняет валидацию корректировки без учёта уже имеющихся записей в БД.
     * (первый этап валидации)
     *
     * @param modifier         У modifier может не быть campaignId и adGroupId
     *                         (например при добавлении новой группы через комплексную операцию)
     *                         И если нужно понять, к чему относится создаваемая корректировка -- нужно смотреть на
     *                         adGroupWithType
     * @param campaignType     Тип кампании, всегда указан
     * @param adGroupWithType  Группа с полным типом (включая criterion_type).
     *                         Если она указана, то корректировка добавляется на группу.
     *                         В противном случае -- на кампанию.
     *                         Класс объекта должен соответствовать типу группы, то есть видео группа
     *                         должна быть CpmVideoAdGroup, а не просто AdGroup с типом = CPM_VIDEO
     *                         (так сложилось, что часть валидаций выполняется через instanceof -- может быть, имеет
     *                         смысл это потом переделать)
     * @param clientId         клиент
     * @param featuresProvider объект, позволяющий получать фичи клиента без повторных обращений в базу
     */
    ValidationResult<B, Defect> validateAddStep1(B modifier,
                                                 CampaignType campaignType,
                                                 @Nullable AdGroup adGroupWithType,
                                                 ClientId clientId,
                                                 CachingFeaturesProvider featuresProvider);

    default ValidationResult<Integer, Defect> validatePercent(Integer percent,
                                                              CampaignType campaignType,
                                                              AdGroup adGroupWithType,
                                                              ClientId clientId,
                                                              CachingFeaturesProvider featuresProvider,
                                                              @Nullable BidModifierMobile modifier) {
        BidModifierLimits bidModifierLimits = BidModifierLimitsAdvanced.getLimits(
                getType(), campaignType, adGroupWithType, clientId, featuresProvider);

        return BidModifierValidationHelper.percentValidator(percent, bidModifierLimits);
    }

    //Если есть фича на мобильные ОС, и выбрана ОС, то 0 выставлять можно
    default BidModifierLimits getBidModifierLimitsForNonSelectedOs(BidModifierLimits bidModifierLimits) {
        return new BidModifierLimits(MIN_PERCENT_WITHOUT_DEVICE_MODIFIERS_ENABLED,
                bidModifierLimits.percentMax, bidModifierLimits.enabled,
                bidModifierLimits.maxConditions);
    }

    /**
     * Выполняет валидацию (массовую) множества наборов корректировок с учётом уже имеющихся записей в БД.
     * (второй этап валидации).
     * <p>
     * Вызывается непосредственно из ValidationService.
     * Реализация по умолчанию покрывает стандартный случай: случай, когда совместную валидацию
     * старых+новых корректировок можно свести к группировке их по BidModifierKey
     * (по объектам, к которым привязываются наборы корректировок) и к валидации таких пар.
     *
     * @param clientId                        id клиента
     * @param modifiersToValidate             все наборы корректировок для валидации
     *                                        (среди них могут быть наборы, относящиеся к разным кампаним/группам)
     * @param existingModifiers               существующие в БД наборы корректировок, сгруппированные по
     *                                        кампаниям/группам и типам.
     * @param campaignTypes                   информация о типах кампаний
     * @param allValidBidModifiersInOperation все остальные валидные наборы корректировок в операции
     *                                        (видимо, нужно для того, чтобы выполнять совместную валидацию
     *                                        наборов корректировок разных типов -- если они конфликтуют, например)
     */
    default ValidationResult<List<B>, Defect> validateAddStep2(ClientId clientId, List<B> modifiersToValidate,
                                                               Map<BidModifierKey, B> existingModifiers,
                                                               Map<Long, CampaignType> campaignTypes,
                                                               Map<Long, AdGroup> adGroupsWithType,
                                                               CachingFeaturesProvider featuresProvider,
                                                               Map<BidModifierKey, BidModifier> allValidBidModifiersInOperation) {
        ListValidationBuilder<B, Defect> lvb = ListValidationBuilder.of(modifiersToValidate);
        lvb.checkEachBy(modifier -> {
            BidModifierKey key = new BidModifierKey(modifier);
            B existingModifier = existingModifiers.get(key);
            if (existingModifier == null) {
                return ValidationResult.success(modifier);
            }
            return validateAddStep2(
                    modifier,
                    existingModifier,
                    campaignTypes.get(key.getCampaignId()),
                    key.getAdGroupId() != null ? adGroupsWithType.get(key.getAdGroupId()) : null,
                    clientId,
                    featuresProvider);
        }, When.isValid());
        return lvb.getResult();
    }

    /**
     * Выполняет валидацию одного из наборов корректировок с учётом уже имеющегося набора корректировок в БД
     * (для той же кампании/группы) -- второй этап валидации.
     */
    default ValidationResult<B, Defect> validateAddStep2(B modifier, B existingModifier, CampaignType campaignType,
                                                         @Nullable AdGroup adGroupWithType, ClientId clientId,
                                                         CachingFeaturesProvider featuresProvider) {
        return ValidationResult.success(modifier);
    }
}
