package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationTypeSupportDispatcher {
    private final Map<BidModifierType, BidModifierValidationTypeSupport> typeSupportMap;

    @Autowired
    public BidModifierValidationTypeSupportDispatcher(List<BidModifierValidationTypeSupport> typeSupportList) {
        this.typeSupportMap = Maps.uniqueIndex(typeSupportList, BidModifierValidationTypeSupport::getType);
    }

    public BidModifierValidationTypeSupport<BidModifier> getValidationTypeSupportByType(BidModifierType type) {
        //noinspection unchecked
        return checkNotNull(typeSupportMap.get(type), "unsupported BidModifier's type %s", type);
    }

    /**
     * Выполняет валидацию корректировки без учёта уже имеющихся записей в БД.
     * (первый этап валидации)
     *
     * @param modifier         У modifier может не быть campaignId и adGroupId
     *                         (например при добавлении новой группы через комплексную операцию)
     *                         И если нужно понять, к чему относится создаваемая корректировка -- нужно смотреть на
     *                         adGroupWithType
     * @param campaignType     Тип кампании, всегда указан
     * @param adGroupWithType  Группа с полным типом (включая criterion_type).
     *                         Если она указана, то корректировка добавляется на группу.
     *                         В противном случае -- на кампанию.
     *                         Класс объекта должен соответствовать типу группы, то есть видео группа
     *                         должна быть CpmVideoAdGroup, а не просто AdGroup с типом = CPM_VIDEO
     *                         (так сложилось, что часть валидаций выполняется через instanceof -- может быть, имеет
     *                         смысл это потом переделать)
     * @param clientId         клиент
     * @param featuresProvider объект, позволяющий получать фичи клиента без повторных обращений в базу
     */
    @SuppressWarnings("unchecked")
    public ValidationResult<BidModifier, Defect> validateAddStep1(BidModifier modifier,
                                                                  CampaignType campaignType,
                                                                  @Nullable AdGroup adGroupWithType,
                                                                  ClientId clientId,
                                                                  CachingFeaturesProvider featuresProvider) {
        BidModifierValidationTypeSupport typeSupport = typeSupportMap.get(modifier.getType());
        return typeSupport.validateAddStep1(modifier, campaignType, adGroupWithType, clientId, featuresProvider);
    }

    public BidModifierType getTypeOf(BidModifier modifier) {
        return modifier.getType();
    }
}
