package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.List;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherLiteral;
import ru.yandex.direct.core.entity.bidmodifier.WeatherType;
import ru.yandex.direct.core.entity.bidmodifiers.service.CachingFeaturesProvider;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefectIds;
import ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather.WEATHER_ADJUSTMENTS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.CLOUDNESS_VALUES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.MAX_TEMP_VALUE;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.MIN_TEMP_VALUE;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.PREC_STRENGTH_VALUES;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.WEATHER_MULTIPLIER_EXPRESSION_AVAILABLE_OPERATIONS;
import static ru.yandex.direct.core.entity.bidmodifiers.Constants.WEATHER_MULTIPLIER_EXPRESSION_AVAILABLE_PARAMETER;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport.BidModifierValidationHelper.validateAdjustmentsCommon;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

@Component
@ParametersAreNonnullByDefault
public class BidModifierValidationWeatherTypeSupport implements BidModifierValidationTypeSupport<BidModifierWeather> {
    @Override
    public BidModifierType getType() {
        return BidModifierType.WEATHER_MULTIPLIER;
    }

    @Override
    public ValidationResult<BidModifierWeather, Defect> validateAddStep1(
            BidModifierWeather modifier,
            CampaignType campaignType, @Nullable AdGroup adGroupWithType, ClientId clientId,
            CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierWeather> vb = ModelItemValidationBuilder.of(modifier);

        vb.item(WEATHER_ADJUSTMENTS).checkBy(adjustments ->
                validateAdjustmentsCommon(adjustments, getType(), campaignType,
                        adGroupWithType, clientId, featuresProvider,
                        BidModifiersDefectIds.Number.TOO_MANY_WEATHER_CONDITIONS));

        vb.item(WEATHER_ADJUSTMENTS)
                .checkBy(this::validateWeatherAdjustments, When.isValid());

        return vb.getResult();
    }

    @Override
    public ValidationResult<BidModifierWeather, Defect> validateAddStep2(
            BidModifierWeather modifier, BidModifierWeather existingModifier, CampaignType campaignType,
            @Nullable AdGroup adGroupWithType, ClientId clientId, CachingFeaturesProvider featuresProvider) {
        ModelItemValidationBuilder<BidModifierWeather> vb = ModelItemValidationBuilder.of(modifier);
        vb.list(WEATHER_ADJUSTMENTS)
                .checkBy(adjustments -> {
                    ListValidationBuilder<BidModifierWeatherAdjustment, Defect> lvb =
                            ListValidationBuilder.of(adjustments);

                    // Собираем вместе старые+новые значения
                    List<BidModifierWeatherAdjustment> allToValidate = Stream.of(modifier, existingModifier)
                            .flatMap(it -> it.getWeatherAdjustments().stream())
                            .collect(toList());

                    // Общие проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierWeatherAdjustment>, Defect>
                            commonChecksValidationResult =
                            validateAdjustmentsCommon(allToValidate, getType(), campaignType,
                                    adGroupWithType, clientId, featuresProvider,
                                    BidModifiersDefectIds.Number.TOO_MANY_WEATHER_CONDITIONS);
                    if (commonChecksValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(commonChecksValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    // Типоспецифичные проверки совместного списка старых+новых
                    ValidationResult<List<BidModifierWeatherAdjustment>, Defect>
                            specificValidationResult =
                            validateWeatherAdjustments(allToValidate);
                    if (specificValidationResult.hasAnyErrors()) {
                        transferSubNodesWithIssues(specificValidationResult, lvb.getResult());
                        return lvb.getResult();
                    }

                    return lvb.getResult();
                });
        return vb.getResult();
    }

    private ValidationResult<List<BidModifierWeatherAdjustment>, Defect> validateWeatherAdjustments(
            List<BidModifierWeatherAdjustment> adjustments) {
        ListValidationBuilder<BidModifierWeatherAdjustment, Defect> listValidationBuilder =
                ListValidationBuilder.of(adjustments);
        listValidationBuilder
                .checkEachBy(this::validateWeatherAdjustmentSchema);
        return listValidationBuilder.getResult();
    }

    private ValidationResult<BidModifierWeatherAdjustment, Defect> validateWeatherAdjustmentSchema(
            BidModifierWeatherAdjustment adjustment) {
        ModelItemValidationBuilder<BidModifierWeatherAdjustment> vb = ModelItemValidationBuilder.of(adjustment);

        vb.list(BidModifierWeatherAdjustment.EXPRESSION)
                .check(notNull(), BidModifiersDefects.requiredAtLeastOneExpression())
                .check(notEmptyCollection(), BidModifiersDefects.requiredAtLeastOneExpression())
                .checkEachBy(this::validateExpressionList, When.isValid());
        return vb.getResult();
    }

    private ValidationResult<List<BidModifierWeatherLiteral>, Defect> validateExpressionList(
            List<BidModifierWeatherLiteral> expressions) {
        ListValidationBuilder<BidModifierWeatherLiteral, Defect> vb = ListValidationBuilder.of(expressions);
        vb.checkEachBy(this::validateWeatherExpressionSchema, When.isValid());
        vb.checkEachBy(this::validateWeatherExpression, When.isValid());
        return vb.getResult();
    }

    private ValidationResult<BidModifierWeatherLiteral, Defect> validateWeatherExpressionSchema(
            BidModifierWeatherLiteral expression) {
        return ItemValidationBuilder.<BidModifierWeatherLiteral, Defect>
                of(expression).check(
                fromPredicate(
                        exp -> exp.getOperation() != null && exp.getParameter() != null && exp.getValue() != null,
                        BidModifiersDefects.requiredExpressionFields())
        ).getResult();
    }

    private ValidationResult<BidModifierWeatherLiteral, Defect> validateWeatherExpression(
            BidModifierWeatherLiteral expression) {
        ModelItemValidationBuilder<BidModifierWeatherLiteral> vb = ModelItemValidationBuilder.of(expression);
        vb.item(BidModifierWeatherLiteral.VALUE)
                .check(tempMinValue(),
                        When.isTrue(expression.getParameter() == WeatherType.TEMP))
                .check(tempMaxValue(),
                        When.isTrue(expression.getParameter() == WeatherType.TEMP))
                .check(precStrengthValidValue(),
                        When.isTrue(expression.getParameter() == WeatherType.PREC_STRENGTH))
                .check(cloudnessValidValue(),
                        When.isTrue(expression.getParameter() == WeatherType.CLOUDNESS));

        vb.item(BidModifierWeatherLiteral.PARAMETER)
                .check(notNull(), BidModifiersDefects.requiredExpressionFields())
                .check(validParameter(),
                        When.isTrue(WEATHER_MULTIPLIER_EXPRESSION_AVAILABLE_OPERATIONS
                                .containsKey(expression.getParameter())));

        vb.item(BidModifierWeatherLiteral.OPERATION)
                .check(inSet(WEATHER_MULTIPLIER_EXPRESSION_AVAILABLE_OPERATIONS.get(expression.getParameter())));

        return vb.getResult();
    }

    private Constraint<Integer, Defect> tempMinValue() {
        return fromPredicate(v -> v >= MIN_TEMP_VALUE,
                BidModifiersDefects.tempMustBeGreaterThanOrEqualToMin(MIN_TEMP_VALUE));
    }

    private Constraint<Integer, Defect> tempMaxValue() {
        return fromPredicate(v -> v <= MAX_TEMP_VALUE,
                BidModifiersDefects.tempMustBeLessThanOrEqualToMax(MAX_TEMP_VALUE));
    }

    private Constraint<Integer, Defect> precStrengthValidValue() {
        return fromPredicate(PREC_STRENGTH_VALUES::contains, BidModifiersDefects.invalidParameterValue());
    }

    private Constraint<Integer, Defect> cloudnessValidValue() {
        return fromPredicate(CLOUDNESS_VALUES::contains, BidModifiersDefects.invalidParameterValue());
    }

    private Constraint<WeatherType, Defect> validParameter() {
        return fromPredicate(WEATHER_MULTIPLIER_EXPRESSION_AVAILABLE_PARAMETER::contains,
                BidModifiersDefects.invalidParameter());
    }

}
