package ru.yandex.direct.core.entity.bidmodifiers.validation.typesupport;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.container.BidModifierKey;
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepositoryInterface;
import ru.yandex.direct.dbutil.ShardByClient;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.deviceBidModifiersAllZeros;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.expressionConditionsIntersection;
import static ru.yandex.direct.core.entity.bidmodifiers.validation.BidModifiersDefects.singleValueModifierAlreadyExists;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Запрещено добавлять одновременно мобильные и десктопные корректировки с нулевым множителем
 * т.к. их совместное действие перекроет весь трафик
 *
 * Запрещено добавлять одновременно DESKTOP и DESKTOP_ONLY - т.к. они обе затрагивают десктопы
 * Запрещено добавлять одновременно DESKTOP и TABLET - т.к. они обе затрагивают планшеты
 *
 * и т.п.
 */
@Component
public class DeviceModifiersConflictChecker {
    private final ShardByClient shardHelper;
    private final BidModifierRepositoryInterface bidModifierRepository;

    public DeviceModifiersConflictChecker(ShardByClient shardHelper,
                                          BidModifierRepositoryInterface bidModifierRepository) {
        this.shardHelper = shardHelper;
        this.bidModifierRepository = bidModifierRepository;
    }

    public enum CheckedPredicates {
        ALL_PASS,
        SOME_FAILED,
        ALL_FAILED
    }

    public static CheckedPredicates checkPredicates(
            Collection<BidModifier> adjacentModifiers,
            Map<BidModifierType, Predicate<BidModifier>> adjacentModifierChecks) {
        int failedChecks = 0;
        for (var modifier : adjacentModifiers) {
            if (
                adjacentModifierChecks.containsKey(modifier.getType()) &&
                        adjacentModifierChecks.get(modifier.getType()).test(modifier)
            ) {
                failedChecks++;
            }
        }
        if (failedChecks == 0) return CheckedPredicates.ALL_PASS;
        if (failedChecks == adjacentModifierChecks.size()) return CheckedPredicates.ALL_FAILED;
        return CheckedPredicates.SOME_FAILED;
    }

    static <B extends BidModifier> Constraint<B, Defect> setDeviceBidModifierAllZerosDefect(
            Set<BidModifierKey> keysWithConflict) {
        return Constraint.fromPredicate(
                m -> !keysWithConflict.contains(new BidModifierKey(m)), deviceBidModifiersAllZeros());
    }

    static <B extends BidModifier> Constraint<B, Defect> setConflictingModifiersDefect(
            Set<BidModifierKey> keysWithConflict) {
        return Constraint.fromPredicate(
                m -> !keysWithConflict.contains(new BidModifierKey(m)), expressionConditionsIntersection());
    }

    static <T extends BidModifier, V> ValidationResult<T, Defect>
    validateModifierDoNotRewriteExisting(T modifier, Map<BidModifierKey, T> existingModifiers, ModelProperty<T, V> prop) {
        BidModifierKey key = new BidModifierKey(modifier);
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(modifier);
        // Так как уже существует запись в hierarchicalMultiplier, то добавить туда уже ничего нельзя
        vb.item(prop)
                .check(Constraint.fromPredicate(
                        adj -> !existingModifiers.containsKey(key), singleValueModifierAlreadyExists()));
        return vb.getResult();
    }

    /**
     * Найти ключи корректировок имеющих конфликт с другими корректировками. Для мобильных ищутся конфликтующие
     * десктопные и наоборот. Это нужно, чтобы проверить требование отсутствия одновременно и мобильной и десктопной
     * корректировки с нулевым множителем. Конфликт может быть с корректировками в операции (добавление) или с
     * уже присутствующими в базе.
     *
     * @param modifiersToValidate             корректировки к которым ищутся конфликтующие
     * @param modifiersInOperation все корректировки, участвующие в операции для проверки конфликтов среди них
     * @param checkBaseTypeForPossibleConflict                предикат определения возможности конфликта на корректировках к которым ищутся конфликты
     * @param checkAdjacentTypesForPossibleConflicts    хеш: тип корректировок, среди которых ищутся конфликты ->
     *                                        предикат определения возможности конфликта на корректировках в которых ищутся конфликты
     * @param <B>                             тип корректировок к которым ищутся конфликты
     * @return множество ключей корректировок с наличием конфликтов
     */

    <B extends BidModifier> Set<BidModifierKey> findConflictingModifiers(
            ClientId clientId,
            List<B> modifiersToValidate,
            Map<BidModifierKey, BidModifier> modifiersInOperation,
            Predicate<B> checkBaseTypeForPossibleConflict,
            Map<BidModifierType, Predicate<BidModifier>> checkAdjacentTypesForPossibleConflicts
    ) {
       return findConflictingModifiers(
               clientId,
               modifiersToValidate,
               modifiersInOperation,
               checkBaseTypeForPossibleConflict,
               checkAdjacentTypesForPossibleConflicts,
               false);
    }

    <B extends BidModifier> Set<BidModifierKey> findConflictingModifiers(
            ClientId clientId,
            List<B> modifiersToValidate,
            Map<BidModifierKey, BidModifier> modifiersFromOperation,
            Predicate<B> checkBaseTypeForPossibleConflict,
            Map<BidModifierType, Predicate<BidModifier>> checkAdjacentTypesForPossibleConflicts,
            /*
             * Считать ли за ошибку, если провалено только часть проверок.
             * При false ошибкой будет считаться только если перовалены все проверки
             */
            boolean countAnyFailedCheckAsFail
    ) {

        Map<BidModifierKey, List<BidModifierKey>> possiblyConflictingKeys = computePossiblyConflictingKeys(
                modifiersToValidate, checkBaseTypeForPossibleConflict, checkAdjacentTypesForPossibleConflicts.keySet());

        if (possiblyConflictingKeys.isEmpty()) {
            // нет никаких возможных конфликтов, например, все корректировки с множителем больше 0
            return emptySet();
        }

        // сначала нужно проверить корректировки в операции, т.к. они могут перекрывать то, что лежит в БД
        Map<BidModifierKey, Res> operationLevelConflicts =
                checkConflicts(
                    possiblyConflictingKeys,
                    modifiersFromOperation,
                    checkAdjacentTypesForPossibleConflicts);

        Map<BidModifierKey, List<BidModifierKey>> possiblyConflictingKeysInDb = EntryStream.of(possiblyConflictingKeys)
                .filterKeys(
                        key -> operationLevelConflicts.get(key) == Res.UNKNOWN ||
                                (!countAnyFailedCheckAsFail && operationLevelConflicts.get(key) == Res.SOME_BAD)
                )
                .toMap();
        Set<BidModifierKey> keysWithConflictOnOperationLevel = EntryStream.of(operationLevelConflicts)
                .filterValues(res -> res == Res.BAD || (countAnyFailedCheckAsFail && res == Res.SOME_BAD))
                .keys()
                .toSet();
        if (possiblyConflictingKeysInDb.isEmpty()) {
            // всё решилось на уровне операции
            return keysWithConflictOnOperationLevel;
        }

        int shard = shardHelper.getShardByClientId(clientId);
        Set<BidModifierKey> keysToRetrieve = ImmutableSet.copyOf(
                possiblyConflictingKeysInDb.values().stream().flatMap(Collection::stream).collect(Collectors.toSet()));
        Map<BidModifierKey, BidModifier> modifiersFromDb =
                bidModifierRepository.getBidModifiersByKeys(shard, keysToRetrieve);

        Map<BidModifierKey, BidModifier> combinedModifiers = new HashMap<>();
        combinedModifiers.putAll(modifiersFromDb);
        combinedModifiers.putAll(modifiersFromOperation);

        Map<BidModifierKey, Res> combinedConflicts =
                checkConflicts(possiblyConflictingKeysInDb, combinedModifiers, checkAdjacentTypesForPossibleConflicts);

        Set<BidModifierKey> keysWithConflictOnBothLevels = EntryStream.of(combinedConflicts)
                .filterValues(res -> res == Res.BAD || (countAnyFailedCheckAsFail && res == Res.SOME_BAD))
                .keys()
                .toSet();
        return Sets.union(keysWithConflictOnOperationLevel, keysWithConflictOnBothLevels);
    }

    private static <B extends BidModifier> Map<BidModifierKey, List<BidModifierKey>> computePossiblyConflictingKeys(
        List<B> modifiersToValidate,
        Predicate<B> checkBaseTypeFotConflict,
        Collection<BidModifierType> adjacentTypes
    ) {
        return StreamEx.of(modifiersToValidate)
                .filter(checkBaseTypeFotConflict)
                .map(BidModifierKey::new)
                .toSetAndThen(StreamEx::of)
                .mapToEntry(oneKey -> computePossiblyConflictingKey(oneKey, adjacentTypes))
                .toMap();
    }

    private static List<BidModifierKey> computePossiblyConflictingKey(BidModifierKey key, Collection<BidModifierType> adjacentTypes) {
        return mapList(adjacentTypes,
                type -> new BidModifierKey(key.getCampaignId(), key.getAdGroupId(), type));
    }

    private static Map<BidModifierKey, Res> checkConflicts(
            Map<BidModifierKey, List<BidModifierKey>> keys,
            Map<BidModifierKey, BidModifier> modifiers,
            Map<BidModifierType, Predicate<BidModifier>> adjacentTypesWithPossibleConflict) {
        return EntryStream.of(keys)
                .mapValues(key -> checkKeyOnConflicts(key, modifiers, adjacentTypesWithPossibleConflict))
                .toMap();
    }

    /**
     * Внутренний результат проверки конфликта
     */
    enum Res {
        GOOD, // все смежные корректировки протестированы, все проверки не выявили конфликта
        BAD, // все смежные корректировки протестированы, все проверки выявили конфликт
        SOME_BAD, // найден по крайней мере один конфликт, протестирована по крайней мере одна корректировка
        UNKNOWN // Не все корректировки протестированы
    }

    private static Res checkKeyOnConflicts(
            List<BidModifierKey> keys,
            Map<BidModifierKey, BidModifier> modifiers,
            Map<BidModifierType, Predicate<BidModifier>> checksAdjacentTypesForPossibleConflict) {

        var modifierKeysToCheck = keys.stream()
                .filter(modifiers::containsKey)
                .collect(Collectors.toList());

        if (modifierKeysToCheck.isEmpty()) {
            return Res.UNKNOWN;
        }

        var failedModifierKeys = modifierKeysToCheck.stream()
                .filter(
                        key -> checksAdjacentTypesForPossibleConflict.getOrDefault(key.getType(), defaultPredicate -> false)
                            .test(modifiers.get(key))
                )
                .collect(Collectors.toList());

        if (failedModifierKeys.size() == keys.size()) {
            // every key is checked, every key is bad
            return Res.BAD;
        }

        if (failedModifierKeys.isEmpty() && modifierKeysToCheck.size() == keys.size()) {
            // every key is checked, every key is good
            return Res.GOOD;
        }

        if (!failedModifierKeys.isEmpty()) {
            // at least some keys failed
            return Res.SOME_BAD;
        }

        // none failed, but not all are checked yet (if that is a DB level check - than GOOD)
        return Res.UNKNOWN;
    }
}
