package ru.yandex.direct.core.entity.bids.container.interpolator;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.utils.math.Point;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PointConverter {
    public static final int TRAFFIC_VOLUME_SCALE = 10_000;

    private PointConverter() {
    }

    public static List<Point> toBidTrafficVolumePoints(List<TrafaretBidItem> trafaretBidItems) {
        return mapList(trafaretBidItems, x -> Point.fromDoubles(x.getBid().bigDecimalValue().doubleValue(),
                (double) x.getPositionCtrCorrection() / TRAFFIC_VOLUME_SCALE));
    }

    public static List<Point> toPriceTrafficVolumePoints(List<TrafaretBidItem> trafaretBidItems) {
        return mapList(trafaretBidItems, x -> Point.fromDoubles(x.getPrice().bigDecimalValue().doubleValue(),
                (double) x.getPositionCtrCorrection() / TRAFFIC_VOLUME_SCALE));
    }

    /**
     * Объединяет список сглаженных точек [bids, trafficVolume] и [amnestyPrice, trafficVolume] по trafficVolume
     */
    public static List<TrafaretBidItem> toTrafaretBidItems(List<Point> bidsTrafficVolumePoints,
                                                           List<Point> priceTrafficVolumePoints,
                                                           CurrencyCode currencyCode) {
        Map<Long, List<Double>> bidsByTrafficVolume = StreamEx.of(bidsTrafficVolumePoints)
                .mapToEntry(p -> (long) (p.getY() * TRAFFIC_VOLUME_SCALE), Point::getX)
                .grouping();
        Map<Long, List<Double>> amnestyPriceByTrafficVolume = StreamEx.of(priceTrafficVolumePoints)
                .mapToEntry(p -> (long) (p.getY() * TRAFFIC_VOLUME_SCALE), Point::getX)
                .grouping();

        return EntryStream.of(bidsByTrafficVolume)
                .mapKeyValue(
                        (trafficVolume, bid) -> unionToTrafaretBidItem(bid,
                                amnestyPriceByTrafficVolume.get(trafficVolume),
                                trafficVolume,
                                currencyCode))
                .flatMap(StreamEx::of)
                .sortedByLong(TrafaretBidItem::getPositionCtrCorrection)
                .toList();
    }

    /**
     * Объединяет полученные точки c одинаковым trafficVolume.
     * Для одного traffic volume может быть 2 bid и 1 amnesty price(и наоборот), тогда возвращаем 2 AuctionItem с разным
     * bid и одинаковым amnesty price.
     */
    static List<TrafaretBidItem> unionToTrafaretBidItem(List<Double> bidsList, List<Double> amnestyPriceList,
                                                        Long trafficVolume, CurrencyCode currencyCode) {
        checkArgument(!isEmpty(bidsList) && bidsList.size() <= 2, "bidsList should have one ore two elements");
        checkArgument(!isEmpty(amnestyPriceList) && amnestyPriceList.size() <= 2,
                "amnestyPriceList should have one ore two elements");

        ArrayList<TrafaretBidItem> result = new ArrayList<>();
        result.add(toTrafaretBidItem(bidsList.get(0), amnestyPriceList.get(0), trafficVolume, currencyCode));

        if (bidsList.size() == 2 || amnestyPriceList.size() == 2) {
            double bid = bidsList.get(0);
            if (bidsList.size() == 2) {
                bid = bidsList.get(1);
            }

            double amnestyPrice = amnestyPriceList.get(0);
            if (amnestyPriceList.size() == 2) {
                amnestyPrice = amnestyPriceList.get(1);
            }
            result.add(toTrafaretBidItem(bid, amnestyPrice, trafficVolume, currencyCode));
        }
        return result;
    }

    private static TrafaretBidItem toTrafaretBidItem(double bid, double amnestyPrice, long trafficVolume,
                                                     CurrencyCode currencyCode) {
        Money bidMoney = Money.valueOf(bid, currencyCode);
        return new TrafaretBidItem()
                .withPositionCtrCorrection(trafficVolume)
                // должен выполняться инвариант: bid >= amnestyPrice.
                .withPrice(bid < amnestyPrice ? bidMoney : Money.valueOf(amnestyPrice, currencyCode))
                .withBid(bidMoney);
    }

}
