package ru.yandex.direct.core.entity.bids.interpolator;

import java.util.Arrays;
import java.util.List;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bids.container.interpolator.Cap;
import ru.yandex.direct.core.entity.bids.container.interpolator.CapKey;
import ru.yandex.direct.utils.math.Point;

import static com.google.common.base.Preconditions.checkState;

/**
 * Фабрика колпаков для интерполяции.
 * На текущий момент, для добавления колпака нужно добавить его в capsMap в конструкторе.
 */
public class CapFactory {
    private final ImmutableMap<CapKey, Cap> capsMap;

    /**
     * Для колпаков должно соблюдаться правило что сумма по второму аргументу должна быть всегда == 1
     * Делаем эту проверку в getCap, чтобы приложение могло подняться при некорректно заданных колпаках.
     */
    public CapFactory() {
        this.capsMap = ImmutableMap.<CapKey, Cap>builder()
                .put(new CapKey(false, 1, "any"), defaultCap())
                .put(new CapKey(false, 1, "noInterpolation"), noInterpolationCap())
                .build();
    }

    /**
     * Сглаживание по умолчанию
     */
    private static Cap defaultCap() {
        List<Point> pairs = Arrays.asList(
                Point.fromDoubles(-0.2, 0.0),
                Point.fromDoubles(-0.15, 0.05),
                Point.fromDoubles(-0.1, 0.1),
                Point.fromDoubles(-0.05, 0.35),
                Point.fromDoubles(0.05, 0.35),
                Point.fromDoubles(0.1, 0.1),
                Point.fromDoubles(0.15, 0.05),
                Point.fromDoubles(0.2, 0.0)
        );
        return Cap.fromPoints(pairs);
    }

    /**
     * Отсутствие сглаживания
     */
    private static Cap noInterpolationCap() {
        List<Point> pairs = Arrays.asList(
                Point.fromDoubles(0, 1),
                Point.fromDoubles(0.0000001, 0)
        );
        return Cap.fromPoints(pairs);
    }

    /**
     * Возвращает колпак по заданному ключу. Если ключ не найден, возвращает дефолтный колпак.
     */
    public Cap getCap(CapKey key) {
        Cap cap = capsMap.get(key);
        if (cap == null) {
            cap = defaultCap();
        }
        double sum = StreamEx.of(cap.getPoints())
                .foldLeft(0.0, (s, point) -> s + point.getY());
        checkState(sum == 1, "Sum of part must be equal 1");
        return cap;
    }
}
