package ru.yandex.direct.core.entity.bids.interpolator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.LongStreamEx;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.bids.container.interpolator.Cap;
import ru.yandex.direct.core.entity.bids.container.interpolator.CapKey;
import ru.yandex.direct.core.entity.bids.container.interpolator.PointConverter;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.MoneyUtils;
import ru.yandex.direct.utils.math.Point;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.springframework.util.CollectionUtils.isEmpty;

@Service
@ParametersAreNonnullByDefault
public class InterpolatorService {
    private static final Logger logger = LoggerFactory.getLogger(InterpolatorService.class);
    private static final BigDecimal MIN_PREMIUM_BID = BigDecimal.valueOf(0.05);
    private static final Money MIN_PREMIUM_BID_YND_FIXED = Money.valueOf(MIN_PREMIUM_BID, CurrencyCode.YND_FIXED);
    private static final List<Double> GUARANTEES_TRAFFIC_VOLUMES = asList(6.1, 6.7, 7.2, 8.5);
    private static final long DEFAULT_MIN_ORDINATE = 5L;

    public static final int HALF_OF_THE_DEFAULT_MAX_TRAFFIC_VOLUME = 500000;

    private final CurrencyRateService currencyRateService;
    private final CapFactory capFactory;

    @Autowired
    public InterpolatorService(CurrencyRateService currencyRateService, CapFactory capFactory) {
        this.currencyRateService = currencyRateService;
        this.capFactory = capFactory;
    }

    /**
     * Из списка {@link ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem} интерполируем по отдельности
     * [bid, positionCtrCorrection] и [price, positionCtrCorrection] и возвращаем объединенный по
     * positionCtrCorrection(он же trafficVolume) результат
     *
     * @param capKey           -- ключ по которому выберем колпак
     * @param trafaretBidItems -- список данных для интерполяции
     * @param customOrdinates  -- кастомные ординаты, по которым будет считать точки интерполирующей функции
     * @param currencyCode     -- валюта
     */
    public List<TrafaretBidItem> getInterpolatedTrafaretBidItems(CapKey capKey, List<TrafaretBidItem> trafaretBidItems,
                                                                 @Nullable List<Double> customOrdinates,
                                                                 CurrencyCode currencyCode) {
        Cap cap = capFactory.getCap(capKey);

        // Не учитываем при интерполяции точки с ценой выше 1.5 от максимально показываемой
        Money maxCountableBid = Money.valueOf(currencyCode.getCurrency().getMaxShowBid(), currencyCode).multiply(1.5);

        List<TrafaretBidItem> filteredBidItems = StreamEx.of(trafaretBidItems)
                .remove(b -> b.getBid().compareTo(maxCountableBid) > 0)
                .toList();

        if (filteredBidItems.isEmpty()) {
            // если все точки отфильтровались из-за высокой ставки, от безысходности сглаживаем оригинальные точки
            filteredBidItems = trafaretBidItems;
        }

        List<Point> bidTrafficVolumePoints = PointConverter.toBidTrafficVolumePoints(filteredBidItems);
        List<Point> interpolatedBidsTrafficVolumePoints =
                getInterpolatedPoints(cap, currencyCode, bidTrafficVolumePoints, customOrdinates);

        List<Point> interpolatedAmnestyPriceTrafficVolumePoints =
                calcAmnestyPriceFromBids(interpolatedBidsTrafficVolumePoints, currencyRounder(currencyCode));

        return PointConverter
                .toTrafaretBidItems(interpolatedBidsTrafficVolumePoints, interpolatedAmnestyPriceTrafficVolumePoints,
                        currencyCode);
    }

    /**
     * Вычисление списываемой цены в соответсвии с правилами VCG на основе зависимости {@code TrafficVolume(Bid)}.
     * <p>
     * Правило вычисления списываемой цены:<br/>
     * Участник аукциона платит свою цену лишь за то, что получил сверх предыдущего участника аукциона.
     */
    private List<Point> calcAmnestyPriceFromBids(List<Point> points, Function<Double, Double> rounder) {
        if (points.isEmpty()) {
            return emptyList();
        }
        List<Point> result = new ArrayList<>(points.size());
        double curAmnestyPrice = points.get(0).getX();
        // не забываем добавить соответсвующую точку в результирующий список
        result.add(Point.fromDoubles(rounder.apply(curAmnestyPrice), points.get(0).getY()));

        for (int i = 1; i < points.size(); i++) {
            Point point = points.get(i);
            double bid = point.getX();
            double trafficVol = point.getY();

            Point prevPoint = points.get(i - 1);
            double prevTrafficVol = prevPoint.getY();

            // curAmnestyPrice * (prevTrafficVol / trafficVol) -- за эту часть списываем цены предыдущей ставки
            // bid * ((trafficVol - prevTrafficVol) / trafficVol) -- это добавка по цене текущей ставки
            curAmnestyPrice = (curAmnestyPrice * prevTrafficVol + bid * (trafficVol - prevTrafficVol)) / trafficVol;

            double roundedAmnestyPrice = rounder.apply(curAmnestyPrice);

            result.add(Point.fromDoubles(roundedAmnestyPrice, point.getY()));
        }
        return result;
    }

    /**
     * Возвращает функцию для округления значений цен вверх до шага Торгов указанной валюты
     */
    private static Function<Double, Double> currencyRounder(CurrencyCode currencyCode) {
        return m -> MoneyUtils.roundToAuctionStepUpDouble(m, currencyCode.getCurrency().getAuctionStep().doubleValue());
    }

    /**
     * Пытаемся сгладить исходные точки c помощью {@link ru.yandex.direct.core.entity.bids.interpolator.Smoother}
     * Сглаженные точки определяют интерполирующую функцию, считаем точки функции по customOrdinates.
     * Если не удается сгладить точки, делаем интерполяцию по исходным.
     *
     * @param cap             -- колпак
     * @param currencyCode    -- валюта
     * @param points          -- список точек для сглаживания и интерполяции
     * @param customOrdinates -- кастомные ординаты, если не указаны считаются дефолтные
     */
    List<Point> getInterpolatedPoints(Cap cap, CurrencyCode currencyCode, List<Point> points,
                                      @Nullable List<Double> customOrdinates) {
        checkArgument(!isEmpty(points), "Points should be not empty");
        List<Point> pointsForInterpolation;
        List<Point> sortedPoints = StreamEx.of(points)
                .sorted()
                .toList();
        List<Point> nonDecreasingPoints = removePointsViolateNonDecreasingOrder(sortedPoints);
        List<Double> ordinates = getOrdinatesToCalculate(customOrdinates, points);

        try {
            Money minPrice = currencyRateService.convertMoney(MIN_PREMIUM_BID_YND_FIXED, currencyCode);
            Smoother smoother = new Smoother(cap, currencyCode, minPrice.bigDecimalValue().doubleValue());
            pointsForInterpolation = smoother.execute(nonDecreasingPoints);
        } catch (Exception e) {
            logger.warn("Can't smooth points", e);
            pointsForInterpolation = StreamEx.of(Point.ZERO)
                    .append(nonDecreasingPoints)
                    .toList();
        }
        return calculateAbscissaByOrdinate(pointsForInterpolation, ordinates, currencyCode);
    }

    /**
     * Удаляет точки, нарушающие неубывание.
     * <p>
     * Сейчас из Perl'а могут приходить ставки для Гарантии дороже входа в Спецразмещение.
     * Так как стратегии "Показ под результатами поиска" больше нет, то при установке такой ставки объявление попадёт
     * в Спецразмещение.
     */
    List<Point> removePointsViolateNonDecreasingOrder(List<Point> points) {
        List<Point> resultPoints = new ArrayList<>(points.size());
        Point lastResultPoint = points.get(0);
        resultPoints.add(lastResultPoint);
        List<Point> violations = new ArrayList<>(points.size());
        for (int i = 1; i < points.size(); i++) {
            Point point = points.get(i);

            if (point.getY() < lastResultPoint.getY()) {
                violations.add(point);
            } else {
                lastResultPoint = point;
                resultPoints.add(lastResultPoint);
            }
        }
        if (!violations.isEmpty()) {
            // Логируем нарушения порядка с уровнем debug, так как сейчас считаем их допустимыми
            logger.debug("Next points violate non-decreasing order: {}", violations);
        }

        return resultPoints;
    }

    /**
     * Возвращает координаты точки по заданной ординате
     * Функция неубывающая => может быть несколько абсцисс по заданной ординате.
     * В таком случае, берутся границы интервала, в который попала ордината.
     */
    List<Point> calculateAbscissaByOrdinate(List<Point> points, List<Double> ordinates, CurrencyCode currencyCode) {
        List<Point> resultPoints = new ArrayList<>(ordinates.size());
        int lastPointIndex = 0;
        for (Double ordinate : ordinates) {
            for (int i = lastPointIndex; i < points.size() - 1; i++) {
                Point point = points.get(i);
                Point nextPoint = points.get(i + 1);

                boolean isConstFunction = point.getY() == nextPoint.getY();
                boolean isInInterval = point.getY() <= ordinate && nextPoint.getY() > ordinate;

                if (!isConstFunction && isInInterval) {
                    resultPoints.add(interpolateLinear(point, nextPoint, ordinate, currencyCode));
                    lastPointIndex = i;
                    break;
                }

                if (isConstFunction && point.getY() == ordinate) {
                    resultPoints.add(point);
                    resultPoints.add(nextPoint);
                    lastPointIndex = i;
                    break;
                }

                boolean nextPointIsLast = i + 1 == points.size() - 1;

                if (nextPointIsLast && nextPoint.getY() == ordinate) {
                    resultPoints.add(nextPoint);
                    break;
                }
            }
        }
        return resultPoints;
    }

    /**
     * Линейная интерполяция по ординате
     *
     * @param beginPoint   -- точка начала отрезка
     * @param endPoint     -- точка конца отрезка
     * @param ordinate     -- значение по оси ординат, по которому будет вычисляться значание по оси абсцисс
     * @param currencyCode -- код валюты, для округления значения по оси абсцисс.
     * @return -- результат интерполяции, точка принадлежащая отрезку [beginPoint, endPoint] с ординатой ordinate
     */
    Point interpolateLinear(Point beginPoint, Point endPoint, double ordinate, CurrencyCode currencyCode) {
        checkArgument(beginPoint.getY() != (endPoint.getY()), "points should have different ordinates");

        double abscissa = (endPoint.getX() - beginPoint.getX()) * (ordinate - beginPoint.getY()) /
                (endPoint.getY() - beginPoint.getY()) + (beginPoint.getX());

        double roundedAbscissa = MoneyUtils.roundToAuctionStepUpDouble(abscissa,
                currencyCode.getCurrency().getAuctionStep().doubleValue());
        return Point.fromDoubles(roundedAbscissa, ordinate);
    }

    /**
     * Если установлены кастомные ординаты, сортируем их, удаляем ординаты большие максимальной ординаты points
     * Если нет, возвращаем ординаты от 5 до максимальной ординаты points.
     *
     * @param customOrdinates -- кастомные ординаты
     * @param points          -- исходные точки
     */
    @Nonnull
    private List<Double> getOrdinatesToCalculate(@Nullable List<Double> customOrdinates, List<Point> points) {
        double maxOrdinate = StreamEx.of(points)
                .mapToDouble(Point::getY)
                .max()
                .orElseThrow(() -> new IllegalStateException("Cannot find max ordinate"));


        if (!isEmpty(customOrdinates)) {
            return StreamEx.of(customOrdinates)
                    .remove(ordinate -> ordinate > maxOrdinate)
                    .sorted()
                    .toList();
        }

        return LongStreamEx
                .rangeClosed(DEFAULT_MIN_ORDINATE, (long) maxOrdinate)
                .mapToObj(Double::valueOf)
                .append(GUARANTEES_TRAFFIC_VOLUMES)
                .sorted()
                .toList();
    }

}
