package ru.yandex.direct.core.entity.bids.interpolator;

import java.util.ArrayList;
import java.util.List;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bids.container.interpolator.Cap;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.MoneyUtils;
import ru.yandex.direct.utils.math.Point;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.Double.max;
import static java.lang.Double.min;
import static ru.yandex.direct.core.entity.bids.interpolator.InterpolatorUtils.checkPointsAreNonDecreasing;
import static ru.yandex.direct.core.entity.bids.interpolator.InterpolatorUtils.checkPointsAreSorted;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сглаживатель.
 * <p>
 * Создает дополнительные точки по уже известным.
 * <p>
 * Алгоритм.
 * <p>
 * Пусть S_1 -- изначальный массив ступенек. S_1 = (x_i, y_i), где i = [1, n].
 * x -- bid или amnesty_price, значение по оси абсцисс
 * y -- это traffic_volume(иногда используется название X, НЕ ПУТАТЬ С ОСЬЮ АБСЦИСС), значение по оси ординат.
 * <p>
 * D -- обозначим результат операции {@link #diffAndSmooth}.
 * <p>
 * Каждая точка из массива ступенек порождает m пар,
 * где m -- это размер {@link ru.yandex.direct.core.entity.bids.container.interpolator.Cap},
 * таких что: (x_i_j, y_i_j) = (max(x_i * (1 + step_j), min(x_i, minX)), (y_i - y_(i-1)) * d_j),
 * где step_j -- j-ый сдвиг в колпаке,
 * а d_j -- j-ая доля колпака от роста ступеньки
 * minX -- минимальная bid/amnesty_price
 * Отсортированный список по x, затем по y, этих точек является D.
 * <p>
 * I -- обозначим результат операции {@link #integrate}.
 * Элементом I будет являться` (x_i, y_i) = (x_i, Sum{j=1..i}{y_j}), где x_i и y_j элементы D.
 * В I добавляется элемент (0, 0)
 * <p>
 * S -- обозначим результат операции {@link #shift}.
 * Элементом S будет являться (x_i, y_i) = (x_i, y_i-1), где (x_i, y_i-1) элементы I.
 * <p>
 * удаляем одинаковые точки
 * <p>
 * {@link #removeNotBordersPointsOfConstantFunctionIntervals} -- из интервалов на которых функция == const удаляем
 * точки не являющиеся границами интервала
 */
public class Smoother {
    private final Cap cap;
    private final CurrencyCode currencyCode;
    private final double minX;
    private final double maxPrice;
    private final double auctionStep;

    /**
     * @param cap          -- колпак
     * @param currencyCode -- валюта
     * @param minX         -- минимальная абсцисса
     */
    public Smoother(Cap cap, CurrencyCode currencyCode, double minX) {
        this.cap = cap;
        this.currencyCode = currencyCode;
        this.minX = minX;
        this.maxPrice = currencyCode.getCurrency().getMaxPrice().doubleValue();
        this.auctionStep = currencyCode.getCurrency().getAuctionStep().doubleValue();
    }

    public List<Point> execute(List<Point> points) {
        checkArgument(checkPointsAreSorted(points), "Points should be sorted");
        checkArgument(checkPointsAreNonDecreasing(points), "Points should be non-decreasing");

        List<Point> diffedAndSmoothed = diffAndSmooth(points);
        List<Point> integrated = integrate(diffedAndSmoothed);
        List<Point> shifted = shift(integrated);
        List<Point> uniqueShifted = StreamEx.of(shifted)
                .distinct()
                .toList();
        return removeNotBordersPointsOfConstantFunctionIntervals(uniqueShifted);
    }

    /**
     * Для каждой из исходных точек получаем набор точек сдвинутых по оси абсцисс,
     * с ординатой являющейся долей исходной, сдвиг и доля определяется элементом
     * {@link ru.yandex.direct.core.entity.bids.container.interpolator.Cap}
     */
    List<Point> diffAndSmooth(List<Point> points) {
        List<Point> resultPoints = new ArrayList<>(points.size() * cap.size());
        for (int i = 0; i < points.size(); i++) {
            Point point = points.get(i);
            Point previousPoint = i == 0 ? Point.ZERO : points.get(i - 1);

            List<Point> diffedAndSmoothed = generateDiffAndSmoothPoints(point, previousPoint);

            resultPoints.addAll(diffedAndSmoothed);
        }
        resultPoints.sort(Point.COMPARATOR);
        return resultPoints;
    }

    List<Point> generateDiffAndSmoothPoints(Point point, Point previousPoint) {
        return mapList(cap.getPoints(), capPoint -> generateDiffAndSmoothPoint(point, previousPoint, capPoint));
    }

    /**
     * Считаем разницу между высотами ступеней, делим на части пропорциональные d_j'ым колпака
     * Считаем ширину ступени, используя step_j получаем сдвиги
     */
    Point generateDiffAndSmoothPoint(Point point, Point previousPoint, Point capPoint) {
        double x = min(max(point.getX() * (capPoint.getX() + 1.0), min(point.getX(), minX)), maxPrice);
        double y = (point.getY() - previousPoint.getY()) * (capPoint.getY());

        return Point.fromDoubles(x, y);
    }

    /**
     * Пересчитываем относительные величины в абсолютные
     */
    List<Point> integrate(List<Point> points) {
        List<Point> resultPoints = new ArrayList<>(points.size() + 1);

        resultPoints.add(Point.ZERO);
        double sum = 0.0;
        for (Point point : points) {
            sum += point.getY();
            double x = MoneyUtils.roundToAuctionStepUpDouble(point.getX(), auctionStep);
            resultPoints.add(Point.fromDoubles(x, Math.round(sum)));
        }
        return resultPoints;
    }

    /**
     * Вычисляем точки переломов
     */
    List<Point> shift(List<Point> points) {
        List<Point> resultPoints = new ArrayList<>(points.size());

        for (int i = 0; i < points.size() - 1; i++) {
            Point point = Point.fromDoubles(points.get(i).getX(), points.get(i + 1).getY());
            resultPoints.add(point);
        }
        resultPoints.add(points.get(points.size() - 1));
        return resultPoints;
    }

    /**
     * Оставляем только границы интервалов на которых функция является const
     */
    List<Point> removeNotBordersPointsOfConstantFunctionIntervals(List<Point> points) {
        List<Point> resultPoints = new ArrayList<>();
        resultPoints.add(points.get(0));
        for (int i = 1; i < points.size() - 1; i++) {
            Point point = points.get(i);
            Point previousPoint = points.get(i - 1);
            Point nextPoint = points.get(i + 1);
            boolean isConstantFunctionIntervalBorder =
                    point.getY() != previousPoint.getY() || point.getY() != nextPoint.getY();
            if (isConstantFunctionIntervalBorder) {
                resultPoints.add(point);
            }
        }
        if (points.size() > 1) {
            resultPoints.add(points.get(points.size() - 1));
        }
        return resultPoints;
    }
}
