package ru.yandex.direct.core.entity.bids.repository;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.SelectField;
import org.jooq.SelectForUpdateStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.ShowConditionStateSelection;
import ru.yandex.direct.core.entity.bids.container.ShowConditionStatusSelection;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.model.BidDynamicPrices;
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.keyword.model.ServingStatus;
import ru.yandex.direct.core.entity.keyword.repository.KeywordMapping;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BidsDynamicStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BidsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BidsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BidsWarn;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.Bids;
import ru.yandex.direct.dbschema.ppc.tables.BidsBase;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsBaseRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.select;
import static org.jooq.impl.DSL.selectCount;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_ARC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS;
import static ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.tables.BidsManualPrices.BIDS_MANUAL_PRICES;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperties;

@Repository
@ParametersAreNonnullByDefault
public class BidRepository {
    private static final Logger logger = LoggerFactory.getLogger(BidRepository.class);

    private static final int UPDATE_CHUNK_SIZE = 500;
    private static final Long AUTOBUDGET_PRIORITY_DEFAULT_VAL = 3L;
    protected static final int HEAVY_CAMP_BIDS_BORDER = 1000;

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Bid> jooqMapper;
    private final JooqMapperWithSupplier<Bid> jooqMapperForBidManual;
    private final JooqMapperWithSupplier<Bid> jooqMapperForBid;
    private final JooqMapperWithSupplier<BidDynamicPrices> jooqMapperForBidDynamicPrices;


    @Autowired
    public BidRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = buildBidJooqMapper();
        jooqMapperForBidManual = buildjooqMapperForBidManual();
        jooqMapperForBid = buildBidJooqMapperForBid();
        jooqMapperForBidDynamicPrices = buildJooqMapperForBidDynamicPrices();
    }

    @Nonnull
    private JooqMapperWithSupplier<Bid> buildBidJooqMapper() {
        // NOTE: при изменении маппинга необходимо поправить запрос в #getBids
        return JooqMapperWithSupplierBuilder.builder(Bid::new)
                .map(property(Bid.ID, BIDS_BASE.BID_ID))
                .map(property(Bid.CAMPAIGN_ID, BIDS_BASE.CID))
                .map(property(Bid.AD_GROUP_ID, BIDS_BASE.PID))
                .map(convertibleProperty(Bid.TYPE, BIDS_BASE.BID_TYPE,
                        ShowConditionType::fromBidsBaseBidType,
                        ShowConditionType::toBidsBaseBidType))
                .map(integerProperty(Bid.AUTOBUDGET_PRIORITY, BIDS_BASE.AUTOBUDGET_PRIORITY))
                .map(property(Bid.PRICE, BIDS_BASE.PRICE))
                .map(property(Bid.PRICE_CONTEXT, BIDS_BASE.PRICE_CONTEXT))
                .map(property(Bid.LAST_CHANGE, BIDS_BASE.LAST_CHANGE))
                .map(convertibleProperty(Bid.STATUS_BS_SYNCED, BIDS_BASE.STATUS_BS_SYNCED,
                        BidMappings::statusBsSyncedFromDbFormat,
                        BidMappings::statusBsSyncedToDbFormat))
                .readProperty(Bid.IS_SUSPENDED,
                        fromField(BIDS_BASE.OPTS).by(BidMappings::isSuspendedFromDbOpts))
                .readProperty(Bid.IS_DELETED,
                        fromField(BIDS_BASE.OPTS).by(BidMappings::isDeletedFromDbOpts))
                .writeField(BIDS_BASE.OPTS,
                        fromProperties(Bid.IS_SUSPENDED, Bid.IS_DELETED).by(BidMappings::bidPropsToDbOpts))
                .build();
    }

    @Nonnull
    private JooqMapperWithSupplier<Bid> buildjooqMapperForBidManual() {
        return JooqMapperWithSupplierBuilder.builder(Bid::new)
                .map(property(Bid.ID, BIDS_MANUAL_PRICES.ID))
                .map(property(Bid.CAMPAIGN_ID, BIDS_MANUAL_PRICES.CID))
                .map(property(Bid.AD_GROUP_ID, BIDS.PID))
                .map(property(Bid.PRICE, BIDS_MANUAL_PRICES.PRICE))
                .map(property(Bid.PRICE_CONTEXT, BIDS_MANUAL_PRICES.PRICE_CONTEXT))
                .build();
    }

    @Nonnull
    private JooqMapperWithSupplier<Bid> buildBidJooqMapperForBid() {
        return JooqMapperWithSupplierBuilder.builder(Bid::new)
                .map(property(Bid.ID, BIDS.ID))
                .map(property(Bid.CAMPAIGN_ID, BIDS.CID))
                .map(property(Bid.AD_GROUP_ID, BIDS.PID))
                .map(property(Bid.PRICE, BIDS.PRICE))
                .map(property(Bid.PRICE_CONTEXT, BIDS.PRICE_CONTEXT))
                .map(integerProperty(Bid.AUTOBUDGET_PRIORITY, BIDS.AUTOBUDGET_PRIORITY))
                .map(convertibleProperty(Bid.STATUS_BS_SYNCED, BIDS.STATUS_BS_SYNCED,
                        BidMappings::statusBsSyncedFromDbFormatForBid,
                        BidMappings::statusBsSyncedToDbFormatForBid))
                .build();
    }

    @Nonnull
    private JooqMapperWithSupplier<BidDynamicPrices> buildJooqMapperForBidDynamicPrices() {
        return JooqMapperWithSupplierBuilder.builder(BidDynamicPrices::new)
                .map(property(BidDynamicPrices.AD_GROUP_ID, BIDS_DYNAMIC.PID))
                .map(property(BidDynamicPrices.ID, BIDS_DYNAMIC.DYN_ID))
                .map(property(BidDynamicPrices.PRICE, BIDS_DYNAMIC.PRICE))
                .map(property(BidDynamicPrices.PRICE_CONTEXT, BIDS_DYNAMIC.PRICE_CONTEXT))
                .readProperty(BidDynamicPrices.CONTEXT_PRICE_COEF, fromField(CAMPAIGNS.CONTEXT_PRICE_COEF))
                .build();
    }

    /**
     * Добавляет бесфразный таргетинг в таблицу bids_base
     */
    public void addBids(DSLContext context, Collection<Bid> bids) {
        new InsertHelper<>(context, BIDS_BASE)
                .addAll(jooqMapper, bids)
                .executeIfRecordsAdded();
    }

    /**
     * Получает объекты Bid из таблицы bids_base c типом RELEVANCE_MATCH для всех кампании переданных по id в {@code campaignIds}
     */
    public List<Bid> getRelevanceMatchByCampaignIdsNotDeleted(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_BASE)
                .where(BIDS_BASE.CID.in(campaignIds))
                .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match_search)
                        .or(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match)))
                .andNot(setField(BIDS_BASE.OPTS).contains(BidBaseOpt.DELETED.getTypedValue()))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Получает объекты Bid из таблицы bids_base c типом RELEVANCE_MATCH для переданных id в {@code ids}
     */
    public List<Bid> getRelevanceMatchByIdsNotDeleted(int shard, List<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_BASE)
                .where(BIDS_BASE.BID_ID.in(ids))
                .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match_search)
                        .or(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match)))
                .andNot(setField(BIDS_BASE.OPTS).contains(BidBaseOpt.DELETED.getTypedValue()))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Сбрасывает STATUS_BS_SYNCED у бесфразного таргетинга (relevanceMatches) по {@param ids}
     */
    public void resetBidsBaseSyncedQuet(int shard, List<Long> ids) {
        dslContextProvider.ppc(shard).update(BIDS_BASE)
                .set(BIDS_BASE.STATUS_BS_SYNCED, BidsBaseStatusbssynced.No)
                .set(BIDS_BASE.LAST_CHANGE, BIDS_BASE.LAST_CHANGE)
                .where(BIDS_BASE.BID_ID.in(ids))
                .execute();
    }

    public void resetBidsSynced(int shard, List<Long> cids) {
        dslContextProvider.ppc(shard).update(BIDS)
                .set(BIDS.STATUS_BS_SYNCED, BidsStatusbssynced.No)
                .where(BIDS.CID.in(cids))
                .execute();
    }

    /**
     * Получает бесфразный таргетинг bid из таблицы bids_base по их id
     */
    public List<Bid> getRelevanceMatchByIds(int shard, List<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_BASE)
                .where(BIDS_BASE.BID_ID.in(ids))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Получает объекты Bid из таблицы bids для всех кампании переданных по id в {@code campaignIds}
     */
    public List<Bid> getBidsByCampaignIds(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapperForBid.getFieldsToRead())
                .from(BIDS)
                .where(BIDS.CID.in(campaignIds))
                .fetch(jooqMapperForBid::fromDb);
    }

    /**
     * Возвращает ставки по перенному {@code selection}. Ставки берутся из таблиц
     * {@link ru.yandex.direct.dbschema.ppc.tables.BidsBase} и {@link ru.yandex.direct.dbschema.ppc.tables.Bids}.
     * <p>
     * Порядок сортировки элементов в ответе: по ID.
     */
    public List<Bid> getBids(int shard, ShowConditionSelectionCriteria selection, LimitOffset limitOffset) {
        if (selection.getShowConditionIds().isEmpty() && selection.getAdGroupIds().isEmpty() && selection
                .getCampaignIds().isEmpty()) {
            return Collections.emptyList();
        }

        Set<ServingStatus> bsRarelyLoaded = selection.getServingStatuses();
        Set<Long> bsRarelyLoadedDbValues =
                StreamEx.of(bsRarelyLoaded).filter(Objects::nonNull).map(ServingStatus::dbValue).toSet();

        SelectForUpdateStep<Record> select = dslContextProvider.ppc(shard)
                .select(new SelectField[]{BIDS_BASE.BID_ID, BIDS_BASE.CID, BIDS_BASE.PID, BIDS_BASE.BID_TYPE,
                        BIDS_BASE.AUTOBUDGET_PRIORITY, BIDS_BASE.PRICE, BIDS_BASE.PRICE_CONTEXT,
                        BIDS_BASE.STATUS_BS_SYNCED,
                        BIDS_BASE.OPTS, DSL.castNull(Long.class).as(BIDS.PLACE)}
                )
                .from(BIDS_BASE)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_BASE.PID))
                .where(selection.getCampaignIds().isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID)
                        : BIDS_BASE.CID.in(selection.getCampaignIds()))
                .and(selection.getShowConditionIds().isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID)
                        : BIDS_BASE.BID_ID.in(selection.getShowConditionIds()))
                .and(selection.getAdGroupIds().isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID)
                        : BIDS_BASE.PID.in(selection.getAdGroupIds()))
                // записи keyword получаем из bids, поэтому тут пропускаем
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .and(bsRarelyLoadedDbValues.isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID)
                        : PHRASES.IS_BS_RARELY_LOADED.in(bsRarelyLoadedDbValues))
                .and(convertRelevanceMatchStatesToCondition(selection))
                .and(convertRelevanceMatchStatusesToCondition(selection))
                .and(selection.getModifiedSince() == null ? DSL.trueCondition()
                        : BIDS_BASE.LAST_CHANGE.greaterOrEqual(selection.getModifiedSince()))
                // не отображаем удалённые
                .andNot(setField(BIDS_BASE.OPTS).contains(BidBaseOpt.DELETED.getTypedValue()))
                .unionAll(
                        select(BIDS.ID, BIDS.CID, BIDS.PID, DSL.inline(BidsBaseBidType.keyword),
                                BIDS.AUTOBUDGET_PRIORITY, BIDS.PRICE, BIDS.PRICE_CONTEXT,
                                BIDS.STATUS_BS_SYNCED,
                                DSL.when(BIDS.IS_SUSPENDED.eq(1L), BidBaseOpt.SUSPENDED.getTypedValue()),
                                BIDS.PLACE)
                                .from(BIDS)
                                .join(PHRASES).on(PHRASES.PID.eq(BIDS.PID))
                                .where(selection.getCampaignIds().isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : BIDS.CID.in(selection.getCampaignIds()))
                                .and(selection.getShowConditionIds().isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : BIDS.ID.in(selection.getShowConditionIds()))
                                .and(selection.getAdGroupIds().isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : BIDS.PID.in(selection.getAdGroupIds()))
                                .and(selection.getStates().isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : convertKeywordStatesToCondition(selection.getStates()))
                                .and(selection.getStatuses().isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : convertKeywordStatusesToCondition(selection.getStatuses()))
                                .and(selection.getModifiedSince() == null ? BIDS.ID.eq(BIDS.ID)
                                        : BIDS.MODTIME.greaterOrEqual(selection.getModifiedSince()))
                                .and(bsRarelyLoadedDbValues.isEmpty() ? BIDS.ID.eq(BIDS.ID)
                                        : PHRASES.IS_BS_RARELY_LOADED.in(bsRarelyLoadedDbValues))
                )
                .orderBy(1) // сортировка по ID
                .limit(limitOffset.limit())
                .offset(limitOffset.offset());
        return select
                .fetch().stream().map(jooqMapper::fromDb).collect(toList());
    }

    private static Condition convertKeywordStatesToCondition(Set<ShowConditionStateSelection> states) {
        List<Condition> conditions = new ArrayList<>();

        for (ShowConditionStateSelection state : states) {
            switch (state) {
                case SUSPENDED:
                    conditions.add(BIDS.IS_SUSPENDED.eq(1L));
                    break;
                case ON:
                    conditions.add(BIDS.IS_SUSPENDED.eq(0L)
                            .and(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.Yes)));
                    break;
                case OFF:
                    conditions.add(BIDS.IS_SUSPENDED.eq(0L)
                            .and(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.New)));

                    conditions.add(BIDS.IS_SUSPENDED.eq(0L)
                            .and(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.No)));
                    break;
                default:
                    throw new IllegalArgumentException("Not supported state: " + state);
            }
        }

        return conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException("Can not build conditions for selection by states!"));
    }

    private static Condition convertKeywordStatusesToCondition(Set<ShowConditionStatusSelection> statuses) {
        List<Condition> conditions = new ArrayList<>();

        for (ShowConditionStatusSelection status : statuses) {
            switch (status) {
                case DRAFT:
                    conditions.add(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.New));
                    break;
                case ACCEPTED:
                    conditions.add(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.Yes));
                    break;
                case REJECTED:
                    conditions.add(BIDS.STATUS_MODERATE.eq(BidsStatusmoderate.No));
                    break;
            }
        }

        return conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException("Can not build conditions for selection by statuses!"));
    }

    private static Condition convertRelevanceMatchStatesToCondition(ShowConditionSelectionCriteria selection) {
        return selection.getStates().isEmpty() ? DSL.trueCondition()
                : DSL.or(
                selection.getStates().contains(ShowConditionStateSelection.SUSPENDED) ?
                        BidsBase.BIDS_BASE.OPTS.contains(BidBaseOpt.SUSPENDED.getTypedValue())
                        : DSL.falseCondition(),
                selection.getStates().contains(ShowConditionStateSelection.ON) ?
                        BidsBase.BIDS_BASE.OPTS.notContains(BidBaseOpt.SUSPENDED.getTypedValue())
                        : DSL.falseCondition());
    }

    private static Condition convertRelevanceMatchStatusesToCondition(ShowConditionSelectionCriteria selection) {
        if (selection.getStatuses().isEmpty() || selection.getStatuses()
                .contains(ShowConditionStatusSelection.ACCEPTED)) {
            return DSL.trueCondition();
        } else {
            return DSL.falseCondition();
        }
    }


    /**
     * Получает набор уникальных {@code campaignId} по переданным {@code bidIds} и {@code adGroupIds}
     */
    public List<Long> getCampaignIdsForBids(int shard, Collection<Long> bidIds, Collection<Long> adGroupIds) {
        if (bidIds.isEmpty() && adGroupIds.isEmpty()) {
            return Collections.emptyList();
        }

        return dslContextProvider.ppc(shard)
                .selectDistinct(BIDS_BASE.CID)
                .from(BIDS_BASE)
                .where(bidIds.isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID) : BIDS_BASE.BID_ID.in(bidIds))
                .and(adGroupIds.isEmpty() ? BIDS_BASE.BID_ID.eq(BIDS_BASE.BID_ID) : BIDS_BASE.PID.in(adGroupIds))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .unionAll(select(BIDS.CID)
                        .from(BIDS)
                        .where(bidIds.isEmpty() ? Tables.BIDS.ID.eq(Tables.BIDS.ID) : Tables.BIDS.ID.in(bidIds))
                        .and(adGroupIds.isEmpty() ? Tables.BIDS.ID.eq(Tables.BIDS.ID) : Tables.BIDS.PID.in(adGroupIds)))
                .fetch().stream().map(Record1::value1).collect(toList());
    }

    public int setBidsInBidsBase(int shard, List<AppliedChanges<Bid>> changes) {
        int changed = 0;
        for (List<AppliedChanges<Bid>> chunk : Lists.partition(changes, UPDATE_CHUNK_SIZE)) {
            JooqUpdateBuilder<BidsBaseRecord, Bid> ub =
                    new JooqUpdateBuilder<>(BidsBase.BIDS_BASE.BID_ID, chunk);
            ub.processProperty(Bid.AUTOBUDGET_PRIORITY, BidsBase.BIDS_BASE.AUTOBUDGET_PRIORITY, Integer::longValue);
            ub.processProperty(Bid.PRICE, BidsBase.BIDS_BASE.PRICE);
            ub.processProperty(Bid.PRICE_CONTEXT, BidsBase.BIDS_BASE.PRICE_CONTEXT);
            ub.processProperty(Bid.LAST_CHANGE, BidsBase.BIDS_BASE.LAST_CHANGE, Function.identity());
            ub.processProperty(Bid.STATUS_BS_SYNCED, BidsBase.BIDS_BASE.STATUS_BS_SYNCED,
                    status -> BidsBaseStatusbssynced.valueOf(status.toDbFormat()));

            dslContextProvider.ppc(shard)
                    .update(BidsBase.BIDS_BASE)
                    .set(ub.getValues())
                    .where(BidsBase.BIDS_BASE.BID_ID.in(ub.getChangedIds()))
                    .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                    .execute();
            changed += ub.getChangedIds().size();
        }
        return changed;
    }

    public int setBidsInBids(int shard, List<AppliedChanges<Keyword>> changes) {
        int changed = 0;
        for (List<AppliedChanges<Keyword>> chunk : Lists.partition(changes, UPDATE_CHUNK_SIZE)) {
            JooqUpdateBuilder<BidsRecord, Keyword> ub =
                    new JooqUpdateBuilder<>(Bids.BIDS.ID, chunk);
            ub.processProperty(Keyword.AUTOBUDGET_PRIORITY, Bids.BIDS.AUTOBUDGET_PRIORITY, Integer::longValue);
            ub.processProperty(Keyword.PRICE, Bids.BIDS.PRICE);
            ub.processProperty(Keyword.PRICE_CONTEXT, Bids.BIDS.PRICE_CONTEXT);
            ub.processProperty(Keyword.MODIFICATION_TIME, Bids.BIDS.MODTIME);
            ub.processProperty(Keyword.STATUS_BS_SYNCED, Bids.BIDS.STATUS_BS_SYNCED,
                    status -> BidsStatusbssynced.valueOf(status.toDbFormat()));
            //по хорошему, это поле должно быть в Keyword и обновляться отдельно. Однако, и Keyword и Bid лежат в bids,
            //поэтому, чтобы сделать апдейт за 1н запрос, поле находится в Bid
            ub.processProperty(Keyword.NEED_CHECK_PLACE_MODIFIED, Bids.BIDS.WARN,
                    KeywordMapping::needCheckPlaceModifiedToDbFormat);
            ub.processProperty(Keyword.PLACE, Bids.BIDS.PLACE, Place::convertToDb);

            dslContextProvider.ppc(shard)
                    .update(Bids.BIDS)
                    .set(ub.getValues())
                    .where(Bids.BIDS.ID.in(ub.getChangedIds()))
                    .execute();
            changed += ub.getChangedIds().size();
        }
        return changed;
    }

    /**
     * Вписывает в таблицу BIDS_MANUAL_PRICES все объекты Bid из {@code bids}
     */
    public void insertBidsToBidsManualPrices(int shard, List<Bid> bids) {
        bids.forEach(bid -> {
            dslContextProvider.ppc(shard).insertInto(
                    BIDS_MANUAL_PRICES,
                    BIDS_MANUAL_PRICES.ID,
                    BIDS_MANUAL_PRICES.CID,
                    BIDS_MANUAL_PRICES.PRICE,
                    BIDS_MANUAL_PRICES.PRICE_CONTEXT
            ).values(
                    bid.getId(),
                    bid.getCampaignId(),
                    bid.getPrice(),
                    bid.getPriceContext()
            ).execute();
        });
    }

    /**
     * Вписывает в таблицу BIDS_MANUAL_PRICES все объекты Bid из {@code bids}
     */
    public List<Bid> getBidsFromBidsManualPricesByCampaignIds(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard).select(
                BIDS_MANUAL_PRICES.ID,
                BIDS_MANUAL_PRICES.PRICE,
                BIDS_MANUAL_PRICES.PRICE_CONTEXT,
                BIDS_MANUAL_PRICES.CID)
                .from(BIDS_MANUAL_PRICES)
                .where(BIDS_MANUAL_PRICES.CID.in(campaignIds))
                .fetch(jooqMapperForBidManual::fromDb);
    }

    /**
     * Все ставки которые есть в таблице BIDS переносим в BIDS_MANUAL_PRICES для переданных по id кампании
     */
    public void copyFromBidsToBidsManualPricesForCampaignIds(int shard, List<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .insertInto(
                        BIDS_MANUAL_PRICES,
                        BIDS_MANUAL_PRICES.ID,
                        BIDS_MANUAL_PRICES.PRICE,
                        BIDS_MANUAL_PRICES.PRICE_CONTEXT,
                        BIDS_MANUAL_PRICES.CID
                ).select(select(
                BIDS.ID,
                BIDS.PRICE,
                BIDS.PRICE_CONTEXT,
                BIDS.CID
        ).from(BIDS).where(BIDS.CID.in(campaignIds)))
                .onDuplicateKeyUpdate()
                .set(BIDS_MANUAL_PRICES.PRICE_CONTEXT, BIDS.PRICE_CONTEXT)
                .set(BIDS_MANUAL_PRICES.PRICE, BIDS.PRICE)
                .execute();
    }

    /**
     * Получает объекты Bid из таблицы BIDS_MANUAL_PRICES, у которых есть соответствующая ставка в таблице BIDS,
     * для всех кампании переданных по id в {@code campaignIds}
     */
    public List<Bid> getBidsManualPricesForCampaignIds(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_MANUAL_PRICES.ID,
                        BIDS_MANUAL_PRICES.CID,
                        BIDS_MANUAL_PRICES.PRICE,
                        BIDS_MANUAL_PRICES.PRICE_CONTEXT)
                .from(BIDS_MANUAL_PRICES)
                .join(BIDS)
                .on(
                        BIDS_MANUAL_PRICES.ID.eq(BIDS.ID)
                                .and(BIDS_MANUAL_PRICES.CID.eq(BIDS.CID))
                )
                .where(BIDS_MANUAL_PRICES.CID.in(campaignIds))
                .fetch().stream().map(jooqMapperForBidManual::fromDb).collect(toList());
    }

    public void deleteBidManualPricesForCampaignIds(int shard, List<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .delete(BIDS_MANUAL_PRICES)
                .where(BIDS_MANUAL_PRICES.CID.in(campaignIds))
                .execute();
    }

    public List<Bid> getBidsWithRelevanceMatchByCampaignIds(int shard, List<Long> campaignIds) {
        List<Bid> bids = getBidsByCampaignIds(shard, campaignIds);
        bids.forEach(b -> b.setType(ShowConditionType.KEYWORD));
        bids.addAll(dslContextProvider.ppc(shard).select(
                jooqMapper.getFieldsToRead())
                .from(BIDS_BASE)
                .where(BIDS_BASE.CID.in(campaignIds))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .fetch(jooqMapper::fromDb));
        return bids;
    }

    public void resetPriceContextToZeroForBidsBaseByCampaignIds(int shard, List<Long> campaignIds) {
        dslContextProvider.ppc(shard).update(BIDS_BASE)
                .set(BIDS_BASE.PRICE_CONTEXT, BigDecimal.ZERO)
                .set(BIDS_BASE.STATUS_BS_SYNCED, BidsBaseStatusbssynced.No)
                .where(BIDS_BASE.CID.in(campaignIds))
                .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                .execute();
    }

    public void resetPriceContextToZeroForBidsByCampaignIds(int shard, List<Long> campaignIds) {
        dslContextProvider.ppc(shard).update(BIDS)
                .set(BIDS.PRICE_CONTEXT, BigDecimal.ZERO)
                .set(BIDS.STATUS_BS_SYNCED, BidsStatusbssynced.No)
                .set(BIDS.WARN, BidsWarn.Yes)
                .where(BIDS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Устанавливает приоритет выставленный для данного условия показа после включения автобюджета
     */
    public void setAutobudgetPriorityAutobudget(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        DSLContext context = dslContextProvider.ppc(shard);
        context.update(BIDS)
                .set(BIDS.AUTOBUDGET_PRIORITY, AUTOBUDGET_PRIORITY_DEFAULT_VAL)
                .where(BIDS.CID.in(campaignIds)
                        .and(BIDS.AUTOBUDGET_PRIORITY.isNull()))
                .execute();
        //Так же выставляем autobudgetPriority для беcфразного таргетинга (исключая удаленный)
        context.update(BIDS_BASE)
                .set(BIDS_BASE.AUTOBUDGET_PRIORITY, AUTOBUDGET_PRIORITY_DEFAULT_VAL)
                .where(BIDS_BASE.CID.in(campaignIds)
                        .and(BIDS_BASE.AUTOBUDGET_PRIORITY.isNull())
                        .and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword))
                        .and(BIDS_BASE.OPTS.notEqual("deleted")))
                .execute();
    }

    /**
     * Проставить условиям показа статус модерации statusModerate = "Yes".
     */
    public void markBidsAsModerated(int shard, Collection<Long> bidIds) {
        dslContextProvider.ppc(shard).update(BIDS)
                .set(BIDS.STATUS_MODERATE, BidsStatusmoderate.Yes)
                .where(BIDS.ID.in(bidIds))
                .execute();
    }

    public Map<Long, List<BidDynamicPrices>> getBidsDynamicPricesByCampaignIds(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .select(jooqMapperForBidDynamicPrices.getFieldsToRead())
                .from(BIDS_DYNAMIC)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_DYNAMIC.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.dynamic))
                .fetchGroups(CAMPAIGNS.CID, jooqMapperForBidDynamicPrices::fromDb);
    }

    public void resetBidsDynamicBsStatusAndPriority(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(BIDS_DYNAMIC.join(Tables.PHRASES).using(BIDS_DYNAMIC.PID))
                .set(BIDS_DYNAMIC.STATUS_BS_SYNCED, BidsDynamicStatusbssynced.No)
                .set(BIDS_DYNAMIC.AUTOBUDGET_PRIORITY, 3L)
                .where(BIDS_DYNAMIC.AUTOBUDGET_PRIORITY.isNull())
                .and(PHRASES.CID.in(campaignIds))
                .execute();
    }

    public List<Long> getBidIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS.ID)
                .from(BIDS)
                .where(BIDS.CID.in(campaignIds))
                .fetch()
                .map(record -> record.get(BIDS.ID));
    }

    public List<Long> getHeavyCampaignIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(selectCount().from(BIDS_ARC).where(BIDS_ARC.CID.eq(CAMPAIGNS.CID)).asField()
                        .plus(selectCount().from(BIDS).where(BIDS.CID.eq(CAMPAIGNS.CID)).asField())
                        .greaterThan(HEAVY_CAMP_BIDS_BORDER))
                .fetch(CAMPAIGNS.CID);
    }

    public int updateBidsDynamicPrices(int shard, List<AppliedChanges<BidDynamicPrices>> changes) {
        return updateBidsDynamicPrices(dslContextProvider.ppc(shard), changes);
    }

    public int updateBidsDynamicPrices(DSLContext context, List<AppliedChanges<BidDynamicPrices>> changes) {
        return StreamEx.ofSubLists(changes, UPDATE_CHUNK_SIZE)
                .map(chunk -> new UpdateHelper<>(context, BIDS_DYNAMIC.DYN_ID)
                        .processUpdateAll(jooqMapperForBidDynamicPrices, chunk)
                        .execute())
                .foldLeft(0, Integer::sum);
    }
}
