package ru.yandex.direct.core.entity.bids.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.auction.container.bs.BsDataWithKeyword;
import ru.yandex.direct.core.entity.bids.container.BidTargetType;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetBidSelectionType;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.utils.BidStorage;
import ru.yandex.direct.core.entity.bids.utils.autoprice.PriceWizard;
import ru.yandex.direct.core.entity.bids.utils.autoprice.PriceWizardHelper;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.bids.container.BidUtils.containsSearch;
import static ru.yandex.direct.core.entity.bids.container.ShowConditionType.KEYWORD;
import static ru.yandex.direct.core.entity.bids.container.ShowConditionType.RELEVANCE_MATCH;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdateContextPrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdatePrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.detectSelectionType;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.groupBidsBySelectionCriteria;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

abstract class AbstractBidSetAutoPriceApplier<T extends BsDataWithKeyword> implements BidSetAutoPriceApplier {
    private static final Logger logger = LoggerFactory.getLogger(AbstractBidSetAutoPriceApplier.class);

    private final Map<SetAutoBidItem, List<Bid>> bidsForSetAuto;
    private final Map<Long, Keyword> keywordsById;
    private final BidStorage<Bid> bidStorage;
    final Map<Long, CompleteBidData<T>> completeBidDataById;

    AbstractBidSetAutoPriceApplier(List<SetAutoBidItem> setAutoBidItems,
                                   Collection<CompleteBidData<T>> completeBidData,
                                   Collection<Keyword> keywords) {
        bidStorage = new BidStorage<>(mapList(completeBidData, CompleteBidData::getBid));
        bidsForSetAuto =
                groupBidsBySelectionCriteria(setAutoBidItems, mapList(completeBidData, CompleteBidData::getBid));
        validateSetAutoBidRequests(bidsForSetAuto);
        completeBidDataById = StreamEx.of(completeBidData).toMap(CompleteBidData::getBidId, Function.identity());
        this.keywordsById = listToMap(keywords, Keyword::getId);
    }

    /**
     * Генерирует исключение если переданные параметры невалидны для расчёта ставок.
     * Инварианты после проверки:
     * <ul>
     * <li>список {@link CompleteBidData} в {@link #bidsForSetAuto} непустой для всех {@link SetAutoBidItem}</li>
     * <li>нет заявок на обновление relevanceMatch ставки напрямую по ID (допускается только обновление через группу/кампанию)</li>
     * </ul>
     */
    private void validateSetAutoBidRequests(Map<SetAutoBidItem, List<Bid>> bidsForSetAuto) {
        for (Map.Entry<SetAutoBidItem, List<Bid>> entry : bidsForSetAuto.entrySet()) {
            SetAutoBidItem setAutoBidItem = entry.getKey();
            List<Bid> bids = entry.getValue();

            // Предварительные проверки. Хотим понятные тексты ошибок, если вдруг что-то пойдёт не так
            checkArgument(bids != null && !bids.isEmpty(),
                    "Can't find existing bids for modification described by %s", setAutoBidItem);

            SetBidSelectionType setBidSelectionType = detectSelectionType(setAutoBidItem);
            Bid firstBid = bids.get(0);
            checkArgument(!(setBidSelectionType == SetBidSelectionType.KEYWORD_ID
                            && firstBid.getType() == ShowConditionType.RELEVANCE_MATCH),
                    "Can't update RelevanceMatch bid in setAuto by KeywordId. "
                            + "It should be updated by AdGroupId of CampaignID. SetAutoBidItem: %s", setAutoBidItem);
        }
    }

    /**
     * Вычисляет новые значения ставок для автотаргетинга и применяет их к существующим {@link Bid}
     *
     * @return список {@link AppliedChanges} &ndash; изменений, применённых к ставкам
     */
    public List<AppliedChanges<Bid>> calcNewPriceForRelevanceMatchAndApply(
            List<AppliedChanges<Keyword>> keywordChanges) {
        ArrayList<AppliedChanges<Bid>> actualChanges = new ArrayList<>();
        for (Map.Entry<SetAutoBidItem, List<Bid>> entry : bidsForSetAuto.entrySet()) {
            SetAutoBidItem setAutoBidItem = entry.getKey();
            List<Bid> bids = entry.getValue();
            // одному setBidItem'у соответствуют ставки из одной кампании. Берём данные о валюте у первого элемента bids
            Bid firstBid = bids.get(0);
            Campaign campaign = completeBidDataById.get(firstBid.getId()).getCampaign();
            DbStrategy strategy = campaign.getStrategy();
            CurrencyCode currencyCode = campaign.getCurrency();
            PriceWizardHelper wizardHelper = new PriceWizardHelper(setAutoBidItem, currencyCode);

            for (Bid bid : bids) {
                if (!bid.getType().equals(RELEVANCE_MATCH)) {
                    continue;
                }
                ModelChanges<BidBase> changes = new ModelChanges<>(bid.getId(), BidBase.class);
                EnumSet<BidTargetType> scopes = setAutoBidItem.getScope();

                Map<Long, BidBase> keywordChangesBydId =
                        listToMap(keywordChanges, t -> t.getModel().getId(), AppliedChanges::getModel);
                //Для автотаргетинга ставка вычисляется как 30-ый перцентиль цен в группе
                Long adGroupId = bid.getAdGroupId();
                List<BidBase> phraseBids = StreamEx.of(bidStorage.getByAdGroupId(adGroupId))
                        .remove(b -> b.getType().equals(ShowConditionType.RELEVANCE_MATCH))
                        .select(BidBase.class)
                        .toList();

                //для вычисления ставки автотаргетинга необходимо использовать новые значения цен, поэтому берём их из keywordChanges
                List<BidBase> bidsUpdated = new ArrayList<>(phraseBids.size());
                for (BidBase phraseBid : phraseBids) {
                    bidsUpdated.add(keywordChangesBydId.getOrDefault(phraseBid.getId(), phraseBid));
                }

                if (containsSearch(scopes) && canUpdatePrice(strategy, RELEVANCE_MATCH)) {
                    calcNewSearchPriceForRelevanceMatch(changes, bidsUpdated, wizardHelper);
                }
                if (scopes.contains(BidTargetType.CONTEXT) && canUpdateContextPrice(strategy, RELEVANCE_MATCH)) {
                    calcNewContextPriceForRelevanceMatch(changes, bidsUpdated, wizardHelper);
                }

                AppliedChanges<Bid> priceChange = changes.applyTo(bid);
                if (!priceChange.getActuallyChangedProps().isEmpty()) {
                    actualChanges.add(priceChange);
                }
            }
        }
        return actualChanges;
    }

    /**
     * Вычисляет новые значения ставок для ключевых фраз и применяет их к существующим {@link Bid}
     *
     * @return список {@link AppliedChanges} &ndash; изменений, применённых к ставкам
     */
    public List<AppliedChanges<Keyword>> calcNewPriceForKeywordAndApply() {
        ArrayList<AppliedChanges<Keyword>> actualChanges = new ArrayList<>();
        for (Map.Entry<SetAutoBidItem, List<Bid>> entry : bidsForSetAuto.entrySet()) {
            SetAutoBidItem setAutoBidItem = entry.getKey();
            List<Bid> bids = entry.getValue();
            // одному setBidItem'у соответствуют ставки из одной кампании. Берём данные о валюте у первого элемента bids
            Bid firstBid = bids.get(0);
            Campaign campaign = completeBidDataById.get(firstBid.getId()).getCampaign();
            DbStrategy strategy = campaign.getStrategy();
            CurrencyCode currencyCode = campaign.getCurrency();
            PriceWizardHelper wizardHelper = new PriceWizardHelper(setAutoBidItem, currencyCode);

            for (Bid bid : bids) {
                if (!bid.getType().equals(KEYWORD)) {
                    continue;
                }

                Long id = bid.getId();
                Keyword keyword = keywordsById.get(id);
                if (keyword == null) {
                    // логируем с уровнем error, так как пока не ясно, что в действительности привело к потере ключевой фразы (DIRECT-70620)
                    logger.error("Keyword with id {} is missed. Skip price change", id);
                    continue;
                }
                ModelChanges<Keyword> changes = new ModelChanges<>(id, Keyword.class);
                EnumSet<BidTargetType> scopes = setAutoBidItem.getScope();
                if (containsSearch(scopes)
                        && canUpdatePrice(strategy, KEYWORD)) {
                    calcNewSearchPriceForKeyword(changes, wizardHelper);
                }
                if (scopes.contains(BidTargetType.CONTEXT)
                        && canUpdateContextPrice(strategy, KEYWORD)) {
                    calcNewContextPriceForKeyword(changes, wizardHelper);
                }

                AppliedChanges<Keyword> priceChange = changes.applyTo(keyword);
                if (!priceChange.getActuallyChangedProps().isEmpty()) {
                    actualChanges.add(priceChange);
                }
            }
        }
        return actualChanges;
    }

    protected abstract void calcNewSearchPriceForKeyword(ModelChanges<? extends BidBase> changes,
                                                         PriceWizardHelper wizardHelper);

    private void calcNewContextPriceForKeyword(ModelChanges<? extends BidBase> changes,
                                               PriceWizardHelper wizardHelper) {
        Long bidId = changes.getId();
        CompleteBidData<T> completeBidDataItem = completeBidDataById.get(bidId);
        KeywordBidDynamicData<T> bidDynamicData = completeBidDataItem.getDynamicData();
        KeywordBidPokazometerData pokazometerData = bidDynamicData.getPokazometerData();
        if (pokazometerData == null || pokazometerData.getCoverageWithPrices().isEmpty()) {
            logger.debug("No pokazometer data for keyword (id={}). No contextPrice update", bidId);
            return;
        }
        PriceWizard<KeywordBidPokazometerData> roundedContextPriceWizard =
                wizardHelper.getRoundedContextPriceWizard();
        BigDecimal newContextPrice = roundedContextPriceWizard.calcPrice(pokazometerData);
        changes.processNotNull(newContextPrice, BidBase.PRICE_CONTEXT);
    }

    private void calcNewSearchPriceForRelevanceMatch(ModelChanges<BidBase> changes,
                                                     List<BidBase> updatedBids, PriceWizardHelper wizardHelper) {
        if (!updatedBids.isEmpty()) {
            PriceWizard<Collection<BidBase>> roundedRelevanceMatchPriceWizard =
                    wizardHelper.getRoundedRelevanceMatchSearchPriceWizard();
            // Обновляем, если есть фразы в группе
            BigDecimal newContextPrice = roundedRelevanceMatchPriceWizard.calcPrice(updatedBids);
            changes.processNotNull(newContextPrice, BidBase.PRICE);
        } else {
            logger.info("No Keywords in adGroup. Skip updating relevanceMatch");
        }
    }

    private void calcNewContextPriceForRelevanceMatch(ModelChanges<BidBase> changes,
                                                      List<BidBase> updatedBids, PriceWizardHelper wizardHelper) {
        if (!updatedBids.isEmpty()) {
            PriceWizard<Collection<BidBase>> roundedRelevanceMatchPriceWizard =
                    wizardHelper.getRoundedRelevanceMatchContextPriceWizard();
            // Обновляем, если есть фразы в группе
            BigDecimal newContextPrice = roundedRelevanceMatchPriceWizard.calcPrice(updatedBids);
            changes.processNotNull(newContextPrice, BidBase.PRICE_CONTEXT);
        } else {
            logger.info("No Keywords in adGroup. Skip updating relevanceMatch");
        }
    }

    @Nonnull
    private Bid getBidById(Long bidId) throws IllegalArgumentException {
        return bidStorage.getByBidId(bidId)
                .orElseThrow(() -> new IllegalArgumentException("Can't find Bid by ID " + bidId));
    }
}
