package ru.yandex.direct.core.entity.bids.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.autoprice.service.AutoPriceCampQueueService;
import ru.yandex.direct.core.entity.bids.container.BidTargetType;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.container.SetPhrasesInitialPricesOption;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.exception.CantSetBidException;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.bids.utils.BidStorage;
import ru.yandex.direct.core.entity.campaign.container.CampaignStrategyChangingSettings;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.mailnotification.model.GenericEvent;
import ru.yandex.direct.core.entity.mailnotification.model.KeywordEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.OperationMeta;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria.fromShowConditionList;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdateContextPrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdatePrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdatePriority;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.min;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.math.MathUtils.max;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class BidService {
    private static final Logger logger = LoggerFactory.getLogger(BidService.class);

    private final BidRepository bidRepository;
    private final ShardHelper shardHelper;
    private final KeywordService keywordService;
    private final SetAutoBidsValidationService setAutoBidsValidationService;
    private final BidsSetValidationService bidsSetValidationService;
    private final KeywordBidDynamicDataService keywordBidDynamicDataService;
    private final BidBsStatisticFacade bidBsStatisticFacade;
    private final MailNotificationEventService mailNotificationEventService;
    private final LogPriceService logPriceService;
    private final BidValidationService bidValidationService;
    private final AutoPriceCampQueueService autoPriceCampQueueService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    @Autowired
    public BidService(BidRepository bidRepository,
                      ShardHelper shardHelper,
                      KeywordService keywordService,
                      SetAutoBidsValidationService setAutoBidsValidationService,
                      BidsSetValidationService bidsSetValidationService,
                      KeywordBidDynamicDataService keywordBidDynamicDataService,
                      BidBsStatisticFacade bidBsStatisticFacade,
                      MailNotificationEventService mailNotificationEventService,
                      LogPriceService logPriceService,
                      BidValidationService bidValidationService,
                      AutoPriceCampQueueService autoPriceCampQueueService,
                      CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.bidRepository = bidRepository;
        this.shardHelper = shardHelper;
        this.keywordService = keywordService;
        this.setAutoBidsValidationService = setAutoBidsValidationService;
        this.bidsSetValidationService = bidsSetValidationService;
        this.keywordBidDynamicDataService = keywordBidDynamicDataService;
        this.bidBsStatisticFacade = bidBsStatisticFacade;
        this.mailNotificationEventService = mailNotificationEventService;
        this.logPriceService = logPriceService;
        this.bidValidationService = bidValidationService;
        this.autoPriceCampQueueService = autoPriceCampQueueService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }


    public List<Bid> getBids(ClientId clientId, long operatorUid, ShowConditionSelectionCriteria selection,
                             LimitOffset limitOffset) {
        return getBids(clientId, operatorUid, selection, limitOffset, false);
    }
    /**
     * Возвращает ставки по указанным условиям выборки {@code selection}.
     * Возвращаемые ставки относятся к ключевым словам и автотаргетингу.
     */
    public List<Bid> getBids(ClientId clientId, long operatorUid, ShowConditionSelectionCriteria selection,
                             LimitOffset limitOffset, boolean filterArchivedCampaigns) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        // мы должны отдать только Keyword'ы из доступных оператору кампаний
        Collection<Long> campaignIdsToCheck;
        if (!selection.getCampaignIds().isEmpty()) {
            campaignIdsToCheck = selection.getCampaignIds();
        } else {
            // если ограничений на campaignIds нет, вычислим их на основе keywordIds и adGroupIds
            campaignIdsToCheck = bidRepository.getCampaignIdsForBids(shard,
                    selection.getShowConditionIds(), selection.getAdGroupIds());
        }

        CampaignSubObjectAccessChecker checker = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIdsToCheck);
        CampaignSubObjectAccessValidator validator = checker.createValidator(CampaignAccessType.READ);

        Set<Long> validCampaignIds = campaignIdsToCheck.stream()
                .map(validator)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .collect(Collectors.toSet());
        if (filterArchivedCampaigns) {
            validCampaignIds = filterToSet(validCampaignIds, Predicate.not(checker::objectInArchivedCampaign));
        }

        if (validCampaignIds.isEmpty()) {
            // не нашлось доступных для оператора кампаний
            return emptyList();
        }

        selection.withCampaignIds(validCampaignIds);

        return bidRepository.getBids(shard, selection, limitOffset);
    }

    /**
     * Устанавливает новые значения ставок в соответствии с заявками {@link SetBidItem}
     */
    public MassResult<SetBidItem> setBids(ClientId clientId, Long operatorUid, List<SetBidItem> setBidItemList) {
        ValidationResult<List<SetBidItem>, Defect> preValidation =
                bidsSetValidationService.preValidate(setBidItemList);

        List<SetBidItem> validItems = getValidItems(preValidation);
        if (preValidation.hasErrors() || validItems.isEmpty()) {
            return MassResult.brokenMassAction(preValidation.getValue(), preValidation);
        }

        ShowConditionSelectionCriteria criteria = ShowConditionSelectionCriteria.fromShowConditionList(setBidItemList);
        List<Bid> bids = getBids(clientId, operatorUid, criteria, maxLimited());

        BidValidationContainer<Bid> bidValidationContainer =
                bidValidationService.initBidValidationContainer(clientId, operatorUid, bids);

        ValidationResult<List<SetBidItem>, Defect> validation =
                bidsSetValidationService.validate(bidValidationContainer, setBidItemList);
        validation.merge(preValidation);

        validItems = getValidItems(validation);

        if (validation.hasErrors() || validItems.isEmpty()) {
            return MassResult.brokenMassAction(validation.getValue(), validation);
        }

        List<Long> keywordIds = StreamEx.of(bids)
                .filter(bid -> bid.getType() == ShowConditionType.KEYWORD)
                .map(Bid::getId)
                .toList();

        List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);

        List<Bid> relevanceMatchBids = StreamEx.of(bids)
                .filter(bid -> bid.getType() == ShowConditionType.RELEVANCE_MATCH)
                .toList();

        List<AppliedChanges<Bid>> relevanceMatchChanges =
                collectBidChanges(bidValidationContainer, validItems, relevanceMatchBids);

        List<AppliedChanges<Keyword>> keywordChanges =
                collectKeywordBidChanges(bidValidationContainer, validItems, keywords);

        setBidsInternal(clientId, keywordChanges, relevanceMatchChanges);

        List<Keyword> changedkeywordBids = StreamEx.of(keywordChanges)
                .map(AppliedChanges::getModel)
                .toList();

        List<Bid> changedRelevanceMatches = StreamEx.of(relevanceMatchChanges)
                .map(AppliedChanges::getModel)
                .toList();

        logPriceChanges(operatorUid, changedkeywordBids, bidValidationContainer.getClientWorkCurrency().getCode());
        logPriceChanges(operatorUid, changedRelevanceMatches, bidValidationContainer.getClientWorkCurrency().getCode());

        Set<Long> updatedCampaigns = new HashSet<>();
        changedkeywordBids.forEach(bid -> updatedCampaigns.add(bid.getCampaignId()));
        changedRelevanceMatches.forEach(bid -> updatedCampaigns.add(bid.getCampaignId()));

        autoPriceCampQueueService.clearAutoPriceQueue(updatedCampaigns);

        return MassResult.successfulMassAction(validation.getValue(), validation)
                .withCounts(relevanceMatchChanges.size(), setBidItemList.size() - validItems.size())
                .withOperationMeta(new OperationMeta()
                        .withAffectedCampaignIds(new ArrayList<>(bidValidationContainer.getVisibleCampaignIds())));
    }

    private List<AppliedChanges<Bid>> collectBidChanges(
            BidValidationContainer<Bid> validationContainer,
            List<SetBidItem> validBidsToSet,
            List<Bid> bids) {
        BidStorage<Bid> bidStorage = new BidStorage<>(bids);
        Stream<AppliedChanges<Bid>> changes = StreamEx.of(validBidsToSet)
                .mapToEntry(bidStorage::getBySelection)
                .flatMapValues(Collection::stream)
                .mapKeyValue((setBidItem, bid) -> applyChanges(validationContainer, setBidItem, bid))
                .filter(AppliedChanges::hasActuallyChangedProps);
        return updateChangeStatuses(changes)
                .collect(toList());
    }

    private List<AppliedChanges<Keyword>> collectKeywordBidChanges(
            BidValidationContainer<Bid> validationContainer,
            List<SetBidItem> validBidsToSet,
            List<Keyword> bids) {
        BidStorage<Keyword> bidStorage = new BidStorage<>(bids);
        Stream<AppliedChanges<Keyword>> changes = StreamEx.of(validBidsToSet)
                .mapToEntry(bidStorage::getBySelection)
                .flatMapValues(Collection::stream)
                .mapKeyValue((setBidItem, bid) -> applyChanges(validationContainer, setBidItem, bid))
                .filter(AppliedChanges::hasActuallyChangedProps);
        return updateChangeStatusesForKeyword(changes)
                .collect(toList());
    }

    private AppliedChanges<Bid> applyChanges(BidValidationContainer<Bid> validationContainer, SetBidItem setBidItem,
                                             Bid bid) {
        ModelChanges<Bid> changes = new ModelChanges<>(bid.getId(), Bid.class);
        if (canUpdatePrice(validationContainer, setBidItem, bid)) {
            changes.processNotNull(setBidItem.getPriceSearch(), Bid.PRICE);
        }
        if (canUpdateContextPrice(validationContainer, setBidItem, bid)) {
            changes.processNotNull(setBidItem.getPriceContext(), Bid.PRICE_CONTEXT);
        }
        if (canUpdatePriority(validationContainer, setBidItem, bid)) {
            changes.processNotNull(setBidItem.getAutobudgetPriority(), Bid.AUTOBUDGET_PRIORITY);
        }
        return changes.applyTo(bid);
    }

    private AppliedChanges<Keyword> applyChanges(BidValidationContainer<Bid> validationContainer, SetBidItem setBidItem,
                                                 Keyword bid) {
        ModelChanges<Keyword> changes = new ModelChanges<>(bid.getId(), Keyword.class);
        DbStrategy strategy = validationContainer.getCampaignStrategy(setBidItem);
        ShowConditionType bidType = ShowConditionType.KEYWORD;
        if (canUpdatePrice(strategy, bidType)) {
            changes.processNotNull(setBidItem.getPriceSearch(), Keyword.PRICE);
        }
        if (canUpdateContextPrice(strategy, bidType)) {
            changes.processNotNull(setBidItem.getPriceContext(), Keyword.PRICE_CONTEXT);
        }
        if (canUpdatePriority(strategy, bidType)) {
            changes.processNotNull(setBidItem.getAutobudgetPriority(), Keyword.AUTOBUDGET_PRIORITY);
        }
        return changes.applyTo(bid);
    }

    /**
     * Обновляет {@code LAST_CHANGE} и {@code STATUS_BS_SYNCED} у ставок в переданном {@code changes}.
     * У ключевых фраз также сбрасывается флаг предупреждения о смене позиции {@code WARN_ON_POSITION_LOST}
     */
    private Stream<AppliedChanges<Bid>> updateChangeStatuses(Stream<AppliedChanges<Bid>> changes) {
        LocalDateTime now = LocalDateTime.now();
        return changes.peek(change -> {
            change.modify(Bid.LAST_CHANGE, now);
            change.modify(Bid.STATUS_BS_SYNCED, StatusBsSynced.NO);
        });
    }

    /**
     * Обновляет {@code LAST_CHANGE} и {@code STATUS_BS_SYNCED} у ставок в переданном {@code changes}.
     * У ключевых фраз также сбрасывается флаг предупреждения о смене позиции {@code WARN_ON_POSITION_LOST}
     */
    private Stream<AppliedChanges<Keyword>> updateChangeStatusesForKeyword(Stream<AppliedChanges<Keyword>> changes) {
        LocalDateTime now = LocalDateTime.now();
        return changes.peek(change -> {
            change.modify(Keyword.MODIFICATION_TIME, now);
            change.modify(Keyword.STATUS_BS_SYNCED, StatusBsSynced.NO);
            //для ключевых слов сбрасываем флаг предупреждения о смене позиции
            change.modify(Keyword.NEED_CHECK_PLACE_MODIFIED, false);
        });
    }

    private void setBidsInternal(ClientId clientId,
                                 List<AppliedChanges<Keyword>> keywordChanges,
                                 List<AppliedChanges<Bid>> relevanceMatchChanges
    ) {
        setKeywordBidsInternal(clientId, keywordChanges);
        setBidsInternal(clientId, relevanceMatchChanges);
    }

    /**
     * Сбрасывает статус синхронизации фраз с БК
     * Выставляет время последнего апдейта
     *
     * @param changes выбирает в какую таблицу писать по {@link Bid#getType()}
     */
    private void setBidsInternal(ClientId clientId, List<AppliedChanges<Bid>> changes) {
        int shard = shardHelper.getShardByClientId(clientId);
        bidRepository.setBidsInBidsBase(shard, changes);
    }

    /**
     * Сбрасывает статус синхронизации фраз с БК
     * Выставляет время последнего апдейта
     * Выставляет флаг о том, нужно ли предупреждать о потере позиции в No
     */
    private void setKeywordBidsInternal(ClientId clientId, List<AppliedChanges<Keyword>> changes) {
        int shard = shardHelper.getShardByClientId(clientId);
        bidRepository.setBidsInBids(shard, changes);
    }

    /**
     * Автоматический расчёт ставок на основе правил вычисления {@link SetAutoBidItem}.
     */
    public MassResult<SetAutoBidItem> setAutoBids(ClientId clientId, long operatorUid,
                                                  List<SetAutoBidItem> setAutoBidItems, boolean isKeywordBidsService) {
        // pre-validation: проверяем однородность параметров и количество элементов
        ValidationResult<List<SetAutoBidItem>, Defect> preValidation =
                setAutoBidsValidationService.preValidate(setAutoBidItems);

        if (preValidation.hasAnyErrors() || getValidItems(preValidation).isEmpty()) {
            return MassResult.brokenMassAction(setAutoBidItems, preValidation);
        }

        // Чтобы два раза не ходить в базу, получаем ставки тут и используем их как в валидации, так при расчёте
        // новых значений
        ShowConditionSelectionCriteria criteria = fromShowConditionList(setAutoBidItems);
        // Для валидации берём все ставки, хотя по факту обновляем лишь те, которые
        // в группах с баннером и без флага "Мало показов". Для таких фраз не ходим
        // в Торги и не получаем данных для установки ставок.
        List<Bid> bids = getBids(clientId, operatorUid, criteria, maxLimited());

        BidValidationContainer<Bid> bidValidationContainer =
                bidValidationService.initBidValidationContainer(clientId, operatorUid, bids);

        // validation: полная проверка заявок на изменение ставок
        ValidationResult<List<SetAutoBidItem>, Defect> validation =
                setAutoBidsValidationService.validate(bidValidationContainer, setAutoBidItems, isKeywordBidsService);
        validation.merge(preValidation);

        List<SetAutoBidItem> validItems = getValidItems(validation);

        if (validItems.isEmpty()) {
            return MassResult.brokenMassAction(setAutoBidItems, validation);
        }

        // Определим, требуется ли получать данные о показах в Сети. Если нет, то ходим только в Торги. Иначе - и в
        // Торги, и в Показометр
        boolean withContext = StreamEx.of(validItems)
                .map(SetAutoBidItem::getScope)
                .findAny(scopes -> scopes.contains(BidTargetType.CONTEXT))
                .isPresent();

        /*
        Метод может вызываться как из Bids.setAuto, где используются данные позиционных Торгов,
        так и из KeywordBids.setAuto, оперирующим "объёмом трафика".
        Эти сценарии взаимоисключающие и для них сейчас предназначены разные ручки получения данных Торгов.
        */
        boolean useTrafficVolume = StreamEx.of(validItems)
                .map(SetAutoBidItem::getScope)
                .findAny(scopes -> scopes.contains(BidTargetType.SEARCH_BY_TRAFFIC_VOLUME))
                .isPresent();
        boolean usePosition = StreamEx.of(validItems)
                .map(SetAutoBidItem::getScope)
                .findAny(scopes -> scopes.contains(BidTargetType.SEARCH))
                .isPresent();

        // Это должно быть гарантировано валидацией. Просто перепроверка
        checkState(!(usePosition && useTrafficVolume),
                "You can't specify search price by positions and by trafficVolume simultaniously");

        List<Long> keywordIds = StreamEx.of(bids)
                .filter(bid -> bid.getType() == ShowConditionType.KEYWORD)
                .map(Bid::getId)
                .toList();

        List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);

        BidSetAutoPriceApplier bidSetAutoPriceApplier;
        Collection<CompleteBidData> completeBidData;
        try {
            // Оборачиваем получение данных из БК в try-catch, чтобы отличать ошибки недоступности Торгов от прочих
            if (!usePosition) {
                // Сюда также попадаем, если данные Торгов не нужны совсем (!usePosition && !useTrafficVolume), чтобы
                // сходить в Показометр
                Collection<CompleteBidData<KeywordTrafaretData>> localBidData = keywordBidDynamicDataService
                        .getCompleteBidDataTrafaretFormat(clientId, bids,
                                withContext,
                                useTrafficVolume /*withBsAuction*/,
                                false /*safePokazometer*/);
                bidSetAutoPriceApplier = new BidSetAutoPriceByTrafficVolumeApplier(validItems, localBidData, keywords);
                completeBidData = convertCompleteBidDataList(localBidData);
            } else {
                Collection<CompleteBidData<KeywordBidBsAuctionData>> localBidData = keywordBidDynamicDataService
                        .getCompleteBidData(clientId, bids,
                                withContext,
                                true /*withBsAuction*/,
                                false /*safePokazometer*/);
                bidSetAutoPriceApplier = new BidSetAutoPriceByPositionApplier(validItems, localBidData, keywords);
                completeBidData = convertCompleteBidDataList(localBidData);
            }
        } catch (Exception e) {
            logger.info("Can't set bids", e);
            throw new CantSetBidException();
        }

        List<AppliedChanges<Keyword>> keywordChanges = bidSetAutoPriceApplier.calcNewPriceForKeywordAndApply();

        List<AppliedChanges<Bid>> relevanceMatchChanges =
                bidSetAutoPriceApplier.calcNewPriceForRelevanceMatchAndApply(keywordChanges);

        List<AppliedChanges<Bid>> completeRelevanceMatchChanges =
                updateChangeStatuses(relevanceMatchChanges.stream()).collect(toList());
        List<AppliedChanges<Keyword>> completeKeywordChanges =
                updateChangeStatusesForKeyword(keywordChanges.stream()).collect(toList());

        setBidsInternal(clientId, completeKeywordChanges, completeRelevanceMatchChanges);

        // логируем изменения ставок
        List<Bid> changedRelevanceMatches = StreamEx.of(completeRelevanceMatchChanges)
                .map(AppliedChanges::getModel)
                .toList();
        logPriceChanges(operatorUid, changedRelevanceMatches, bidValidationContainer.getClientWorkCurrency().getCode());

        List<Keyword> changedKeywords = StreamEx.of(completeKeywordChanges)
                .map(AppliedChanges::getModel)
                .toList();
        logPriceChanges(operatorUid, changedKeywords, bidValidationContainer.getClientWorkCurrency().getCode());

        // рассылаем уведомления через добавление записей в таблицу EVENTS
        sendPriceChangeNotifications(clientId, operatorUid, completeRelevanceMatchChanges, completeBidData);
        sendPriceChangeNotifications(clientId, operatorUid, completeKeywordChanges, completeBidData);

        return MassResult.successfulMassAction(setAutoBidItems, validation)
                .withOperationMeta(new OperationMeta()
                        .withAffectedCampaignIds(new ArrayList<>(bidValidationContainer.getVisibleCampaignIds())));
    }

    /**
     * Нельзя просто так взять и привести {@code Collection<CompleteBidData<?>>} к {@code Collection<CompleteBidData>}.
     * Безошбочный способ: создать новую коллекцию и переложить туда элементы.
     */
    private static Collection<CompleteBidData> convertCompleteBidDataList(
            Collection<? extends CompleteBidData<?>> completeBidDataList) {
        return completeBidDataList.stream()
                .map(c -> (CompleteBidData) c)
                .collect(Collectors.toList());
    }

    /**
     * Записывает информацию о новых значениях ставкок в {@code PPCLOG_PRICE}.
     *
     * @see LogPriceService
     */
    private void logPriceChanges(Long operatorUid, Collection<? extends BidBase> changedBids,
                                 CurrencyCode workCurrency) {
        List<LogPriceData> priceDataList = StreamEx.of(changedBids)
                .map(bid -> bidToLog(bid, workCurrency))
                .toList();
        logPriceService.logPrice(priceDataList, operatorUid);
    }

    private LogPriceData bidToLog(BidBase bid, CurrencyCode currencyCode) {
        return new LogPriceData(
                bid.getCampaignId(),
                bid.getAdGroupId(),
                bid.getId(),
                bid.getPriceContext() != null ? bid.getPriceContext().doubleValue() : 0,
                bid.getPrice() != null ? bid.getPrice().doubleValue() : 0,
                currencyCode,
                LogPriceData.OperationType.UPDATE
        );
    }

    /**
     * Для каждого изменения ставки (на поиске или сети) генерирует уведомление и оправляет в очередь на отправку
     * через {@link MailNotificationEventService#queueEvents(Long, ClientId, Collection)}.
     *
     * @param actualChanges   изменения ставок
     * @param completeBidData данные о ставках, где они хранятся вместе с данными о кампании
     */
    private <T extends BidBase & Model> void sendPriceChangeNotifications(ClientId clientId, Long operatorUid,
                                                                          List<AppliedChanges<T>> actualChanges,
                                                                          Collection<CompleteBidData> completeBidData) {
        int maxExpectedEventsCount = actualChanges.size() * 2; // по два события на один Bid
        List<GenericEvent> events = new ArrayList<>(maxExpectedEventsCount);
        Map<Long, CompleteBidData> completeBidDataByBidId =
                listToMap(completeBidData, CompleteBidData::getBidId);
        for (AppliedChanges<T> change : actualChanges) {
            Long bidId = change.getModel().getId();
            CompleteBidData<?> adGroupForAuction = checkNotNull(completeBidDataByBidId.get(bidId),
                    "Can't find related info for bid with id %s", bidId);
            Long ownerUid = adGroupForAuction.getCampaign().getUserId();
            AdGroup adGroup = adGroupForAuction.getAdGroup();

            // изменилась цена на поиске
            if (change.changed(BidBase.PRICE)) {
                BigDecimal oldValue =
                        checkNotNull(change.getOldValue(BidBase.PRICE), "Old price value can't be null");
                BigDecimal newValue =
                        checkNotNull(change.getNewValue(BidBase.PRICE), "New price value can't be null");
                KeywordEvent priceChangeEvent = KeywordEvent
                        .changedSearchPriceEvent(operatorUid, ownerUid, adGroup, oldValue, newValue);
                events.add(priceChangeEvent);
            }

            // изменилась цена на сети
            if (change.changed(BidBase.PRICE_CONTEXT)) {
                BigDecimal oldValue =
                        checkNotNull(change.getOldValue(BidBase.PRICE_CONTEXT),
                                "Old price_context value can't be null");
                BigDecimal newValue =
                        checkNotNull(change.getNewValue(BidBase.PRICE_CONTEXT),
                                "New price_context value can't be null");
                KeywordEvent priceChangeEvent = KeywordEvent
                        .changedContextPriceEvent(operatorUid, ownerUid, adGroup, oldValue, newValue);
                events.add(priceChangeEvent);
            }
        }
        mailNotificationEventService.queueEvents(operatorUid, clientId, events);
    }

    /**
     * Все ставки которые есть в таблице  BIDS переносим в BIDS_MANUAL_PRICES для заданных кампаний
     *
     * @param campaigns заданные кампаний
     */
    public void saveManualBids(List<Campaign> campaigns) {
        List<Long> campaignIds = mapList(campaigns, Campaign::getId);

        saveManualBidsForCampaignIds(campaignIds);
    }

    public void saveManualBidsForCampaignIds(Collection<Long> campaignIds) {
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .forEach(bidRepository::copyFromBidsToBidsManualPricesForCampaignIds);
    }

    /**
     * Все ставки из BIDS_MANUAL_PRICES переносим в BIDS для заданных кампаний
     *
     * @param campaigns заданные кампаний
     */
    public void restoreManualBids(ClientId clientId,
                                  CampaignStrategyChangingSettings settings,
                                  List<CampaignWithStrategy> campaigns) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<Long> campaignIds = StreamEx.of(campaigns).map(CampaignWithStrategy::getId).toList();
        List<Bid> manualBids = bidRepository.getBidsManualPricesForCampaignIds(shard, campaignIds);
        List<Long> keywordIds = StreamEx.of(manualBids).map(Bid::getId).toList();
        Map<Long, Keyword> keywords = listToMap(keywordService.getKeywords(clientId, keywordIds),
                Keyword::getId, Function.identity());
        Map<Long, List<Bid>> campaignToBids = StreamEx.of(manualBids).groupingBy(Bid::getCampaignId);
        List<AppliedChanges<Keyword>> keywordChanges = new ArrayList<>();
        campaigns.forEach(campaign -> {
            BigDecimal maxPrice = settings.getMaxPrice();
            List<Bid> bidList = campaignToBids.get(campaign.getId());
            if (bidList == null) {
                return;
            }
            List<AppliedChanges<Keyword>> keywordsChangesForCampaign = filterAndMapList(
                    bidList, it -> keywords.containsKey(it.getId()),
                    bid -> priceChangesKeyword(limitPrice(bid.getPrice(), maxPrice),
                            limitPrice(bid.getPriceContext(), maxPrice), keywords.get(bid.getId())
                    ));
            keywordChanges.addAll(keywordsChangesForCampaign);
        });
        setKeywordBidsInternal(clientId, keywordChanges);

        bidRepository.deleteBidManualPricesForCampaignIds(shard, campaignIds);
        bidRepository.resetBidsSynced(shard, campaignIds);
    }

    private static BigDecimal limitPrice(@Nullable BigDecimal price, BigDecimal maxPrice) {
        return price == null ? BigDecimal.ZERO : price.min(maxPrice);
    }

    /**
     * Обнуленяет цены на сеть у ключевых фраз и автотаргетинга для всех кампании.
     *
     * @param operatorUid
     * @param campaigns
     */
    public void resetPriceContext(Long operatorUid, List<CampaignWithStrategy> campaigns) {
        List<Long> campaignIds = mapList(campaigns, CampaignWithStrategy::getId);
        Map<Long, CurrencyCode> campaignIdToCurrencyCode = listToMap(campaigns,
                CampaignWithStrategy::getId, CampaignWithStrategy::getCurrency);

        shardHelper.groupByShard(campaignIds, ShardKey.CID).stream().forKeyValue((shard, cids) -> {
            List<Bid> bids = bidRepository.getBidsWithRelevanceMatchByCampaignIds(shard, cids);
            bidRepository.resetPriceContextToZeroForBidsBaseByCampaignIds(shard, cids);
            bidRepository.resetPriceContextToZeroForBidsByCampaignIds(shard, cids);

            List<LogPriceData> priceDataList = StreamEx.of(bids)
                    .map(bid -> new LogPriceData(
                            bid.getCampaignId(),
                            bid.getAdGroupId(),
                            bid.getId(),
                            0.0,
                            bid.getPrice().doubleValue(),
                            campaignIdToCurrencyCode.get(bid.getCampaignId()),
                            LogPriceData.OperationType.UPDATE_2
                    )).toList();
            logPriceService.logPrice(priceDataList, operatorUid);
        });
    }

    /**
     * обновляем ставки, если переходим на автобюджет
     *
     * @param operatorUid оператор
     * @param campaigns   заданные кампаний
     */
    public void updateBidsOnAutoBudget(ClientId clientId, Long operatorUid, List<CampaignWithStrategy> campaigns) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<Bid> bids = bidRepository.getBidsByCampaignIds(shard, mapList(campaigns, CampaignWithStrategy::getId));
        Map<Long, CampaignWithStrategy> campaignsMap = listToMap(campaigns, CampaignWithStrategy::getId);
        List<Bid> bidsToBeUpdated = filterList(bids,
                bid -> checkPriceNeedChange(bid, campaignsMap.get(bid.getCampaignId())));

        List<Long> keywordIds = mapList(bidsToBeUpdated, Bid::getId);
        List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);
        logPriceService.logPrice(mapList(bidsToBeUpdated, bid -> new LogPriceData(
                bid.getCampaignId(),
                bid.getAdGroupId(),
                bid.getId(),
                autoStrategyPrice(bid.getPriceContext(), campaignsMap.get(bid.getCampaignId())).doubleValue(),
                autoStrategyPrice(bid.getPrice(), campaignsMap.get(bid.getCampaignId())).doubleValue(),
                campaignsMap.get(bid.getCampaignId()).getCurrency(),
                LogPriceData.OperationType.UPDATE_2
        )), operatorUid);
        List<AppliedChanges<Keyword>> keywordChanges = mapList(keywords, bid -> priceChangesKeyword(
                autoStrategyPrice(bid.getPrice(), campaignsMap.get(bid.getCampaignId())),
                autoStrategyPrice(bid.getPriceContext(), campaignsMap.get(bid.getCampaignId())),
                bid
        ));
        setKeywordBidsInternal(clientId, keywordChanges);

        bidRepository.setAutobudgetPriorityAutobudget(shard, campaignsMap.keySet());
    }

    public void updateBidsOnAutoBudgetForCpm(int shard, CampaignStrategyChangingSettings settings,
                                             ClientId clientId, Long operatorUid,
                                             Map<Long, CampaignWithStrategy> campaignsMap,
                                             List<Bid> bidsToBeUpdated) {
        List<Long> keywordIds = mapList(bidsToBeUpdated, Bid::getId);
        List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);
        BigDecimal cpmZeroPrice = BigDecimal.ZERO;

        logPriceService.logPrice(mapList(bidsToBeUpdated, bid -> new LogPriceData(
                bid.getCampaignId(),
                bid.getAdGroupId(),
                bid.getId(),
                autoStrategyPrice(settings, bid.getPriceContext(), campaignsMap.get(bid.getCampaignId())).doubleValue(),
                cpmZeroPrice.doubleValue(),
                campaignsMap.get(bid.getCampaignId()).getCurrency(),
                LogPriceData.OperationType.UPDATE_2)), operatorUid);

        List<AppliedChanges<Keyword>> keywordChanges = mapList(keywords, bid -> priceChangesKeyword(
                cpmZeroPrice,
                autoStrategyPrice(settings, bid.getPriceContext(), campaignsMap.get(bid.getCampaignId())),
                bid
        ));
        setKeywordBidsInternal(clientId, keywordChanges);

        bidRepository.setAutobudgetPriorityAutobudget(shard, campaignsMap.keySet());
    }

    private static AppliedChanges<Keyword> priceChangesKeyword(BigDecimal bidPrice, BigDecimal bidPriceContext,
                                                               Keyword bid) {
        ModelChanges<Keyword> changes = new ModelChanges<>(bid.getId(), Keyword.class);
        changes.processNotNull(bidPrice, Keyword.PRICE);
        changes.processNotNull(bidPriceContext, Keyword.PRICE_CONTEXT);
        changes.process(StatusBsSynced.NO, Keyword.STATUS_BS_SYNCED);
        return changes.applyTo(bid);
    }

    private static boolean checkPriceNeedChange(Bid bid, CampaignWithStrategy campaign) {
        return bid.getPrice().compareTo(autoStrategyPrice(bid.getPrice(), campaign)) != 0
                || bid.getPriceContext().compareTo(autoStrategyPrice(bid.getPriceContext(), campaign)) != 0;
    }

    /**
     * Корректная ставка пользователя по данной фразе на поиске и в РСЯ для автостратегии
     */
    private static BigDecimal autoStrategyPrice(@Nullable BigDecimal bidPrice, CampaignWithStrategy campaign) {
        Currency currency = campaign.getCurrency().getCurrency();
        return autoStrategyPrice(bidPrice, campaign, currency.getMinPrice(), currency.getMaxPrice());
    }

    /**
     * Корректная ставка пользователя по данной фразе на поиске и в РСЯ для автостратегии
     */
    public static BigDecimal autoStrategyPrice(CampaignStrategyChangingSettings campaignStrategyChangingSettings,
                                               @Nullable BigDecimal bidPrice, CampaignWithStrategy campaign) {
        return autoStrategyPrice(bidPrice, campaign, campaignStrategyChangingSettings.getMinPrice(),
                campaignStrategyChangingSettings.getMaxPrice());
    }

    private static BigDecimal autoStrategyPrice(@Nullable BigDecimal bidPrice, CampaignWithStrategy campaign,
                                                BigDecimal minPrice, BigDecimal maxPrice) {
        BigDecimal price = bidPrice == null ? BigDecimal.ZERO : bidPrice;
        if (price.compareTo(BigDecimal.ZERO) <= 0) {
            return BigDecimal.ZERO;
        }
        price = price.max(minPrice);
        BigDecimal strategyBid = campaign.getStrategy().getStrategyData().getBid();
        if (strategyBid != null && strategyBid.compareTo(BigDecimal.ZERO) > 0) {
            price = price.min(strategyBid);
        }
        price = price.min(maxPrice);
        return price;
    }

    /**
     * Установка начальных цен на поиске и в сети
     * <p>
     * В сети:
     * Используется для единичной установки цен(например при переходе
     * к стратегии "независимое управление")
     * <p>
     * На поиске:
     * Установка первоначальных поисковых цен на фразы для которых цена не задавалась ни разу.
     * (Кампания с изначально отключенной поисковой стратегией)
     * Заменяет во всей кампании нулевые ставки на поиске на цену 1-го места + 30% (DIRECT-13851)
     */
    public void setPhrasesInitialPrices(ClientId clientId, Long operatorUid,
                                        List<SetPhrasesInitialPricesOption> options) {
        List<AppliedChanges<Keyword>> keywordChanges = new ArrayList<>();
        int shard = shardHelper.getShardByClientId(clientId);
        List<Bid> bids = filterList(
                bidRepository.getBidsByCampaignIds(shard, mapList(options,
                        SetPhrasesInitialPricesOption::getCampaignId)),
                bid -> priceZeroOrNull(bid.getPrice()) || priceZeroOrNull(bid.getPriceContext()));
        Map<Long, Money> bidFirstPosition = bidFirstPositionMap(clientId, operatorUid, options);
        Map<Long, List<Keyword>> keywordsByCampaignId = StreamEx.of(keywordService.getKeywords(clientId,
                mapList(bids, Bid::getId)))
                .groupingBy(Keyword::getCampaignId);

        for (SetPhrasesInitialPricesOption option : options) {
            List<Keyword> keywords = keywordsByCampaignId.get(option.getCampaignId());
            if (keywords == null || keywords.isEmpty()) {
                continue;
            }
            for (Keyword keyword : keywords) {
                ModelChanges<Keyword> changes = new ModelChanges<>(keyword.getId(), Keyword.class);
                BigDecimal minPrice = option.getCampaignCurrency().getCurrency().getMinPrice();
                if (option.getSetPrice() && (keyword.getPrice() == null || keyword.getPrice().compareTo(minPrice) < 0)) {
                    changes.processNotNull(calcPhrasesInitialPrices(keyword, option, bidFirstPosition), Keyword.PRICE);
                }
                if (option.getSetPriceContext()
                        && (keyword.getPriceContext() == null || keyword.getPriceContext().compareTo(minPrice) < 0)) {
                    changes.processNotNull(calcPhrasesInitialPriceContext(keyword, option), Keyword.PRICE_CONTEXT);
                }
                if (option.getSetMinPriceContext()
                        && (keyword.getPriceContext() == null || keyword.getPriceContext().compareTo(minPrice) < 0)) {
                    changes.processNotNull(setMinPriceContext(option), Keyword.PRICE_CONTEXT);
                }
                if (changes.isAnyPropChanged()) {
                    changes.process(StatusBsSynced.NO, Keyword.STATUS_BS_SYNCED);
                    keywordChanges.add(changes.applyTo(keyword));
                }
            }
        }
        setKeywordBidsInternal(clientId, keywordChanges);
    }

    /**
     * Возвращает хешпам id КФ на его цену 1-го места
     */
    Map<Long, Money> bidFirstPositionMap(ClientId clientId, Long operatorUid,
                                         List<SetPhrasesInitialPricesOption> options) {
        List<Long> campaignIdWithSetPriceOption = StreamEx.of(options)
                .filter(SetPhrasesInitialPricesOption::getSetPrice)
                .filter(SetPhrasesInitialPricesOption::getCalculateBsStatisticsFirstPosition)
                .map(SetPhrasesInitialPricesOption::getCampaignId)
                .toList();
        if (campaignIdWithSetPriceOption.isEmpty()) {
            return new HashMap<>();
        }
        return bidBsStatisticFacade.bidBsStatisticFirstPosition(clientId,
                getBids(clientId, operatorUid,
                        new ShowConditionSelectionCriteria().withCampaignIds(campaignIdWithSetPriceOption),
                        maxLimited()));
    }

    private static BigDecimal calcPhrasesInitialPriceContext(Keyword keyword, SetPhrasesInitialPricesOption option) {
        if (keyword.getPrice() != null && keyword.getPrice().compareTo(BigDecimal.ZERO) > 0) {
            return keyword.getPrice();
        }
        //если выставляем начальные цены и для поиска, и для сети, то в сети должна быть умолчальная ставка
        return option.getCampaignCurrency().getCurrency().getDefaultPrice();
    }

    private static BigDecimal setMinPriceContext(SetPhrasesInitialPricesOption option) {
        return option.getCampaignCurrency().getCurrency().getMinPrice();
    }

    private static BigDecimal calcPhrasesInitialPrices(Keyword keyword, SetPhrasesInitialPricesOption option,
                                                       Map<Long, Money> bidBsStatistic) {
        if (keyword.getPriceContext() != null && keyword.getPriceContext().compareTo(
                option.getCampaignCurrency().getCurrency().getMinPrice()) > 0) {
            //если на поиске нет ставки, а в сетях она есть, копируем ставку с сети
            return keyword.getPriceContext();
        }
        //Прогноз первого места в Гарантии + 30%
        if (bidBsStatistic.containsKey(keyword.getId())) {
            Money position = bidBsStatistic.get(keyword.getId());
            if (position != null) {
                return position.bigDecimalValue().multiply(new BigDecimal("1.3"));
            }
        }
        //если статистики по фразе нет то выставляется ставка по умолчанию
        return option.getCampaignCurrency().getCurrency().getDefaultPrice();
    }

    private static boolean priceZeroOrNull(@Nullable BigDecimal bidPrice) {
        return bidPrice == null || bidPrice.compareTo(BigDecimal.ZERO) == 0;
    }

    public void updatePhrasesContextPriceForCpmCampaignsOnAutobudgetDisabling(
            CampaignStrategyChangingSettings settings,
            ClientId clientId,
            Map<Long, BigDecimal> oldAvgCpmByCampaignId,
            List<Long> keywordIds) {
        List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);
        List<AppliedChanges<Keyword>> appliedChanges = mapList(keywords,
                keyword -> new ModelChanges<>(keyword.getId(), Keyword.class)
                        .process(calculateNewPriceForCpmBidOnDisablingAutobudget(
                                settings,
                                oldAvgCpmByCampaignId.get(keyword.getCampaignId()),
                                keyword.getPriceContext()), Keyword.PRICE_CONTEXT)
                        .applyTo(keyword));
        setKeywordBidsInternal(clientId, appliedChanges);
    }

    public static BigDecimal calculateNewPriceForCpmBidOnDisablingAutobudget(
            CampaignStrategyChangingSettings settings, BigDecimal avgCpm, BigDecimal priceContext) {
        BigDecimal oldPrice = nvl(priceContext, BigDecimal.ZERO);
        return min(max(settings.getMinPrice(), NumberUtils.greaterThanZero(oldPrice) ?
                        oldPrice : avgCpm),
                settings.getMaxPrice());
    }
}
