package ru.yandex.direct.core.entity.bids.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.model.Model;

/*
 * Контейнер для данных bids, используется в валидации и сервисе
 */
public class BidValidationContainer<T extends BiddingShowCondition> {
    private Map<Long, Campaign> campaignsById;
    private Currency clientWorkCurrency;
    private Set<Long> writableCampaignIds;
    private Set<Long> visibleCampaignIds;
    private Map<Long, Long> campaignsByShowConditionIds;
    private Map<Long, Long> campaignsByAdGroupId;
    private List<T> existingShowCondition;
    private Map<Long, AdGroupType> adGroupTypesByIds;
    private Set<Long> bsRarelyLoadedAdGroupIds;

    public Map<Long, Campaign> getCampaignsById() {
        return campaignsById;
    }

    public BidValidationContainer<T> withCampaignsById(
            Map<Long, Campaign> campaignsById) {
        this.campaignsById = campaignsById;
        return this;
    }

    public Currency getClientWorkCurrency() {
        return clientWorkCurrency;
    }

    public BidValidationContainer<T> withCurrency(Currency currency) {
        this.clientWorkCurrency = currency;
        return this;
    }

    public Set<Long> getWritableCampaignIds() {
        return writableCampaignIds;
    }

    public BidValidationContainer<T> withWritableCampaignIds(Set<Long> writableCampaignIds) {
        this.writableCampaignIds = writableCampaignIds;
        return this;
    }

    public Set<Long> getVisibleCampaignIds() {
        return visibleCampaignIds;
    }

    public BidValidationContainer<T> withVisibleCampaignIds(Set<Long> visibleCampaignIds) {
        this.visibleCampaignIds = visibleCampaignIds;
        return this;
    }

    public Map<Long, Long> getCampaignsByShowConditionIds() {
        return campaignsByShowConditionIds;
    }

    public BidValidationContainer<T> withCampaignsByShowConditionIds(
            Map<Long, Long> campaignsByShowConditionIds) {
        this.campaignsByShowConditionIds = campaignsByShowConditionIds;
        return this;
    }

    public Map<Long, Long> getCampaignsByAdGroupId() {
        return campaignsByAdGroupId;
    }

    public BidValidationContainer<T> withCampaignsByAdGroupId(
            Map<Long, Long> campaignsByAdGroupId) {
        this.campaignsByAdGroupId = campaignsByAdGroupId;
        return this;
    }

    public List<T> getExistingShowCondition() {
        return existingShowCondition;
    }

    public BidValidationContainer<T> withExistingShowCondition(List<T> existingShowCondition) {
        this.existingShowCondition = existingShowCondition;
        return this;
    }

    public Map<Long, AdGroupType> getAdGroupTypesByIds() {
        return adGroupTypesByIds;
    }

    public BidValidationContainer<T> withAdGroupTypesByIds(Map<Long, AdGroupType> adGroupTypesByIds) {
        this.adGroupTypesByIds = adGroupTypesByIds;
        return this;
    }

    public Set<Long> getBsRarelyLoadedAdGroupIds() {
        return bsRarelyLoadedAdGroupIds;
    }

    public BidValidationContainer<T> withBsRarelyLoadedAdGroupIds(Set<Long> bsRarelyLoadedAdGroupIds) {
        this.bsRarelyLoadedAdGroupIds = bsRarelyLoadedAdGroupIds;
        return this;
    }

    @Nonnull
    public <W extends Model & BidSelectionCriteria> Optional<DbStrategy> getCampaignStrategyOptional(W bidItem) {
        return Optional.ofNullable(getCampaignIdFor(bidItem))
                .map(campaignsById::get)
                .map(Campaign::getStrategy);
    }

    public <W extends Model & BidSelectionCriteria> DbStrategy getCampaignStrategy(W bidItem) {
        return Optional.ofNullable(getCampaignIdFor(bidItem))
                .map(campaignsById::get)
                .orElseThrow(() -> new IllegalStateException("Can't find campaign for item " + bidItem))
                .getStrategy();
    }

    @Nullable
    public <W extends Model & BidSelectionCriteria> Long getCampaignIdFor(W bidItem) {
        final Long campaignId;
        if (bidItem.getCampaignId() != null) {
            campaignId = bidItem.getCampaignId();
        } else if (bidItem.getAdGroupId() != null) {
            campaignId = campaignsByAdGroupId.get(bidItem.getAdGroupId());
        } else {
            campaignId = campaignsByShowConditionIds.get(bidItem.getId());
        }
        return campaignId;
    }

    @SuppressWarnings("SameParameterValue")
    public <V, W extends Model & BidSelectionCriteria> V getCampaignParam(W setBidItem,
                                                                          Function<CampaignSimple, V> getter, V def) {
        return Optional.ofNullable(getCampaignIdFor(setBidItem))
                .map(id -> getCampaignsById().get(id))
                .map(getter)
                .orElse(def);
    }
}
