package ru.yandex.direct.core.entity.bids.service;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesAdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class BidValidationService {
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final AggregatedStatusesAdGroupRepository aggregatedStatusesAdGroupRepository;
    private final RbacService rbacService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;

    @Autowired
    public BidValidationService(CampaignRepository campaignRepository,
                                AdGroupRepository adGroupRepository,
                                AggregatedStatusesAdGroupRepository aggregatedStatusesAdGroupRepository,
                                RbacService rbacService, ClientService clientService,
                                ShardHelper shardHelper) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.aggregatedStatusesAdGroupRepository = aggregatedStatusesAdGroupRepository;
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
    }

    public <T extends BiddingShowCondition> BidValidationContainer<T> initBidValidationContainer(ClientId clientId,
                                                                                                 Long operatorUid,
                                                                                                 List<T> existingShowConditions) {
        int shard = shardHelper.getShardByClientId(clientId);
        Set<Long> campaignIds = listToSet(existingShowConditions, BiddingShowCondition::getCampaignId);
        Set<Long> adGroupIds = listToSet(existingShowConditions, BiddingShowCondition::getAdGroupId);
        Map<Long, Campaign> campaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIds), Campaign::getId);

        Currency currency = clientService.getWorkCurrency(clientId);

        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, campaignIds);
        Set<Long> visibleCampaignIds = rbacService.getVisibleCampaigns(operatorUid, campaignIds);

        Map<Long, Long> campaignsByRetargetingIds =
                StreamEx.of(existingShowConditions)
                        .mapToEntry(BiddingShowCondition::getId, BiddingShowCondition::getCampaignId)
                        .toMap();
        Map<Long, Long> campaignsByAdGroupId =
                StreamEx.of(existingShowConditions)
                        .mapToEntry(BiddingShowCondition::getAdGroupId, BiddingShowCondition::getCampaignId)
                        .distinctKeys().toMap();
        Map<Long, AdGroupType> adGroupTypesByIds =
                StreamEx.ofValues(adGroupRepository.getAdGroupsWithType(shard, clientId, adGroupIds))
                        .toMap(AdGroupWithType::getId, AdGroupWithType::getType);
        Set<Long> bsRarelyLoadedAdGroupIds =
                aggregatedStatusesAdGroupRepository.getAdGroups(shard, adGroupIds)
                        .stream()
                        .filter(AggregatedStatusAdGroup::getBsRarelyLoaded)
                        .map(AggregatedStatusAdGroup::getId)
                        .collect(Collectors.toSet());
        BidValidationContainer<T> container = new BidValidationContainer<>();
        container.withCampaignsByAdGroupId(campaignsByAdGroupId)
                .withCampaignsById(campaignsById)
                .withCampaignsByShowConditionIds(campaignsByRetargetingIds)
                .withCurrency(currency)
                .withVisibleCampaignIds(visibleCampaignIds)
                .withWritableCampaignIds(writableCampaignIds)
                .withBsRarelyLoadedAdGroupIds(bsRarelyLoadedAdGroupIds)
                .withExistingShowCondition(existingShowConditions)
                .withAdGroupTypesByIds(adGroupTypesByIds);
        return container;
    }
}
