package ru.yandex.direct.core.entity.bids.service;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.validation.BidsValidator;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.bids.Constants.MIN_BIDS_PER_REQUEST;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.selectionFieldCountValidator;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.selectorsAreOfSameType;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.absentRequiredField;

@Service
@ParametersAreNonnullByDefault
public class CommonSetBidsValidationService {

    @Autowired
    public CommonSetBidsValidationService() {
    }

    public ValidationResult<List<SetBidItem>, Defect> preValidate(
            List<SetBidItem> setBidItemList) {
        ListValidationBuilder<SetBidItem, Defect> vb =
                ListValidationBuilder.<SetBidItem, Defect>of(setBidItemList)
                        .check(notNull())
                        .check(minListSize(MIN_BIDS_PER_REQUEST), absentRequiredField(), When.isValid())
                        .check(eachNotNull(), When.isValid())
                        .check(selectorsAreOfSameType(), When.isValid())
                        .checkEachBy(selectionFieldCountValidator(), When.isValid())
                        .checkEach(unique(), duplicatedObject(), When.isValid());
        return vb.getResult();
    }

    public <T extends BiddingShowCondition> ValidationResult<List<SetBidItem>, Defect> validate(
            BidValidationContainer<T> bidValidationContainer,
            List<SetBidItem> setBidItemList, boolean warnOnStrategyMismatch) {
        return ListValidationBuilder.<SetBidItem, Defect>of(setBidItemList)
                .checkBy(new BidsValidator(bidValidationContainer, warnOnStrategyMismatch))
                .getResult();
    }
}
