package ru.yandex.direct.core.entity.bids.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.StatusShowsForecast;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupsShowsForecastService;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.auction.container.AdGroupForAuction;
import ru.yandex.direct.core.entity.auction.container.bs.BsDataWithKeyword;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.service.BsAuctionService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.service.validation.type.BannerTypeValidationPredicates;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.keyword.service.KeywordShowsForecastService;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.nullSafetyFlatMap;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

/**
 * Сервис предназначен для получения динамических данных для ставок на ключевые фразы
 */
@Service
@ParametersAreNonnullByDefault
public class KeywordBidDynamicDataService {
    private static final Logger logger = LoggerFactory.getLogger(KeywordBidDynamicDataService.class);

    private final ShardHelper shardHelper;
    private final KeywordService keywordService;
    private final AdGroupService adGroupService;
    private final BannerService bannerService;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final VcardService vcardService;
    private final CampaignService campaignService;
    private final BsAuctionService bsAuctionService;
    private final PokazometerService pokazometerService;
    private final AdGroupsShowsForecastService adGroupsShowsForecastService;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final MinusKeywordPreparingTool minusKeywordPreparingTool;

    @Autowired
    public KeywordBidDynamicDataService(ShardHelper shardHelper,
                                        KeywordService keywordService,
                                        AdGroupService adGroupService,
                                        BannerService bannerService,
                                        BannerRelationsRepository bannerRelationsRepository,
                                        VcardService vcardService,
                                        CampaignService campaignService,
                                        BsAuctionService bsAuctionService,
                                        PokazometerService pokazometerService,
                                        AdGroupsShowsForecastService adGroupsShowsForecastService,
                                        MinusKeywordsPackRepository minusKeywordsPackRepository,
                                        MinusKeywordPreparingTool minusKeywordPreparingTool) {
        this.shardHelper = shardHelper;
        this.keywordService = keywordService;
        this.adGroupService = adGroupService;
        this.bannerService = bannerService;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.vcardService = vcardService;
        this.campaignService = campaignService;
        this.bsAuctionService = bsAuctionService;
        this.pokazometerService = pokazometerService;
        this.adGroupsShowsForecastService = adGroupsShowsForecastService;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
    }

    /**
     * Для ставок из переданного списка {@code bids} ходит в Торги и Показометр, если указано,
     * за актуальными данными. Возвращает список из {@link CompleteBidData}, где содержится как информация
     * из базы Директа, так и динамические данные из Торгов и Показометра.
     *
     * @see KeywordBidDynamicDataService#getKeywordBidDynamicData
     */
    public Collection<CompleteBidData<KeywordBidBsAuctionData>> getCompleteBidData(ClientId clientId,
                                                                                   List<Bid> bids, boolean withPokazometerData, boolean withBsAuctionData, boolean safePokazometer) {
        return getCompleteBidDataBase(clientId, bids, withPokazometerData, withBsAuctionData, safePokazometer,
                adGroups -> bsAuctionService.getBsResults(clientId, adGroups));
    }

    public Collection<CompleteBidData<KeywordTrafaretData>> getCompleteBidDataTrafaretFormat(ClientId clientId,
                                                                                             List<Bid> bids, boolean withPokazometerData, boolean withBsAuctionData, boolean safePokazometer) {
        return getCompleteBidDataBase(clientId, bids, withPokazometerData, withBsAuctionData, safePokazometer,
                keywords -> bsAuctionService.getBsTrafaretResults(clientId, keywords));
    }

    /**
     * Получение значений показометра, в случае ошибки возвращается пустой список
     */
    public List<KeywordBidPokazometerData> safeGetPokazometerData(ClientId clientId, List<Keyword> keywords) {
        IndexedContainer adGroupsBsData = new IndexedContainer(clientId, keywords);
        List<AdGroupForAuction> adGroupForAuctions = getBsAdGroups(adGroupsBsData);
        return pokazometerService.safeGetPokazometerResults(adGroupForAuctions);
    }

    private <M extends BsDataWithKeyword> Collection<CompleteBidData<M>> getCompleteBidDataBase(ClientId clientId,
                                                                                                List<Bid> bids, boolean withPokazometerData, boolean withBsAuctionData, boolean safePokazometer,
                                                                                                Function<List<AdGroupForAuction>, List<M>> bsResultsGetter) {
        Set<Long> adGroupIds = listToSet(bids, Bid::getAdGroupId);
        List<AdGroup> adGroups = adGroupService.getAdGroups(clientId, adGroupIds);
        Map<Long, AdGroup> adGroupsByIds = listToMap(adGroups, AdGroup::getId);

        Set<Long> campaignIds = StreamEx.of(bids).map(Bid::getCampaignId).toSet();
        Map<Long, Campaign> campaignsByIds =
                listToMap(campaignService.getCampaignsWithStrategies(clientId, campaignIds), Campaign::getId);

        // динамические данные из Торгов и Показометра
        Collection<KeywordBidDynamicData<M>> keywordBidDynamicData = emptyList();
        if (withBsAuctionData || withPokazometerData) {
            IndexedContainer adGroupsBsData = new IndexedContainer(clientId, bids, adGroups, campaignsByIds);
            List<AdGroupForAuction> adGroupForAuctions = getBsAdGroups(adGroupsBsData);

            keywordBidDynamicData = getKeywordBidDynamicData(clientId, adGroupForAuctions,
                    withPokazometerData, withBsAuctionData, safePokazometer, bsResultsGetter);
        }
        Map<Long, KeywordBidDynamicData<M>> dynamicDataByBidId =
                listToMap(keywordBidDynamicData, KeywordBidDynamicData::getBidId);

        return StreamEx.of(bids)
                .map(bid -> buildCompleteBidData(
                        bid,
                        adGroupsByIds.get(bid.getAdGroupId()),
                        campaignsByIds.get(bid.getCampaignId()),
                        dynamicDataByBidId.get(bid.getId())))
                .toList();
    }

    /**
     * Создаёт список {@link AdGroupForAuction} для отправки клиенту Торгов на основе переданных наборов core-объектов
     * (кампаний, баннеров, групп, фраз)
     */
    private List<AdGroupForAuction> getBsAdGroups(IndexedContainer idx) {
        return EntryStream.of(idx.keywordsByAdGroupId)
                .mapKeyValue((adGroupId, keywords) ->
                        {
                            if (!idx.keywordsByAdGroupId.containsKey(adGroupId)) {
                                // нет фраз -- не с чем идти в Торги. Игнорируем такую группу
                                return null;
                            }
                            AdGroup adGroup = idx.adGroups.get(adGroupId);
                            checkState(adGroup != null, "группа должна существовать");

                            Campaign campaign = idx.campaignByCampaignId.get(adGroup.getCampaignId());
                            BannerWithSystemFields mainBanner = idx.mainBannersByAdGroupId.get(adGroupId);
                            return AdGroupForAuction.builder()
                                    .campaign(campaign)
                                    .adGroup(adGroup)
                                    .mainBanner(mainBanner)
                                    .mainBannerVcard(idx.extractVcard(mainBanner))
                                    .publisherDomain(idx.mobileContentPublisherDomainsByAdGroupIds.get(adGroupId))
                                    .storeAppId(idx.mobileContentBundleIdByAdGroupIds.get(adGroupId))
                                    .keywords(idx.keywordsByAdGroupId.get(adGroupId))
                                    .currency(campaign.getCurrency().getCurrency())
                                    .bannerQuantity(idx.bannerQuantitiesByAdGroupIds.get(adGroupId))
                                    .build();
                        }
                )
                .nonNull()
                .toList();
    }

    /**
     * Самособирающийся контейнер информации, нужной для формирования списка групп для отправки в Торги,
     * т.е. списка экземпляров {@link AdGroupForAuction}.
     * <p>
     * Нужен, чтобы метод {@link #getBsAdGroups} принимал один аргумент, а не десяток мап.
     */
    private class IndexedContainer {
        /**
         * В группах все минус фразы: частные и библиотечные - будут объединены в {@link AdGroup#MINUS_KEYWORDS}
         */
        private final Map<Long, AdGroup> adGroups;
        private final Map<Long, List<Keyword>> keywordsByAdGroupId;
        private final Map<Long, BannerWithSystemFields> mainBannersByAdGroupId;
        private final Map<Long, Integer> bannerQuantitiesByAdGroupIds;
        private final Map<Long, Domain> mobileContentPublisherDomainsByAdGroupIds;
        private final Map<Long, String> mobileContentBundleIdByAdGroupIds;
        private final Map<Long, Campaign> campaignByCampaignId;
        private final Map<Long, Vcard> vcards;

        IndexedContainer(ClientId clientId, List<Bid> bids, List<AdGroup> adGroups,
                         Map<Long, Campaign> campaignsByIds) {
            this.adGroups = listToMap(adGroups, AdGroup::getId);
            this.campaignByCampaignId = campaignsByIds;
            Map<Long, Bid> bidsById = listToMap(bids, Bid::getId);
            Set<Long> adGroupIds = listToSet(bids, Bid::getAdGroupId);

            mainBannersByAdGroupId = bannerRelationsRepository.getMainBannerByAdGroupIds(clientId, adGroupIds);
            List<TextBanner> mainTextBanners = selectList(mainBannersByAdGroupId.values(), TextBanner.class);

            Set<Long> vcardIds = listToSet(mainTextBanners, TextBanner::getVcardId);
            vcards = vcardService.getVcardsById(clientId, vcardIds);

            bannerQuantitiesByAdGroupIds =
                    bannerService.getBannerQuantitiesByAdGroupIds(clientId, adGroupIds);

            List<Long> mobileContentAdGroupIds = StreamEx.of(adGroups)
                    .filter(adGroup -> adGroup.getType() == AdGroupType.MOBILE_CONTENT)
                    .map(AdGroup::getId)
                    .toList();
            mobileContentPublisherDomainsByAdGroupIds =
                    adGroupService.getMobileContentPublisherDomains(clientId, mobileContentAdGroupIds);
            mobileContentBundleIdByAdGroupIds =
                    adGroupService.getMobileContentAppIds(clientId, mobileContentAdGroupIds);

            List<Long> keywordIds = StreamEx.of(bids)
                    .filter(bid -> bid.getType() == ShowConditionType.KEYWORD)
                    .map(Bid::getId)
                    .toList();

            List<Keyword> keywords = keywordService.getKeywords(clientId, keywordIds);

            keywords.forEach(keyword -> {
                keyword.setPrice(bidsById.get(keyword.getId()).getPrice());
                keyword.setPriceContext(bidsById.get(keyword.getId()).getPriceContext());
            });

            keywordsByAdGroupId = StreamEx.of(keywords)
                    .mapToEntry(Keyword::getAdGroupId)
                    .invert()
                    .grouping();
            mergeLibraryAndPrivatePacksInAdGroup(adGroups, clientId);
        }

        // отдельный конструктор для получения данных показометра для ручки showconditions.update
        IndexedContainer(ClientId clientId, List<Keyword> keywords) {
            Set<Long> campaignIds = listToSet(keywords, Keyword::getCampaignId);
            keywordsByAdGroupId = StreamEx.of(keywords)
                    .mapToEntry(Keyword::getAdGroupId, identity())
                    .grouping();
            Set<Long> adGroupIds = keywordsByAdGroupId.keySet();
            adGroups = listToMap(adGroupService.getAdGroups(clientId, adGroupIds), AdGroup::getId);
            mainBannersByAdGroupId = bannerService.getMainBannerByAdGroupIds(clientId, adGroupIds);
            bannerQuantitiesByAdGroupIds =
                    bannerService.getBannerQuantitiesByAdGroupIds(clientId, adGroupIds);
            campaignByCampaignId =
                    listToMap(campaignService.getCampaignsWithStrategies(clientId, campaignIds), Campaign::getId);
            mergeLibraryAndPrivatePacksInAdGroup(adGroups.values(), clientId);

            // не заполняем те поля, которые точно не будут использоваться в ответе
            vcards = emptyMap();

            mobileContentPublisherDomainsByAdGroupIds = emptyMap();
            mobileContentBundleIdByAdGroupIds = emptyMap();
        }

        private Vcard extractVcard(@Nullable BannerWithSystemFields banner) {
            return Optional.ofNullable(banner).filter(BannerTypeValidationPredicates::isTextBanner)
                    .map(b -> TextBanner.class.cast(b).getVcardId())
                    .map(vcards::get).orElse(null);
        }

        /**
         * Забирает все необходимые библиотечные наборы минус-слов из базы и подмердживает в соответствующие группы
         * в поле {@link AdGroup#MINUS_KEYWORDS}
         */
        private void mergeLibraryAndPrivatePacksInAdGroup(Collection<AdGroup> adGroups, ClientId clientId) {
            Map<Long, List<String>> libraryMinusKeywords = getLibraryMinusKeywords(adGroups, clientId);
            adGroups.forEach(adGroup -> {
                List<List<String>> adGroupLibraryMinusKeywords =
                        mapList(adGroup.getLibraryMinusKeywordsIds(), libraryMinusKeywords::get);
                adGroup.withMinusKeywords(minusKeywordPreparingTool
                        .mergePrivateAndLibrary(adGroup.getMinusKeywords(), adGroupLibraryMinusKeywords));
            });
        }

        private Map<Long, List<String>> getLibraryMinusKeywords(Collection<AdGroup> adGroups, ClientId clientId) {
            Set<Long> libraryMinusKeywordsPacksIds =
                    nullSafetyFlatMap(adGroups, AdGroup::getLibraryMinusKeywordsIds, toSet());

            int shard = shardHelper.getShardByClientId(clientId);
            List<MinusKeywordsPack> minusKeywordsPacks =
                    minusKeywordsPackRepository.get(shard, clientId, libraryMinusKeywordsPacksIds);

            return listToMap(minusKeywordsPacks, MinusKeywordsPack::getId, MinusKeywordsPack::getMinusKeywords);
        }
    }

    private <M extends BsDataWithKeyword> CompleteBidData<M> buildCompleteBidData(Bid bid, AdGroup adGroup,
                                                                                  Campaign campaign, KeywordBidDynamicData<M> dynamicData) {
        return new CompleteBidData<M>()
                .withBid(bid)
                .withAdGroup(adGroup)
                .withCampaign(campaign)
                .withDynamicData(dynamicData);
    }

    /**
     * Для указанных ставок, сгруппированных в {@link AdGroupForAuction}, возвращает актуальные данные
     * из Торгов и Показометра.
     * <ul>
     * <li>Возвращаются {@link KeywordBidDynamicData} для всех ставок из {@code bsAdGroups}</li>
     * <li>Если у группы флаг {@link AdGroup#BS_RARELY_LOADED} выставлен в {@code true}, то для фраз этой группы
     * не отправляются запрсоы в сторонние системы.</li>
     * <li>Если {@code withPokazometerData} равен {@code false}, в ответе не будет данных от Показометра</li>
     * </ul>
     * <b>Side-effect:</b> обновляет прогнозы на фразах.
     */
    private <M extends BsDataWithKeyword> Collection<KeywordBidDynamicData<M>> getKeywordBidDynamicData(
            ClientId clientId, List<AdGroupForAuction> bsAdGroups,
            boolean withPokazometerData, boolean withBsAuctionData, boolean safePokazometer,
            Function<List<AdGroupForAuction>, List<M>> bsResultsGetter) {
        List<AdGroupForAuction> eligibleAdGroups = StreamEx.of(bsAdGroups)
                .filter(adGroupForAuction -> !adGroupForAuction.getAdGroup().getBsRarelyLoaded())
                .toList();

        // обновляем showsForecast на фразах, которые собираемся отправлять в Торги
        eligibleAdGroups = updatePhrasesShowsForecast(clientId, eligibleAdGroups);

        // идём в торги
        List<M> bsResults = emptyList();
        if (withBsAuctionData) {
            bsResults = bsResultsGetter.apply(eligibleAdGroups);
        }

        // идём в Показометр
        List<KeywordBidPokazometerData> pokazometerResults = emptyList();
        if (withPokazometerData) {
            if (safePokazometer) {
                pokazometerResults = pokazometerService.safeGetPokazometerResults(eligibleAdGroups);
            } else {
                pokazometerResults = pokazometerService.getPokazometerResults(eligibleAdGroups);
            }
        }
        return joinResults(bsAdGroups, bsResults, pokazometerResults);
    }

    /**
     * Объединяет результаты ответов от Торгов и Показометра, объединяя их
     * по {@code bidId} в {@link KeywordBidDynamicData}
     */
    private <M extends BsDataWithKeyword> Collection<KeywordBidDynamicData<M>> joinResults(
            List<AdGroupForAuction> eligibleAdGroups,
            List<M> bsResults, List<KeywordBidPokazometerData> pokazometerResults) {
        Map<Keyword, M> bsAuctionDataByKeyword = listToMap(bsResults, M::getKeyword);
        Map<Long, KeywordBidPokazometerData> pokazometerDataById =
                StreamEx.of(pokazometerResults).mapToEntry(KeywordBidPokazometerData::getKeywordId)
                        .invert()
                        .toMap();

        return StreamEx.of(eligibleAdGroups)
                .flatCollection(AdGroupForAuction::getKeywords)
                .map(keyword -> new KeywordBidDynamicData<M>()
                        .withBidId(keyword.getId())
                        .withBsAuctionData(bsAuctionDataByKeyword.get(keyword))
                        .withPokazometerData(pokazometerDataById.get(keyword.getId())))
                .toList();
    }

    private List<AdGroupForAuction> updatePhrasesShowsForecast(ClientId clientId,
                                                               List<AdGroupForAuction> interestedGroups) {
        Map<Boolean, List<AdGroupForAuction>> adGroupByUpdatingRequired =
                StreamEx.of(interestedGroups)
                        .partitioningBy(this::adGroupForecastUpdateRequired);

        List<AdGroupForAuction> adGroupsToUpdate = adGroupByUpdatingRequired.get(Boolean.TRUE);

        List<Long> adGroupIds = StreamEx.of(adGroupsToUpdate)
                .map(AdGroupForAuction::getAdGroup)
                .map(AdGroup::getId).toList();
        adGroupsShowsForecastService
                .updateShowsForecastIfNeeded(clientId, adGroupIds,
                        KeywordShowsForecastService.DEFAULT_ADVQ_CALL_TIMEOUT);

        // можем подтянуть лишние фразы, потом используем только необходимые
        Map<Long, List<Keyword>> updatedKeywordsByAdGroupIds
                = keywordService.getKeywordsByAdGroupIds(clientId, adGroupIds);

        for (AdGroupForAuction adGroupForAuction : adGroupsToUpdate) {

            Long adGroupId = adGroupForAuction.getAdGroup().getId();
            Map<Long, Keyword> updatedKeywordsById =
                    listToMap(updatedKeywordsByAdGroupIds.get(adGroupId), Keyword::getId);

            // Обновим фразы в группе на актуальные с обновлёнными прогнозами
            Collection<Keyword> usedKeywords = adGroupForAuction.getKeywords();
            for (Keyword oldKeyword : usedKeywords) {
                Keyword newKeyword = updatedKeywordsById.get(oldKeyword.getId());
                if (newKeyword == null) {
                    logger.info("Keyword (id={}) missed during updating showsForecast. Use old one for BSAuction query",
                            oldKeyword.getId());
                    newKeyword = oldKeyword;
                }
                oldKeyword.setShowsForecast(newKeyword.getShowsForecast());
            }
        }

        return interestedGroups;
    }

    /**
     * Нужно ли обновить прогнозы для фраз в укащанной группе
     */
    private boolean adGroupForecastUpdateRequired(AdGroupForAuction adGroupForAuction) {
        Collection<Keyword> phrases = adGroupForAuction.getKeywords();
        AdGroup adGroup = adGroupForAuction.getAdGroup();
        boolean phrasesPresent = phrases != null && !phrases.isEmpty();
        boolean rarelyLoadedAdGroup = adGroup.getBsRarelyLoaded();
        StatusShowsForecast statusShowsForecast = adGroup.getStatusShowsForecast();
        boolean showsForecastStatusRequiresUpdate =
                statusShowsForecast != StatusShowsForecast.PROCESSED &&
                        statusShowsForecast != StatusShowsForecast.ARCHIVED;
        return showsForecastStatusRequiresUpdate && !rarelyLoadedAdGroup && phrasesPresent;
    }

}
