package ru.yandex.direct.core.entity.bids.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.auction.container.AdGroupForAuction;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.exception.PokazometerFailedException;
import ru.yandex.direct.core.entity.bids.utils.PokazometerUtils;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.pokazometer.GroupRequest;
import ru.yandex.direct.pokazometer.GroupResponse;
import ru.yandex.direct.pokazometer.PhraseRequest;
import ru.yandex.direct.pokazometer.PhraseResponse;
import ru.yandex.direct.pokazometer.PokazometerClient;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class PokazometerService {
    private static final Logger logger = LoggerFactory.getLogger(PokazometerService.class);
    private static final Percent DEFAULT_NDS_IN_RUSSIA = Percent.fromPercent(BigDecimal.valueOf(18));

    private final CurrencyRateService currencyRateService;
    private final PokazometerClient pokazometerClient;

    @Autowired
    public PokazometerService(CurrencyRateService currencyRateService,
                              PokazometerClient pokazometerClient) {
        this.currencyRateService = currencyRateService;
        this.pokazometerClient = pokazometerClient;
    }

    /**
     * {@link #getPokazometerResults(List)}, который при ошибке Показометра возвращает пустой список,
     * а не генерирует исключение.
     */
    public List<KeywordBidPokazometerData> safeGetPokazometerResults(List<AdGroupForAuction> adGroupForAuctions) {
        try {
            return getPokazometerResults(adGroupForAuctions);
        } catch (PokazometerFailedException e) {
            logger.info("Catch Pokazometer error. Return empty list", e);
            return emptyList();
        }
    }

    /**
     * Возвращает данные о покрытии для фраз, переданных в {@code adGroupForAuctions}.
     * <p>
     * Запрос в Показометр не будет производиться для остановленных фраз и тех, чья стратегия - {@code different_places}
     * Если при получении данных для какой-либо группы были ошибки, её фраз не будет в результирующем списке.
     */
    public List<KeywordBidPokazometerData> getPokazometerResults(List<AdGroupForAuction> adGroupForAuctions)
            throws PokazometerFailedException {
        // не ходим в Показометер за данными для фраз относящихся к кампаниям с strategy != different_places
        Predicate<AdGroupForAuction> strategyNotDifferentPlaces =
                adGroup -> adGroup.getCampaign().getStrategy().isDifferentPlaces();
        // не ходим за данными Показометра для групп без баннера (нет баннера -- нет показов в РСЯ)
        Predicate<AdGroupForAuction> adGroupWithAd = adGroup -> adGroup.getBanner() != null;
        // и за данными для остановленных фраз
        List<AdGroupForAuction> adGroupsForPokazometer =
                StreamEx.of(adGroupForAuctions)
                        .filter(strategyNotDifferentPlaces)
                        .filter(adGroupWithAd)
                        .filter(adGroup -> isNotEmpty(adGroup.getKeywords()))
                        .toList();

        List<GroupRequest> groupRequests = mapList(adGroupsForPokazometer, this::buildRequestByAdGroup);
        IdentityHashMap<GroupRequest, GroupResponse> results = tryPokazometerRequest(groupRequests);

        return EntryStream.of(results)
                .removeValues(GroupResponse::hasErrors)
                .mapKeys(InternalGroupRequest.class::cast)
                .flatMapKeyValue(this::convertSingleGroupResponse)
                .toList();
    }

    /**
     * Для фраз, сгруппированных по геотаргетингу (GroupRequest) возвращает данные по кликам/ставкам в сети
     *
     * @param groupRequests список запросов с ключевыми фразами, сгруппированными по геотаргетингу
     * @param currencyCode  код валюты клиента, к которой нужно пересчитать ответ показометра
     * @return - список keywordId c данными по кликам и ставками для фиксированных значений покрытия
     */
    public List<KeywordBidPokazometerData> getPokazometerResults(List<GroupRequest> groupRequests,
                                                                 CurrencyCode currencyCode)
            throws PokazometerFailedException {
        IdentityHashMap<GroupRequest, GroupResponse> results = tryPokazometerRequest(groupRequests);

        return StreamEx.of(results.values())
                .remove(GroupResponse::hasErrors)
                .flatMap(this::convertSingleGroupResponse)
                .map(convertToClientCurrency(currencyCode))
                .toList();
    }

    private IdentityHashMap<GroupRequest, GroupResponse> tryPokazometerRequest(List<GroupRequest> groupRequests)
            throws PokazometerFailedException {
        IdentityHashMap<GroupRequest, GroupResponse> results;
        try {
            results = pokazometerClient.get(groupRequests);
        } catch (Exception e) {
            // тут debug, потому что всё равно пробрасываем исключение выше
            logger.debug("Can't get Pokazometer data", e);
            throw new PokazometerFailedException(e);
        }
        return results;
    }

    private UnaryOperator<KeywordBidPokazometerData> convertToClientCurrency(CurrencyCode clientCurrencyCode) {
        return moneyCurrencyConverter(money -> {
            Money priceInClientCurrency = currencyRateService.convertMoney(money, clientCurrencyCode);
            if (!clientCurrencyCode.equals(CurrencyCode.YND_FIXED)) {
                priceInClientCurrency = priceInClientCurrency.subtractNds(DEFAULT_NDS_IN_RUSSIA);
            }
            return priceInClientCurrency.roundToAuctionStepUp();
        });
    }

    private StreamEx<KeywordBidPokazometerData> convertSingleGroupResponse(GroupResponse groupResponse) {
        return StreamEx.of(groupResponse.getPhrases())
                .map(PokazometerUtils::convertPhraseResponse);
    }

    private Stream<KeywordBidPokazometerData> convertSingleGroupResponse(
            InternalGroupRequest request,
            GroupResponse response) {
        Collection<Keyword> requestPhrases = request.getBsAdGroup().getKeywords();
        Campaign campaign = request.getBsAdGroup().getCampaign();
        CurrencyCode currencyCode = campaign.getCurrency();
        List<PhraseResponse> responsePhrases = response.getPhrases();

        // PokazometerClient сохраняет порядок фраз в ответе
        return StreamEx.of(requestPhrases)
                .zipWith(responsePhrases.stream())
                .mapKeyValue(PokazometerUtils::convertPhraseResponse)
                .map(convertToClientCurrency(currencyCode));
    }

    /**
     * Возвращает {@link UnaryOperator} для преобразования цен в валюту клиента в ответе показометра.
     * Можно было бы конвертировать на этапе заполнения {@link KeywordBidPokazometerData}, но тогда пришлось бы дальше
     * пробрасывать информацию о валюте клиента.
     */
    private UnaryOperator<KeywordBidPokazometerData> moneyCurrencyConverter(UnaryOperator<Money> moneyConverter) {
        return phraseData -> {
            Map<PhraseResponse.Coverage, Money> convertedPricesByCoverage =
                    EntryStream.of(phraseData.getCoverageWithPrices())
                            .mapValues(moneyConverter)
                            .toMap();
            //после конвертации есть вероятность получить разное кол-во кликов при одинаковой ставке. Выбираем большее значение.
            Map<Money, Integer> convertedCostAndClicks =
                    EntryStream.of(phraseData.getAllCostsAndClicks())
                            .mapKeys(moneyConverter)
                            .toMap(Integer::max);
            return new KeywordBidPokazometerData(phraseData.getKeywordId(), convertedPricesByCoverage,
                    convertedCostAndClicks);
        };
    }

    private GroupRequest buildRequestByAdGroup(AdGroupForAuction bsAdGroup) {
        AdGroup adGroup = bsAdGroup.getAdGroup();
        List<Long> geoRegionIds = adGroup.getGeo();
        if (geoRegionIds.size() == 1 && geoRegionIds.get(0) == 0L) {
            // geo = [0] отправляем в Показометр как пустой массив []
            geoRegionIds = emptyList();
        }

        List<PhraseRequest> phrases = mapList(bsAdGroup.getKeywords(),
                // передаем null вместо цены в сетях, т.к. она нужна только для расчета прогноза покрытия,
                // а он здесь никак не используется
                keyword -> new PhraseRequest(keyword.getPhrase(), null)
        );
        return new InternalGroupRequest(bsAdGroup, phrases, geoRegionIds);
    }

    /**
     * Расширение {@link GroupRequest} для хранения ссылки на исходный {@link AdGroupForAuction}
     */
    private static class InternalGroupRequest extends GroupRequest {

        private final AdGroupForAuction bsAdGroup;

        InternalGroupRequest(AdGroupForAuction bsAdGroup, List<PhraseRequest> phrases, List<Long> geo) {
            super(phrases, geo);
            this.bsAdGroup = bsAdGroup;
        }

        AdGroupForAuction getBsAdGroup() {
            return bsAdGroup;
        }
    }

}
