package ru.yandex.direct.core.entity.bids.utils;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.SetBidSelectionType;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidValidationContainer;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.model.Model;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Утилиты для обработки условий отбора ставок {@link BidSelectionCriteria}
 */
@ParametersAreNonnullByDefault
public class BidSelectionUtils {

    /**
     * @return {@link SetBidSelectionType} для <b>первого</b> элемента переданного списка {@code setAutoBidItems}
     */
    @Nonnull
    public static SetBidSelectionType detectSelectionType(List<? extends BidSelectionCriteria> setAutoBidItems) {
        return detectSelectionType(setAutoBidItems.get(0));
    }

    /**
     * Группирует ставки {@code bids} по условиям отбора {@code selectionCriteria}.
     */
    @Nonnull
    public static <T extends BidSelectionCriteria> Map<T, List<Bid>> groupBidsBySelectionCriteria(
            List<T> selectionCriteria,
            Collection<Bid> bids) {
        checkArgument(!selectionCriteria.isEmpty(), "Valid items list can't be empty");

        // После валидации у всех валидных элементов заполнен только один параметр выбора, он уникален и одинаков для всех item'ов
        SetBidSelectionType selectionType = detectSelectionType(selectionCriteria);

        Function<Bid, Long> bidGroupingKeyExtractor;
        Function<T, Long> setAutoBidKeyExtractor;
        switch (selectionType) {
            case KEYWORD_ID:
                bidGroupingKeyExtractor = Bid::getId;
                setAutoBidKeyExtractor = BidSelectionCriteria::getId;
                break;
            case AD_GROUP_ID:
                bidGroupingKeyExtractor = Bid::getAdGroupId;
                setAutoBidKeyExtractor = BidSelectionCriteria::getAdGroupId;
                break;
            case CAMPAIGN_ID:
                bidGroupingKeyExtractor = Bid::getCampaignId;
                setAutoBidKeyExtractor = BidSelectionCriteria::getCampaignId;
                break;
            default:
                throw new IllegalStateException("Unexpected value for SetBidSelectionType: " + selectionType);
        }
        Map<Long, List<Bid>> bidsByKey =
                StreamEx.of(bids).groupingBy(bidGroupingKeyExtractor);
        return StreamEx.of(selectionCriteria)
                .mapToEntry(setAutoBidKeyExtractor)
                .mapValues(bidsByKey::get)
                .toMap();
    }

    /**
     * @return {@link SetBidSelectionType} для переданного {@link BidSelectionCriteria}
     */
    @Nonnull
    public static SetBidSelectionType detectSelectionType(BidSelectionCriteria firstSetAutoBidItem) {
        SetBidSelectionType selectionType;
        if (firstSetAutoBidItem.getCampaignId() != null) {
            selectionType = SetBidSelectionType.CAMPAIGN_ID;
        } else if (firstSetAutoBidItem.getAdGroupId() != null) {
            selectionType = SetBidSelectionType.AD_GROUP_ID;
        } else {
            selectionType = SetBidSelectionType.KEYWORD_ID;
        }
        return selectionType;
    }

    public static <T extends Model & BidSelectionCriteria> boolean canUpdatePrice(
            BidValidationContainer<Bid> validationContainer, T setBidItem, Bid bid) {
        DbStrategy strategy = validationContainer.getCampaignStrategy(setBidItem);
        ShowConditionType bidType = bid.getType();
        return canUpdatePrice(strategy, bidType);
    }

    public static boolean canUpdatePrice(DbStrategy strategy, ShowConditionType bidType) {
        return !strategy.isAutoBudget() &&
                !strategy.isSearchStop() &&
                bidType.shouldCheckBidValueFor(Bid.PRICE);
    }

    public static <T extends Model & BidSelectionCriteria> boolean canUpdateContextPrice(
            BidValidationContainer<Bid> validationContainer, T setBidItem,
            Bid bid) {
        DbStrategy strategy = validationContainer.getCampaignStrategy(setBidItem);
        ShowConditionType bidType = bid.getType();
        return canUpdateContextPrice(strategy, bidType);
    }

    public static boolean canUpdateContextPrice(DbStrategy strategy, ShowConditionType bidType) {
        return !strategy.isAutoBudget() &&
                !strategy.isNetStop() &&
                strategy.isDifferentPlaces() &&
                bidType.shouldCheckBidValueFor(Bid.PRICE_CONTEXT);
    }

    public static <T extends Model & BidSelectionCriteria> boolean canUpdatePriority(
            BidValidationContainer<Bid> validationContainer, T setBidItem,
            Bid bid) {
        DbStrategy strategy = validationContainer.getCampaignStrategy(setBidItem);
        ShowConditionType bidType = bid.getType();
        return canUpdatePriority(strategy, bidType);
    }

    public static boolean canUpdatePriority(DbStrategy strategy, ShowConditionType bidType) {
        return strategy.isAutoBudget() && bidType.shouldCheckBidValueFor(Bid.AUTOBUDGET_PRIORITY);
    }
}
