package ru.yandex.direct.core.entity.bids.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.bids.model.Bid;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.memoize;

/**
 * Хранилище {@link Bid} с интерфейсом для удобного получения набора подходящих ставок по {@link BiddingShowCondition}
 */
@ParametersAreNonnullByDefault
public class BidStorage<T extends BiddingShowCondition> {

    private final Collection<T> bids;

    // предполагается, что за время жизни экземпляра к нему будут обращаться только с одним и тем же типом ID'шников
    private Supplier<Map<Long, List<T>>> bidsById = memoize(this::calcBidsById);
    private Supplier<Map<Long, List<T>>> bidsByAdGroupId = memoize(this::calcBidsByAdGroupId);
    private Supplier<Map<Long, List<T>>> bidsByCampaignId = memoize(this::calcBidsByCampaignId);

    public BidStorage(Collection<T> bids) {
        this.bids = checkNotNull(bids, "Bids should be specified");
    }

    /**
     * По переданному условию отбора ставок {@code condition} находит подходящие {@link Bid}.
     * <p>
     * Если элементы находятся, они возвращаются списком.
     * Если найти ничего не получилось, возвращается {@link Collections#emptyList()}.
     */
    @Nonnull
    public List<T> getBySelection(BidSelectionCriteria condition) {
        if (condition.getId() != null) {
            return bidsById.get().getOrDefault(condition.getId(), emptyList());
        } else if (condition.getAdGroupId() != null) {
            return bidsByAdGroupId.get().getOrDefault(condition.getAdGroupId(), emptyList());
        } else if (condition.getCampaignId() != null) {
            return bidsByCampaignId.get().getOrDefault(condition.getCampaignId(), emptyList());
        }
        return emptyList();
    }

    /**
     * Возвращает набор условий показа ({@link BiddingShowCondition}), принадлежащих указанной группе.
     */
    @Nonnull
    public List<T> getByAdGroupId(Long adGroupId) {
        return bidsByAdGroupId.get().getOrDefault(adGroupId, emptyList());
    }

    /**
     * Возвращает условие показа ({@link BiddingShowCondition}) по ID.
     */
    @Nonnull
    public Optional<T> getByBidId(Long bidId) {
        return Optional.ofNullable(bidsById.get().get(bidId)).map(list -> Iterables.getFirst(list, null));
    }

    private Map<Long, List<T>> calcBidsById() {
        Function<T, Long> groupingKeyExtractor = BiddingShowCondition::getId;
        return groupBidsBy(groupingKeyExtractor);
    }

    private Map<Long, List<T>> calcBidsByAdGroupId() {
        Function<T, Long> groupingKeyExtractor = BiddingShowCondition::getAdGroupId;
        return groupBidsBy(groupingKeyExtractor);
    }

    private Map<Long, List<T>> calcBidsByCampaignId() {
        Function<T, Long> groupingKeyExtractor = BiddingShowCondition::getCampaignId;
        return groupBidsBy(groupingKeyExtractor);
    }

    private Map<Long, List<T>> groupBidsBy(Function<T, Long> groupingKeyExtractor) {
        return StreamEx.of(bids)
                .mapToEntry(groupingKeyExtractor)
                .invert()
                .grouping();
    }
}
