package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;
import java.util.List;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.pokazometer.PhraseResponse;
import ru.yandex.direct.utils.math.MathUtils;
import ru.yandex.direct.utils.math.Point;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

public class ContextPriceWizard implements PriceWizard<KeywordBidPokazometerData> {

    private final Integer increasePercent;
    private final Integer contextCoverage;

    private ContextPriceWizard(Integer increasePercent, Integer contextCoverage) {
        this.increasePercent = increasePercent;
        this.contextCoverage = contextCoverage;
    }

    static ContextPriceWizard createWizard(Integer increasePercent, Integer contextCoverage) {
        return new ContextPriceWizard(increasePercent, contextCoverage);
    }

    @Override
    public BigDecimal calcPrice(KeywordBidPokazometerData pokazometerData) {
        checkNotNull(pokazometerData, "Can't interpolate cost of click if pokazometer is null");
        checkArgument(pokazometerData.getCoverageWithPrices().size() > 1,
                "More than one point is required for interpolation");
        List<Point> pokazometerDataAsList = EntryStream.of(pokazometerData.getCoverageWithPrices())
                // ключи - проценты покрытия, преобразуем в double
                .mapKeys(PhraseResponse.Coverage::getPercentage)
                .mapKeys(Number::doubleValue)
                // значения - ставки, преобразуем в double
                .mapValues(Money::bigDecimalValue)
                .mapValues(Number::doubleValue)
                .mapKeyValue(Point::fromDoubles)
                .toList();
        BigDecimal coveragePrice = BigDecimal.valueOf(
                MathUtils.interpolateLinear(contextCoverage.doubleValue(), pokazometerDataAsList).getY()
        );

        BigDecimal multiplicand;
        if (increasePercent != null) {
            multiplicand = BigDecimal.valueOf(increasePercent, 2);
        } else {
            multiplicand = BigDecimal.ZERO;
        }
        return coveragePrice.add(coveragePrice.multiply(multiplicand));
    }

}
