package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;
import java.util.function.Function;

/**
 * Стратегия расчёта ставки на поиске.
 */
class PriceCalculationStrategy {
    private final BigDecimal increasePercent;
    private final Base base;

    static PriceCalculationStrategy byDiff(Integer increasePercent) {
        return new PriceCalculationStrategy(BigDecimal.valueOf(increasePercent, 2), Base.DIFF);
    }

    static PriceCalculationStrategy byValue(Integer increasePercent) {
        return new PriceCalculationStrategy(BigDecimal.valueOf(increasePercent, 2), Base.VALUE);
    }

    /**
     * Определяет, относительно чего должна вычисляться надбавка: относительно абсолютной величины
     * или разницы между ценами входа
     */
    private enum Base {
        DIFF(PriceInPosition::getDiff),
        VALUE(PriceInPosition::getValue);

        private Function<PriceInPosition, BigDecimal> baseExtractor;

        Base(Function<PriceInPosition, BigDecimal> baseExtractor) {
            this.baseExtractor = baseExtractor;
        }

        BigDecimal fromPosition(PriceInPosition position) {
            return baseExtractor.apply(position);
        }
    }

    private PriceCalculationStrategy(BigDecimal increasePercent, Base base) {
        this.increasePercent = increasePercent;
        this.base = base;
    }

    /**
     * На основе заданного {@code increasePercent} и {@code base} (от чего счатать процент: от абсолюта
     * или разницы между соседними позициями) по переданному описанию позиции на поиске вычисляет ставку.
     */
    BigDecimal calcNewPrice(PriceInPosition priceInPosition) {
        BigDecimal increaseBase = base.fromPosition(priceInPosition);
        return priceInPosition.getValue().add(increaseBase.multiply(increasePercent));
    }
}
