package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.keyword.model.Place;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;

/**
 * Правила поиска ставки по позиции. Плюс, есть возможность искать "предыдущую" позицию
 */
class PriceRule {
    // Порядок важен: он учитывается в методе #getNext(..)
    private static final List<PlaceItem> API_PLACES = asList(
            PlaceItem.placeItem(KeywordBidBsAuctionData::getPremium, 0,
                    Place.PREMIUM1),
            PlaceItem.placeItem(KeywordBidBsAuctionData::getPremium, 1,
                    Place.PREMIUM2),
            PlaceItem.placeItem(KeywordBidBsAuctionData::getPremium, 2,
                    Place.PREMIUM3),
            PlaceItem.placeItem(KeywordBidBsAuctionData::getPremium, -1,
                    Place.PREMIUM4, Place.PREMIUM),
            PlaceItem.placeItem(KeywordBidBsAuctionData::getGuarantee, 0,
                    Place.GUARANTEE1, Place.GUARANTEE2, Place.GUARANTEE3, Place.FIRST),
            PlaceItem.placeItem(KeywordBidBsAuctionData::getGuarantee, -1,
                    Place.GUARANTEE4, Place.GARANT));
    private static final PriceRule API_PRICE_RULE = new PriceRule(API_PLACES);

    private final List<PlaceItem> placeItemList;
    private final Map<Place, PlaceItem> placeItemMap;

    private PriceRule(List<PlaceItem> placeItemList) {
        this.placeItemList = placeItemList;
        this.placeItemMap = StreamEx.of(placeItemList)
                .mapToEntry(PlaceItem::getRelatedPlaces)
                .invert()
                .flatMapKeys(Collection::stream)
                .toMap();
    }

    static PriceRule forApi() {
        return API_PRICE_RULE;
    }

    /**
     * @return {@link PriceInPosition} описание позиции для вычисления ставок: значение и разница со следующим местом
     */
    PriceInPosition describePlace(Place place, KeywordBidBsAuctionData auctionData) {
        PlaceItem targetPlace = getRuleByPlace(place);
        PlaceItem nextPlace = getNext(targetPlace);
        // цены -- обычные, не micro-деньги
        BigDecimal targetPrice = targetPlace.getPosition(auctionData).getBidPrice().bigDecimalValue();
        BigDecimal nextPrice = nextPlace.getPosition(auctionData).getBidPrice().bigDecimalValue();

        BigDecimal diff = nextPrice.subtract(targetPrice);
        if (diff.compareTo(BigDecimal.ZERO) < 0) {
            diff = BigDecimal.ZERO;
        }
        return new PriceInPosition(targetPrice, diff);
    }

    /**
     * По {@link Place} возвращает соответствующее правило расчёта ставки {@link PriceRule}
     */
    private PlaceItem getRuleByPlace(Place place) {
        return Optional.ofNullable(placeItemMap.get(place))
                .orElseThrow(() -> new IllegalArgumentException("Can't get priceRule for place " + place));
    }

    /**
     * Возвращает правило {@link PriceRule} для поиска предыдущей позиции.
     * Для первой позиции Спецразмещения возвращает её саму.
     */
    private PlaceItem getNext(PlaceItem rule) {
        int idx = placeItemList.indexOf(rule);
        checkState(idx > -1);
        if (idx == 0) {
            // premium1
            return rule;
        }
        return placeItemList.get(idx - 1);
    }
}
