package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;
import java.util.Collection;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidCalculationType;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetAutoNetworkByCoverage;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByPosition;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByTrafficVolume;
import ru.yandex.direct.core.entity.bids.service.BidBase;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.currency.CurrencyCode;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Помогает создавать {@link PriceWizard} для обновления ставок по правилам, описываемым {@link SetAutoBidItem}.
 */
public class PriceWizardHelper {

    private final SetAutoBidItem setAutoBidItem;
    private final CurrencyCode currencyCode;
    // Lazy initialized price wizards
    private PriceWizard<KeywordBidBsAuctionData> cachesSearchPriceWizard;
    private PriceWizard<KeywordTrafaretData> cachedSearchPriceByTrafficVolumeWizard;
    private PriceWizard<KeywordBidPokazometerData> cachedContextPriceWizard;
    private PriceWizard<Collection<BidBase>> cachedRelevanceMatchPriceWizard;

    public PriceWizardHelper(SetAutoBidItem setAutoBidItem, CurrencyCode currencyCode) {
        this.setAutoBidItem = setAutoBidItem;
        this.currencyCode = currencyCode;
    }

    /**
     * @return {@link SearchPriceWizard}, обёрнутый в {@link RoundedPriceWizard}
     * для осуществления правильного округления вычисляемых цен
     */
    public PriceWizard<KeywordBidBsAuctionData> getRoundedSearchPriceWizard() {
        if (cachesSearchPriceWizard == null) {
            cachesSearchPriceWizard = createSearchPriceWizard(setAutoBidItem.getSearchByPosition());
        }
        return cachesSearchPriceWizard;
    }

    /**
     * @return {@link SearchPriceByTrafficVolumeWizard}, обёрнутый в {@link RoundedPriceWizard}
     * для осуществления правильного округления вычисляемых цен
     */
    public PriceWizard<KeywordTrafaretData> getRoundedSearchPriceByTrafficVolumeWizard() {
        if (cachedSearchPriceByTrafficVolumeWizard == null) {
            cachedSearchPriceByTrafficVolumeWizard = createSearchPriceByTrafficVolumeWizard(
                    setAutoBidItem.getSearchByTrafficVolume());
        }
        return cachedSearchPriceByTrafficVolumeWizard;
    }

    /**
     * @return {@link ContextPriceWizard}, обёрнутый в {@link RoundedPriceWizard}
     * для осуществления правильного округления вычисляемых цен
     */
    public PriceWizard<KeywordBidPokazometerData> getRoundedContextPriceWizard() {
        if (cachedContextPriceWizard == null) {
            cachedContextPriceWizard = createContextPriceWizard(setAutoBidItem.getNetworkByCoverage());
        }
        return cachedContextPriceWizard;
    }

    /**
     * @return {@link RelevanceMatchPriceWizard#searchPrice()}, обёрнутый в {@link RoundedPriceWizard}
     * для осуществления правильного округления вычисляемых цен
     */
    public PriceWizard<Collection<BidBase>> getRoundedRelevanceMatchSearchPriceWizard() {
        if (cachedRelevanceMatchPriceWizard == null) {
            cachedRelevanceMatchPriceWizard = wrap(RelevanceMatchPriceWizard.searchPrice());
        }
        return cachedRelevanceMatchPriceWizard;
    }

    /**
     * @return {@link RelevanceMatchPriceWizard#contextPrice()}, обёрнутый в {@link RoundedPriceWizard}
     * для осуществления правильного округления вычисляемых цен
     */
    public PriceWizard<Collection<BidBase>> getRoundedRelevanceMatchContextPriceWizard() {
        if (cachedRelevanceMatchPriceWizard == null) {
            cachedRelevanceMatchPriceWizard = wrap(RelevanceMatchPriceWizard.contextPrice());
        }
        return cachedRelevanceMatchPriceWizard;
    }


    /**
     * Обёртка с округлением до шага торгов и ограничением
     * в {@link ru.yandex.direct.currency.Currency#getMaxShowBid maxShowBid}
     */
    private <T> PriceWizard<T> wrap(PriceWizard<T> origin) {
        return wrapWithCeiling(origin, null);
    }

    /**
     * Обёртка с округлением до шага торгов и ограничением {@code ceilingBid}.
     * Если {@code ceilingBid} {@code null} или выше, чем {@link ru.yandex.direct.currency.Currency#getMaxShowBid maxShowBid},
     * ставка ограничивается {@link ru.yandex.direct.currency.Currency#getMaxShowBid maxShowBid}
     */
    private <T> PriceWizard<T> wrapWithCeiling(PriceWizard<T> origin, @Nullable BigDecimal ceilingBid) {
        return new RoundedPriceWizard<>(origin, ceilingBid, currencyCode);
    }

    private PriceWizard<KeywordBidBsAuctionData> createSearchPriceWizard(SetAutoSearchByPosition searchByPosition) {
        Place position = searchByPosition.getPosition();
        Integer increasePercent = searchByPosition.getIncreasePercent();
        // Пользователь может не указать IncreasePercent. Это означает, что он целится точно в позицию
        PriceWizard<KeywordBidBsAuctionData> rawWizard;
        if (increasePercent == null) {
            rawWizard = SearchPriceWizard.byValue(0, position);
        } else {
            SetAutoBidCalculationType calculatedBy = checkNotNull(searchByPosition.getCalculatedBy(),
                    "CalculatedBy is required if IncreasePercent set. %s", searchByPosition);
            switch (calculatedBy) {
                case DIFF:
                    rawWizard = SearchPriceWizard.byDiff(increasePercent, position);
                    break;
                case VALUE:
                    rawWizard = SearchPriceWizard.byValue(increasePercent, position);
                    break;
                default:
                    throw new IllegalArgumentException(
                            "Unexpected value for calculatedBy: " + calculatedBy);
            }
        }
        return wrapWithCeiling(rawWizard, searchByPosition.getMaxBid());
    }

    private PriceWizard<KeywordTrafaretData> createSearchPriceByTrafficVolumeWizard(
            SetAutoSearchByTrafficVolume searchByTrafficVolume) {

        Integer increasePercent = searchByTrafficVolume.getIncreasePercent();
        PriceWizard<KeywordTrafaretData> rawWizard;
        if (searchByTrafficVolume.getSetMaximum() != null && searchByTrafficVolume.getSetMaximum()) {
            // установка ставки по максимальному объёму трафика
            rawWizard = SearchPriceByMaxTrafficVolumeWizard.createWizard(increasePercent);
        } else {
            Integer targetTrafficVolume = searchByTrafficVolume.getTargetTrafficVolume();
            rawWizard = SearchPriceByTrafficVolumeWizard
                    .createWizard(increasePercent, targetTrafficVolume.doubleValue());
        }
        return wrapWithCeiling(rawWizard, searchByTrafficVolume.getMaxBid());
    }

    private PriceWizard<KeywordBidPokazometerData> createContextPriceWizard(
            SetAutoNetworkByCoverage networkByCoverage) {
        ContextPriceWizard rawWizard = ContextPriceWizard
                .createWizard(networkByCoverage.getIncreasePercent(), networkByCoverage.getContextCoverage());
        return wrapWithCeiling(rawWizard, networkByCoverage.getMaxBid());
    }

}
