package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bids.service.BidBase;
import ru.yandex.direct.utils.math.MathUtils;

import static com.google.common.base.Preconditions.checkArgument;

@ParametersAreNonnullByDefault
public class RelevanceMatchPriceWizard implements PriceWizard<Collection<BidBase>> {

    private static final double DEFAULT_RELEVANCE_MATCH_BID_QUANTILE = 0.3;
    private final double quantile;
    private final Function<BidBase, BigDecimal> priceExtractor;

    private RelevanceMatchPriceWizard(double quantile,
                                      Function<BidBase, BigDecimal> priceExtractor) {
        this.quantile = quantile;
        this.priceExtractor = priceExtractor;
    }

    public static RelevanceMatchPriceWizard searchPrice() {
        return new RelevanceMatchPriceWizard(DEFAULT_RELEVANCE_MATCH_BID_QUANTILE, BidBase::getPrice);
    }

    public static RelevanceMatchPriceWizard contextPrice() {
        return new RelevanceMatchPriceWizard(DEFAULT_RELEVANCE_MATCH_BID_QUANTILE, BidBase::getPriceContext);
    }

    @Override
    public BigDecimal calcPrice(Collection<BidBase> bids) {
        List<BigDecimal> prices = StreamEx.of(bids)
                .map(priceExtractor)
                .toList();
        checkArgument(!prices.isEmpty(), "Can't calculate RelevanceMatch price by empty set of phrase bids");
        return BigDecimal.valueOf(MathUtils.calcQuantile(prices, quantile));
    }
}
