package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;

/**
 * Декоратор для {@link PriceWizard} для округления ставок, вычисленных какой-либо из реализаций {@link PriceWizard}.
 * <p/>
 * Если ставка упирается в пользовательское ограничение или ограничение для валюты, округление не производится.
 */
@ParametersAreNonnullByDefault
class RoundedPriceWizard<T> implements PriceWizard<T> {

    private final PriceWizard<T> origin;
    private final BigDecimal maxBid;
    private final CurrencyCode currencyCode;

    RoundedPriceWizard(PriceWizard<T> origin, @Nullable BigDecimal maxBid, CurrencyCode currencyCode) {
        this.origin = origin;
        this.maxBid = maxBid;
        this.currencyCode = currencyCode;
    }

    @Override
    public BigDecimal calcPrice(T context) {
        BigDecimal price = origin.calcPrice(context);
        Money money = Money.valueOf(price, currencyCode);
        money = money.roundToAuctionStepUp();
        money = money.adjustToCurrencyRange();
        BigDecimal adjustedPrice = money.bigDecimalValue();

        // Если цена упирается в пользовательское ограничение, то не делаем округление до шага Торгов
        if (maxBid != null && adjustedPrice.compareTo(maxBid) > 0) {
            adjustedPrice = maxBid;
        } else {
            // Если пользовательского ограничение нет, то проверяем что после расчетов не выставили ставку более
            // чем MaxShowBid (DIRECT-74109)
            BigDecimal maxShowBid = currencyCode.getCurrency().getMaxShowBid();
            if (adjustedPrice.compareTo(maxShowBid) > 0) {
                adjustedPrice = maxShowBid;
            }
        }

        return adjustedPrice;
    }

}
